package ru.yandex.direct.api.v5.entity.ads.validation;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.ads.ArchiveRequest;
import com.yandex.direct.api.v5.ads.UnarchiveRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.maxBannersPerArchiveRequest;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.maxBannersPerUnarchiveRequest;
import static ru.yandex.direct.api.v5.entity.ads.Constants.MAX_ARCHIVE_UNARCHIVE_IDS_COUNT;

@Component
@ParametersAreNonnullByDefault
public class AdsArchiveUnarchiveRequestValidator {

    private final AdsAdGroupTypeValidator adsAdGroupTypeValidator;

    @Autowired
    public AdsArchiveUnarchiveRequestValidator(AdsAdGroupTypeValidator adsAdGroupTypeValidator) {
        this.adsAdGroupTypeValidator = adsAdGroupTypeValidator;
    }

    public ValidationResult<ArchiveRequest, DefectType> validate(ArchiveRequest externalRequest) {
        ItemValidationBuilder<ArchiveRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        ListValidationBuilder<Long, DefectType> lvb = vb.item(externalRequest.getSelectionCriteria(),
                ArchiveRequest.PropInfo.SELECTION_CRITERIA.schemaName.getLocalPart())
                .list(externalRequest.getSelectionCriteria().getIds(), "Ids");
        validateIds(lvb, maxBannersPerArchiveRequest(MAX_ARCHIVE_UNARCHIVE_IDS_COUNT));
        return vb.getResult();
    }

    public ValidationResult<UnarchiveRequest, DefectType> validate(UnarchiveRequest externalRequest) {
        ItemValidationBuilder<UnarchiveRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        ListValidationBuilder<Long, DefectType> lvb = vb.item(externalRequest.getSelectionCriteria(),
                UnarchiveRequest.PropInfo.SELECTION_CRITERIA.schemaName.getLocalPart())
                .list(externalRequest.getSelectionCriteria().getIds(), "Ids");
        validateIds(lvb, maxBannersPerUnarchiveRequest(MAX_ARCHIVE_UNARCHIVE_IDS_COUNT));
        return vb.getResult();
    }

    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, DefectType> validateInternalRequest(
            ClientId clientId, List<ModelChanges<BannerWithSystemFields>> internalRequest,
            Set<AdGroupType> allowedAdGroupTypes) {
        return adsAdGroupTypeValidator.validateInternalRequest(clientId, internalRequest, allowedAdGroupTypes);
    }

    private void validateIds(ListValidationBuilder<Long, DefectType> lvb, DefectType maxSizeDefect) {
        lvb
                .check(Constraints.notNull(), When.isValid())
                .check(Constraints.notEmptyCollection(), When.isValid())
                .check(Constraints.eachNotNull(), When.isValid())
                .check(Constraints.maxListSize(MAX_ARCHIVE_UNARCHIVE_IDS_COUNT), maxSizeDefect, When.isValid());
    }
}
