package ru.yandex.direct.api.v5.entity.ads.validation;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.ads.ModerateRequest;
import com.yandex.direct.api.v5.general.IdsCriteria;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.ads.Constants;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.ads.ResumeRequest.PropInfo.SELECTION_CRITERIA;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.maxIdsToModerate;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notEmptyCollection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;

/**
 * Валидатор для запроса по отправке баннеров на модерацию
 */
@Component
@ParametersAreNonnullByDefault
public class AdsModerateRequestValidator {

    private final AdsAdGroupTypeValidator adGroupTypeValidator;

    @Autowired
    public AdsModerateRequestValidator(AdsAdGroupTypeValidator adGroupTypeValidator) {
        this.adGroupTypeValidator = adGroupTypeValidator;
    }

    public ValidationResult<ModerateRequest, DefectType> validate(ModerateRequest request) {
        ItemValidationBuilder<ModerateRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb
                .check(notNull(), When.isValid())
                .checkBy(this::validateRequest, When.isValid());
        return vb.getResult();
    }

    public ValidationResult<List<Long>, DefectType> validateInternalRequest(ClientId clientId,
                                                                            List<Long> internalRequest,
                                                                            Set<AdGroupType> allowedAdGroupTypes) {
        return adGroupTypeValidator.validateInternal(clientId, internalRequest, allowedAdGroupTypes);
    }

    private ValidationResult<ModerateRequest, DefectType> validateRequest(ModerateRequest request) {
        ItemValidationBuilder<ModerateRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb
                .item(request.getSelectionCriteria(), SELECTION_CRITERIA.schemaName.getLocalPart())
                .check(notNull(), When.isValid())
                .checkBy(this::validateIdsCriteria, When.isValid());
        return vb.getResult();
    }

    private ValidationResult<IdsCriteria, DefectType> validateIdsCriteria(IdsCriteria idsCriteria) {
        ItemValidationBuilder<IdsCriteria, DefectType> vb = ItemValidationBuilder.of(idsCriteria);
        vb
                .item(idsCriteria.getIds(), "Ids")
                .check(notNull(), When.isValid())
                .check(notEmptyCollection(), When.isValid())
                .check(eachNotNull(), When.isValid())
                .check(Constraints.maxListSize(Constants.MAX_IDS_PER_MODERATE), maxIdsToModerate(), When.isValid());
        return vb.getResult();
    }
}
