package ru.yandex.direct.api.v5.entity.ads.validation;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.adextensiontypes.AdExtensionSetting;
import com.yandex.direct.api.v5.adextensiontypes.AdExtensionSettingItem;
import com.yandex.direct.api.v5.ads.AdBuilderAdUpdateBase;
import com.yandex.direct.api.v5.ads.AdUpdateItem;
import com.yandex.direct.api.v5.ads.MobileAppAdUpdate;
import com.yandex.direct.api.v5.ads.SmartAdBuilderAdUpdate;
import com.yandex.direct.api.v5.ads.TextAdUpdate;
import com.yandex.direct.api.v5.ads.TextAdUpdateBase;
import com.yandex.direct.api.v5.ads.UpdateRequest;
import com.yandex.direct.api.v5.ads.VideoExtensionUpdateItem;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.ads.AdsUpdateRequestItem;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.Defect;
import ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.DefectsContainer;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.bannersWithCreativeDeprecated;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.duplicatedValue;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.invalidCalloutsModification;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.invalidIdInField;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.logoIsOnlyForBannersWithoutCreative;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.maxAdExtensionsOnBannerExceeded;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.maxBannersPerUpdateRequest;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.warningAdExtensionAlreadyAssigned;
import static ru.yandex.direct.api.v5.entity.ads.AdsDefectTypes.warningAdExtensionIsNotAssigned;
import static ru.yandex.direct.api.v5.entity.ads.Constants.MAX_ELEMENTS_PER_UPDATE;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsAdGroupTypeConstraints.allowedAdGroupTypeForUpdate;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsAdGroupTypeConstraints.allowedChangeFlagsForUpdateTextAd;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.INCOMPATIBLE_OPERATIONS;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.Callouts.LIST_TOO_LONG;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.hasBannerTypeSpecified;
import static ru.yandex.direct.api.v5.entity.ads.validation.AdsApiValidationSignals.hasUnambiguousType;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidUseOfField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidValue;
import static ru.yandex.direct.api.v5.validation.DefectTypes.possibleOnlyOneField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredAtLeastOneOfFields;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.isNull;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.ifModelTypeIs;
import static ru.yandex.direct.core.entity.banner.type.callouts.BannerWithCalloutsConstants.MAX_CALLOUTS_COUNT_ON_BANNER;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

@Component
@ParametersAreNonnullByDefault
public class AdsUpdateRequestValidator {

    private static class Field {
        private static final PathNode.Field CALLOUT_SETTING =
                field(TextAdUpdateBase.PropInfo.CALLOUT_SETTING.propertyName);
        private static final PathNode.Field AD_EXTENSIONS =
                field(AdExtensionSetting.PropInfo.AD_EXTENSIONS.propertyName);
        private static final PathNode.Field AD_EXTENSION_ID =
                field(AdExtensionSettingItem.PropInfo.AD_EXTENSION_ID.propertyName);

        private Field() {
            // no instantiation
        }
    }

    private final ApiAuthenticationSource auth;
    private final AdGroupService adGroupService;

    @Autowired
    public AdsUpdateRequestValidator(ApiAuthenticationSource auth, AdGroupService adGroupService) {
        this.auth = auth;
        this.adGroupService = adGroupService;
    }

    public ValidationResult<UpdateRequest, DefectType> validate(UpdateRequest externalRequest,
                                                                boolean smartNoCreatives) {
        ItemValidationBuilder<UpdateRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);

        vb.item(externalRequest.getAds(), UpdateRequest.PropInfo.ADS.schemaName.getLocalPart())
                .check(Constraints.notEmptyCollection())
                .check(Constraints.maxListSize(MAX_ELEMENTS_PER_UPDATE),
                        maxBannersPerUpdateRequest(MAX_ELEMENTS_PER_UPDATE), When.isValid());

        vb.list(externalRequest.getAds(), UpdateRequest.PropInfo.ADS.schemaName.getLocalPart())
                .checkEach(fromPredicate(
                        item -> item.getContentPromotionServiceAd() == null || auth.isServicesApplication(),
                        invalidUseOfField()
                ))
                .checkEachBy(item -> {
                    ItemValidationBuilder<AdUpdateItem, DefectType> ivb = ItemValidationBuilder.of(item);
                    ivb.item(item.getTextAd(), AdUpdateItem.PropInfo.TEXT_AD.schemaName.getLocalPart())
                            .checkBy(validateDisplayUrlTextAllowed(auth.isServicesApplication()), When.notNull())
                            .checkBy(validateLeadformAttributesAllowed(auth.isLeadformAttributesAllowed()), When.notNull());
                    ivb.item(item.getSmartAdBuilderAd(), AdUpdateItem.PropInfo.SMART_AD_BUILDER_AD.schemaName.getLocalPart())
                            .checkBy(validateSmart(smartNoCreatives), When.notNull());
                    ivb.item(item.getMobileAppAd(), AdUpdateItem.PropInfo.MOBILE_APP_AD.schemaName.getLocalPart())
                            .checkBy(validateMobileAppShowTitleAndBody(), When.notNull());
                    return ivb.getResult();
                });

        return vb.getResult();
    }

    Validator<MobileAppAdUpdate, DefectType> validateMobileAppShowTitleAndBody() {
        return item -> {
            ItemValidationBuilder<MobileAppAdUpdate, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getVideoExtension(), MobileAppAdUpdate.PropInfo.VIDEO_EXTENSION.schemaName.getLocalPart())
                    .checkBy(extension -> {
                        ItemValidationBuilder<VideoExtensionUpdateItem, DefectType> ivbLocal = ItemValidationBuilder.of(extension);
                        ivbLocal
                                .item(extension.getShowTitleAndBody(),
                                        VideoExtensionUpdateItem.PropInfo.SHOW_TITLE_AND_BODY.schemaName.getLocalPart())
                                .check(fromPredicate(flag -> flag == YesNoEnum.NO, invalidValue()));
                        return ivbLocal.getResult();
                    }, When.notNull());
            return ivb.getResult();
        };
    }

    /**
     * Валидация проверяет, что тексты отображаемого гринурла устанавливаются только если у пользователя есть права
     *
     * @param isDisplayUrlTextAllowed разрешено ли устанавливать кастомные тексты отображаемого гринурла
     * @see <a href="https://st.yandex-team.ru/DIRECT-142656">DIRECT-142656</a>
     */
    private Validator<TextAdUpdate, DefectType> validateDisplayUrlTextAllowed(boolean isDisplayUrlTextAllowed) {
        return item -> {
            ItemValidationBuilder<TextAdUpdate, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getDutPrefix(), TextAdUpdate.PropInfo.DUT_PREFIX.schemaName.getLocalPart())
                    .check(fromPredicate(prefix -> isDisplayUrlTextAllowed, invalidUseOfField()));
            ivb.item(item.getDutSuffix(), TextAdUpdate.PropInfo.DUT_SUFFIX.schemaName.getLocalPart())
                    .check(fromPredicate(suffix -> isDisplayUrlTextAllowed, invalidUseOfField()));
            return ivb.getResult();
        };
    }

    /**
     * Валидация проверяет, что атрибуты лидформ баннера устанавливаются, только если у пользователя есть права.
     *
     * @param isLeadformAttributesAllowed разрешено ли устанавливать атрибуты лидформ для баннера
     * @see <a href="https://st.yandex-team.ru/DIRECT-147478">DIRECT-147478</>
     */
    private Validator<TextAdUpdate, DefectType> validateLeadformAttributesAllowed(boolean isLeadformAttributesAllowed) {
        return item -> {
            ItemValidationBuilder<TextAdUpdate, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getLfHref(), TextAdUpdate.PropInfo.LF_HREF.schemaName.getLocalPart())
                    .check(fromPredicate(href -> isLeadformAttributesAllowed, invalidUseOfField()));
            ivb.item(item.getLfButtonText(), TextAdUpdate.PropInfo.LF_BUTTON_TEXT.schemaName.getLocalPart())
                    .check(fromPredicate(text -> isLeadformAttributesAllowed, invalidUseOfField()));
            return ivb.getResult();
        };
    }

    private Validator<SmartAdBuilderAdUpdate, DefectType> validateSmart(boolean smartNoCreatives) {
        return item -> {
            ItemValidationBuilder<SmartAdBuilderAdUpdate, DefectType> ivb = ItemValidationBuilder.of(item);
            ivb.item(item.getCreative(), AdBuilderAdUpdateBase.PropInfo.CREATIVE.schemaName.getLocalPart())
                    .weakCheck(isNull(), bannersWithCreativeDeprecated(), When.isTrue(smartNoCreatives));
            ivb.item(item.getLogoExtensionHash(), SmartAdBuilderAdUpdate.PropInfo.LOGO_EXTENSION_HASH.schemaName.getLocalPart())
                    .check(isNull(), logoIsOnlyForBannersWithoutCreative(), When.isTrue(item.getCreative() != null));
            return ivb.getResult();
        };
    }

    /**
     * На все {@code null}-элементы списка вешается ошибка
     * {@link ru.yandex.direct.api.v5.validation.DefectTypes#possibleOnlyOneField}.
     *
     * @see ru.yandex.direct.api.v5.entity.ads.converter.AdsUpdateRequestConverter
     */
    public ValidationResult<List<AdsUpdateRequestItem<BannerWithSystemFields>>, DefectType> validateInternalRequest(
            List<AdsUpdateRequestItem<BannerWithSystemFields>> requestItems,
            Set<AdGroupType> allowedAdGroupTypes,
            String allowedAdTypes) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Set<Long> bannerIds = StreamEx.of(requestItems)
                .map(AdsUpdateRequestItem::getBannerId)
                .filter(Objects::nonNull)
                .toSet();

        Map<Long, AdGroupType> adGroupTypesByBannerIds = adGroupService.getAdGroupTypesByBannerIds(clientId, bannerIds);

        return ListValidationBuilder.<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType>of(requestItems)
                .checkEach(atLeastOneTypeWasSpecified(allowedAdTypes), When.isValid())
                .checkEach(atMostOneTypeWasSpecified(allowedAdTypes), When.isValid())
                .checkEachBy(AdsUpdateRequestValidator::validateCalloutsModification, When.isValid())
                .checkEach(creativeIdFieldIsFilledWhenRequired(),
                        When.isValidAnd(When.valueIs(requestItem ->
                                requestItem.getInternalItem().toModel() instanceof BannerWithCreative)))
                .checkEach(allowedAdGroupTypeForUpdate(adGroupTypesByBannerIds, allowedAdGroupTypes), When.isValid())
                .weakCheckEach(allowedChangeFlagsForUpdateTextAd(), When.isValidAnd(When.valueIs(requestItem -> requestItem.getExternalItem().getTextAd() != null )))
                .getResult();
    }

    private static ValidationResult<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> validateCalloutsModification(
            AdsUpdateRequestItem<BannerWithSystemFields> requestItem) {
        final ValidationResult<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> rootVr =
                new ValidationResult<>(requestItem);
        Optional<List<Long>> optionalCalloutsUpdate = getCalloutsUpdate(requestItem);
        if (!optionalCalloutsUpdate.isPresent()) {
            return rootVr;
        }
        List<Long> update = optionalCalloutsUpdate.get();

        ValidationResult<List<Long>, DefectType> calloutsVr = rootVr
                .getOrCreateSubValidationResult(Field.CALLOUT_SETTING, update)
                .getOrCreateSubValidationResult(Field.AD_EXTENSIONS, update);

        if (hasTooManyItems(update)) {
            calloutsVr.addError(maxAdExtensionsOnBannerExceeded(MAX_CALLOUTS_COUNT_ON_BANNER));
            return rootVr;
        }

        if (hasIncompatibleOperations(update)) {
            calloutsVr.addError(invalidCalloutsModification());
            return rootVr;
        }

        if (mayContainItemDefects(update)) {
            processItemDefects((DefectsContainer) update, calloutsVr);
        }

        return rootVr;
    }

    private static Optional<List<Long>> getCalloutsUpdate(AdsUpdateRequestItem<BannerWithSystemFields> requestItem) {
        return ifModelTypeIs(requestItem.getInternalItem(), BannerWithCallouts.class)
                .map(mc -> mc.getPropIfChanged(BannerWithCallouts.CALLOUT_IDS));
    }

    private static boolean hasTooManyItems(List<Long> update) {
        return update == LIST_TOO_LONG;
    }

    private static boolean hasIncompatibleOperations(List<Long> update) {
        return update == INCOMPATIBLE_OPERATIONS;
    }

    private static void processItemDefects(DefectsContainer defectsContainer, ValidationResult<?, DefectType> vr) {
        defectsContainer.getDefects().forEach((id, defect) ->
                toValidationResultAppender(id, defect, defectsContainer::getOriginalIndices).accept(vr));
    }

    private static Consumer<ValidationResult<?, DefectType>> toValidationResultAppender(Long id, Defect defect,
                                                                                        Function<Long,
                                                                                                Collection<Integer>>
                                                                                                valueToIndices) {
        switch (defect) {
            case DUPLICATE:
                return vr -> vr.getOrCreateSubValidationResult(Field.AD_EXTENSION_ID, null)
                        .addError(duplicatedValue(id));
            case INVALID_ID:
                return vr -> valueToIndices.apply(id).forEach(i ->
                        vr.getOrCreateSubValidationResult(index(i), id)
                                .getOrCreateSubValidationResult(Field.AD_EXTENSION_ID, id)
                                .addError(invalidIdInField()));
            case ALREADY_LINKED:
                return vr -> vr.addWarning(warningAdExtensionAlreadyAssigned(id));
            case NOT_LINKED:
                return vr -> vr.addWarning(warningAdExtensionIsNotAssigned(id));
            default:
                throw new IllegalStateException("Unknown callouts defect: " + defect);
        }
    }

    private static boolean mayContainItemDefects(List<Long> calloutIds) {
        return DefectsContainer.class.isAssignableFrom(calloutIds.getClass());
    }

    private static Constraint<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> creativeIdFieldIsFilledWhenRequired() {
        return Constraint.fromPredicateOfNullable(requestItem -> (
                        requestItem.getExternalItem().getTextAd() == null
                                || requestItem.getExternalItem().getTextAd().getVideoExtension() == null
                                || requestItem.getExternalItem().getTextAd().getVideoExtension().getCreativeId() != null
                ) && (requestItem.getExternalItem().getCpcVideoAdBuilderAd() == null
                        || requestItem.getExternalItem().getCpcVideoAdBuilderAd().getCreative() == null
                        || requestItem.getExternalItem().getCpcVideoAdBuilderAd().getCreative().getCreativeId() != null
                ),
                requiredAtLeastOneOfFields(capitalize(BannerWithCreative.CREATIVE_ID.name())));
    }

    /**
     * Проверяет что тип баннера был определен во время конвертации.
     * См {@link ru.yandex.direct.api.v5.entity.ads.converter.AdsUpdateRequestConverter#convert(UpdateRequest)}
     *
     * @param allowedAdTypes
     */
    private static Constraint<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> atLeastOneTypeWasSpecified(String allowedAdTypes) {
        return fromPredicate(requestItem -> hasBannerTypeSpecified(requestItem.getInternalItem().toModel()),
                requiredAtLeastOneOfFields(allowedAdTypes));
    }

    /**
     * Проверяет, что тип баннера был определен однозначно.
     * См {@link ru.yandex.direct.api.v5.entity.ads.converter.AdsUpdateRequestConverter#convert(UpdateRequest)}
     *
     * @param allowedAdTypes
     */
    private static Constraint<AdsUpdateRequestItem<BannerWithSystemFields>, DefectType> atMostOneTypeWasSpecified(String allowedAdTypes) {
        return fromPredicate(requestItem -> hasUnambiguousType(requestItem.getInternalItem().toModel()),
                possibleOnlyOneField(allowedAdTypes));
    }
}
