package ru.yandex.direct.api.v5.entity.advideos.converter;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.advideos.AdVideoAddItem;
import com.yandex.direct.api.v5.advideos.AdVideoFieldEnum;
import com.yandex.direct.api.v5.advideos.AdVideoGetItem;
import com.yandex.direct.api.v5.advideos.AdVideosSelectionCriteria;
import com.yandex.direct.api.v5.advideos.AddRequest;
import com.yandex.direct.api.v5.advideos.AddResponse;
import com.yandex.direct.api.v5.advideos.GetRequest;
import com.yandex.direct.api.v5.advideos.GetResponse;
import com.yandex.direct.api.v5.advideos.StatusEnum;
import com.yandex.direct.api.v5.advideos.VideoActionResult;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.utils.converter.Converter;
import ru.yandex.direct.validation.result.PathConverter;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.mappingValueConverter;

@Component
@ParametersAreNonnullByDefault
public class AdVideosHelperConverter {
    private static final Map<VideoUploadResponse.FileStatus, StatusEnum> CORE_TO_API_FILE_STATUS =
            Map.of(VideoUploadResponse.FileStatus.READY, StatusEnum.READY,
                    //статус SEMI_READY актуален только для видеоконструктора, но в случае его возвращения отдаем READY
                    VideoUploadResponse.FileStatus.SEMI_READY, StatusEnum.READY,
                    VideoUploadResponse.FileStatus.ERROR, StatusEnum.ERROR,
                    VideoUploadResponse.FileStatus.CONVERTING, StatusEnum.CONVERTING,
                    VideoUploadResponse.FileStatus.NEW, StatusEnum.NEW);
    private static final Converter<VideoUploadResponse.FileStatus, StatusEnum> VIDEO_STATUS_CONVERTER =
            mappingValueConverter(CORE_TO_API_FILE_STATUS);

    protected static final Set<AdVideoFieldEnum> DEFAULT_FIELD_NAMES =
            Set.of(AdVideoFieldEnum.ID, AdVideoFieldEnum.STATUS);

    private final ResultConverter resultConverter;
    private final EnumPropertyFilter<AdVideoFieldEnum> propertyFilter;

    @Autowired
    public AdVideosHelperConverter(PropertyFilter propertyFilter, ResultConverter resultConverter) {
        this.propertyFilter = EnumPropertyFilter.from(AdVideoFieldEnum.class, propertyFilter);
        this.resultConverter = resultConverter;
    }

    public List<CreativeService.VideoItemForUpload> convertAddRequest(AddRequest addRequest) {
        List<AdVideoAddItem> creatives = addRequest.getAdVideos();
        return mapList(creatives, this::convertAdVideos);
    }

    private CreativeService.VideoItemForUpload convertAdVideos(AdVideoAddItem item) {
        return new CreativeService.VideoItemForUpload(
                item.getUrl(),
                item.getVideoData(),
                item.getName()
        );
    }

    @Nonnull
    public AddResponse convertAddResponse(ApiResult<List<ApiResult<CreativeService.VideoItem>>> result,
                                          PathConverter pathConverter) {
        List<VideoActionResult> actionResults = StreamEx.of(result)
                .map(rlist -> StreamEx.of(rlist.getResult())
                        .map(r -> resultConverter.convertToAdVideoActionResult(r, pathConverter))
                        .toList())
                .toFlatList(Function.identity());

        return new AddResponse().withAddResults(actionResults);
    }

    public Set<AdVideoFieldEnum> getAdVideoFieldEnums(GetRequest getRequest) {
        List<AdVideoFieldEnum> fieldNames = getRequest.getFieldNames();
        if (isEmpty(fieldNames)) {
            return DEFAULT_FIELD_NAMES;
        }
        return new HashSet<>(fieldNames);
    }

    public List<CreativeService.VideoItem> convertGetSelectionCriteria(
            AdVideosSelectionCriteria selectionCriteria) {
        List<String> videoIds = selectionCriteria.getIds();
        return mapList(videoIds, CreativeService.VideoItem::new);
    }

    public GetResponse convertGetResponse(List<CreativeService.VideoItemWithStatus> videoItems,
                                          Set<AdVideoFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        var videos = mapList(videoItems, v -> new AdVideoGetItem()
                .withId(v.getExternalVideoId())
                .withStatus(convertVideoStatus(v.getStatus())));

        propertyFilter.filterProperties(videos, requestedFields);

        return new GetResponse()
                .withAdVideos(videos)
                .withLimitedBy(limitedBy);
    }

    protected StatusEnum convertVideoStatus(VideoUploadResponse.FileStatus videoStatus) {
        return VIDEO_STATUS_CONVERTER.convert(videoStatus);
    }
}
