package ru.yandex.direct.api.v5.entity.agencyclients.delegate;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.agencyclients.AddRequest;
import com.yandex.direct.api.v5.agencyclients.AddResponse;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientSettingAddEnum;
import com.yandex.direct.api.v5.generalclients.ClientSettingAddItem;
import com.yandex.direct.api.v5.generalclients.EmailSubscriptionEnum;
import com.yandex.direct.api.v5.generalclients.EmailSubscriptionItem;
import com.yandex.direct.api.v5.generalclients.GrantItem;
import com.yandex.direct.api.v5.generalclients.PrivilegeEnum;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.core.entity.client.model.AddAgencyClientRequest;
import ru.yandex.direct.core.entity.client.model.AddAgencyClientResponse;
import ru.yandex.direct.core.entity.client.service.AddAgencyClientService;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.service.RequestInfoProvider;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.rbac.RbacRole;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

@Service
public class AddAgencyClientsDelegate extends
        BaseApiServiceDelegate<AddRequest, AddResponse, AddAgencyClientRequest, AddAgencyClientResponse> {
    private static final Set<RbacRole> PROHIBITED_ROLES = ImmutableSet.of(
            RbacRole.SUPERREADER, RbacRole.SUPPORT, RbacRole.MEDIA, RbacRole.PLACER);

    private final AddAgencyClientService addAgencyClientService;
    private final RequestInfoProvider requestInfoProvider;
    private final ResultConverter resultConverter;

    @Autowired
    public AddAgencyClientsDelegate(ApiAuthenticationSource auth,
                                    AddAgencyClientService addAgencyClientService, RequestInfoProvider requestInfoProvider,
                                    ResultConverter resultConverter) {
        super(ApiPathConverter.forAgencyClientsAdd(), auth);
        this.addAgencyClientService = addAgencyClientService;
        this.requestInfoProvider = requestInfoProvider;
        this.resultConverter = resultConverter;
    }

    @Override
    public AddAgencyClientRequest convertRequest(AddRequest request) {
        Set<EmailSubscriptionEnum> subscriptions = request.getNotification()
                .getEmailSubscriptions()
                .stream()
                .filter(s -> s.getValue().equals(YesNoEnum.YES))
                .map(EmailSubscriptionItem::getOption)
                .collect(toSet());
        Set<PrivilegeEnum> privileges = request.getGrants()
                .stream()
                .filter(s -> s.getValue().equals(YesNoEnum.YES))
                .map(GrantItem::getPrivilege)
                .collect(toSet());

        Map<ClientSettingAddEnum, Optional<Boolean>> clientSettings = request.getSettings().stream()
                .collect(toMap(ClientSettingAddItem::getOption, v -> Optional.of(v.getValue() == YesNoEnum.YES)));

        return new AddAgencyClientRequest()
                .withLogin(request.getLogin())
                .withFirstName(request.getFirstName())
                .withLastName(request.getLastName())
                .withCurrency(CurrencyCode.valueOf(request.getCurrency().value()))
                .withNotificationEmail(request.getNotification().getEmail())
                .withNotificationLang(Language.valueOf(request.getNotification().getLang().value()))
                .withSendNews(subscriptions.contains(EmailSubscriptionEnum.RECEIVE_RECOMMENDATIONS))
                .withSendAccNews(subscriptions.contains(EmailSubscriptionEnum.TRACK_MANAGED_CAMPAIGNS))
                .withSendWarn(subscriptions.contains(EmailSubscriptionEnum.TRACK_POSITION_CHANGES))
                .withHideMarketRating(
                        !clientSettings.getOrDefault(
                                ClientSettingAddEnum.DISPLAY_STORE_RATING, Optional.of(false)).get())
                .withNoTextAutocorrection(
                        !clientSettings.getOrDefault(
                                ClientSettingAddEnum.CORRECT_TYPOS_AUTOMATICALLY, Optional.of(false)).get())
                .withSharedAccountEnabled(
                        clientSettings.getOrDefault(
                                ClientSettingAddEnum.SHARED_ACCOUNT_ENABLED, Optional.empty()).orElse(null))
                .withAllowEditCampaigns(privileges.contains(PrivilegeEnum.EDIT_CAMPAIGNS))
                .withAllowImportXls(privileges.contains(PrivilegeEnum.IMPORT_XLS))
                .withAllowTransferMoney(privileges.contains(PrivilegeEnum.TRANSFER_MONEY));
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<AddAgencyClientResponse>> result) {
        AddAgencyClientResponse response = result.getResult().get(0);
        return new AddResponse()
                .withLogin(response.getLogin())
                .withEmail(response.getEmail())
                .withPassword(response.getPassword())
                .withClientId(response.getClientId().asLong());
    }

    @Override
    public ApiResult<List<AddAgencyClientResponse>> processRequest(AddAgencyClientRequest request) {
        ApiUser operator = auth.getOperator();
        checkOperatorRights(operator);

        // Так данный метод, является методом агентского сервиса, то проверка на
        // то, что subclient является агентством уже сделана
        ApiUser agency = auth.getSubclient();

        // Определяем главного представителя агентства. Используется для того, чтобы
        // проверить возможности агентства по созданию клиентов
        UidAndClientId agencyMainChief = auth.getChiefSubclient()
                .toUidAndClientId();

        ApiResult<AddAgencyClientResponse> response = resultConverter.toApiResult(addAgencyClientService.processRequest(
                requestInfoProvider,
                operator.toUidAndClientId(),
                agency.toUidAndClientId(),
                agencyMainChief,
                request));

        if (!response.isSuccessful()) {
            return ApiResult.broken(response.getErrors(), response.getWarnings());
        } else {
            return ApiResult.successful(singletonList(response.getResult()));
        }
    }

    private void checkOperatorRights(ApiUser operator) {
        if (PROHIBITED_ROLES.contains(operator.getRole())) {
            throw new AccessDeniedException(
                    String.format("Оператор с ролью %s не допускается к созданию новых субклиентов агенства",
                            operator.getRole().toString()));
        }
    }
}
