package ru.yandex.direct.api.v5.entity.agencyclients.delegate;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.agencyclients.AgencyClientFieldEnum;
import com.yandex.direct.api.v5.agencyclients.AgencyClientsSelectionCriteria;
import com.yandex.direct.api.v5.agencyclients.GetRequest;
import com.yandex.direct.api.v5.agencyclients.GetResponse;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.ApiFaultTranslations;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.agencyclients.service.AgencyClientDataFetcher;
import ru.yandex.direct.api.v5.entity.agencyclients.service.RequestedField;
import ru.yandex.direct.api.v5.entity.agencyclients.validation.GetRequestValidator;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class GetAgencyClientsDelegate extends
        GetApiServiceDelegate<GetRequest, GetResponse, RequestedField, AgencyClientsSelectionCriteria, ClientGetItem> {

    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final UserService userService;
    private final AgencyClientRelationService agencyClientRelationService;
    private final AgencyClientDataFetcher clientDataFetcher;

    @Autowired
    public GetAgencyClientsDelegate(
            ApiAuthenticationSource auth,
            ShardHelper shardHelper,
            RbacService rbacService,
            UserService userService,
            AgencyClientRelationService agencyClientRelationService,
            AgencyClientDataFetcher clientDataFetcher) {
        super(PathConverter.identity(), auth);
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.userService = userService;
        this.agencyClientRelationService = agencyClientRelationService;
        this.clientDataFetcher = clientDataFetcher;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        return GetRequestValidator.validateRequest(externalRequest);
    }

    @Override
    public AgencyClientsSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        return externalRequest.getSelectionCriteria();
    }

    @Override
    public Set<RequestedField> extractFieldNames(GetRequest externalRequest) {
        return listToSet(externalRequest.getFieldNames(), this::convertFieldName);
    }

    @Override
    public List<ClientGetItem> get(GenericGetRequest<RequestedField, AgencyClientsSelectionCriteria> getRequest) {
        // для агентских сервисов API в поле Client-Login указывается представитель агентства
        ApiUser agencyRep = auth.getSubclient();
        ApiUser agencyChiefRep = auth.getChiefSubclient();

        // общая проверка доступности указанного в Client-Login пользователя от оператора выполняется ранее,
        // в рамках авторизации всего запроса
        extraCheckOperatorRights(auth.getOperator(), agencyRep);

        Set<Long> filteredMainChiefUids = getFilteredMainChiefUids(agencyRep, agencyChiefRep, getRequest);

        List<Long> requestedUidsChunk = StreamEx.of(filteredMainChiefUids)
                .sorted()
                .skip(getRequest.getLimitOffset().offset())
                .limit(getRequest.getLimitOffset().limit())
                .toList();

        if (requestedUidsChunk.isEmpty()) {
            return emptyList();
        }

        return clientDataFetcher.getData(getRequest.getRequestedFields(), requestedUidsChunk, agencyChiefRep);
    }

    private Set<Long> getFilteredMainChiefUids(ApiUser agencyRep,
                                               ApiUser agencyChiefRep, GenericGetRequest<RequestedField, AgencyClientsSelectionCriteria> getRequest) {
        // преобразовать запрошенные логины в список uid
        List<String> requestedLogins = getRequestedLogins(getRequest.getSelectionCriteria());

        boolean calledByAgencyRep = agencyChiefRep.getRole() == RbacRole.AGENCY;
        List<Long> requestedSubclientUids;
        if (calledByAgencyRep) {
            requestedSubclientUids = getRequestedSubclientUids(agencyRep, requestedLogins);
        } else {
            // фрилансер
            requestedSubclientUids = getRequestedRelatedClientUids(agencyRep, requestedLogins);
        }

        // возвращаем только главных представителей
        List<Long> subclientMainChiefUids = rbacService.getChiefSubclients(
                requestedSubclientUids);

        // Удаляем из выдачи пользователей у которых только mcb или geo кампании
        Set<Long> filteredMainChiefUids = userService.getUserUidsWithoutHavingOnlyGeoOrMcbCampaigns(
                subclientMainChiefUids);

        // Фильтруем по полю Archived, если это необходимо
        if (getRequest.getSelectionCriteria().getArchived() != null) {
            Map<Long, Long> uidToClientId = shardHelper.getClientIdsByUids(filteredMainChiefUids);

            Set<Long> unarchivedClientIds;
            if (calledByAgencyRep) {
                unarchivedClientIds = agencyClientRelationService.getUnArchivedAgencyClients(
                        agencyChiefRep.getClientId().asLong(), uidToClientId.values());
            } else {
                // фрилансер
                // Заказчики фрилансера -- самоходные клиенты. Определяем архивированность по статусу главного представителя
                Collection<User> users = userService.massGetUser(filteredMainChiefUids);
                unarchivedClientIds = users.stream()
                        .filter(user -> !user.getStatusArch())
                        .map(User::getClientId)
                        .map(ClientId::asLong)
                        .collect(toSet());
            }

            // Фильтрация по архивности
            Predicate<Long> archivedPredicate = clientId -> !unarchivedClientIds.contains(clientId);
            if (getRequest.getSelectionCriteria().getArchived() == YesNoEnum.NO) {
                archivedPredicate = archivedPredicate.negate();
            }
            filteredMainChiefUids = EntryStream.of(uidToClientId)
                    .filterValues(archivedPredicate)
                    .keys()
                    .toSet();
        }
        return filteredMainChiefUids;
    }

    @Override
    public GetResponse convertGetResponse(List<ClientGetItem> result, Set<RequestedField> requestedFields,
                                          @Nullable Long limitedBy) {
        GetResponse response = new GetResponse().withLimitedBy(limitedBy);

        if (!result.isEmpty()) {
            response.withClients(result);
        }
        return response;
    }


    @SuppressWarnings("Duplicates")
    private RequestedField convertFieldName(AgencyClientFieldEnum fieldName) {
        switch (fieldName) {
            case ACCOUNT_QUALITY:
                return RequestedField.ACCOUNT_QUALITY;
            case ARCHIVED:
                return RequestedField.ARCHIVED;
            case CLIENT_ID:
                return RequestedField.CLIENT_ID;
            case CLIENT_INFO:
                return RequestedField.CLIENT_INFO;
            case COUNTRY_ID:
                return RequestedField.COUNTRY_ID;
            case CREATED_AT:
                return RequestedField.CREATED_AT;
            case CURRENCY:
                return RequestedField.CURRENCY;
            case GRANTS:
                return RequestedField.GRANTS;
            case BONUSES:
                return RequestedField.BONUSES;
            case LOGIN:
                return RequestedField.LOGIN;
            case NOTIFICATION:
                return RequestedField.NOTIFICATION;
            case OVERDRAFT_SUM_AVAILABLE:
                return RequestedField.OVERDRAFT_SUM_AVAILABLE;
            case PHONE:
                return RequestedField.PHONE;
            case REPRESENTATIVES:
                return RequestedField.REPRESENTATIVES;
            case RESTRICTIONS:
                return RequestedField.RESTRICTIONS;
            case SETTINGS:
                return RequestedField.SETTINGS;
            case TYPE:
                return RequestedField.TYPE;
            case VAT_RATE:
                return RequestedField.VAT_RATE;
            default:
                throw new IllegalStateException("unknown field");
        }
    }

    private List<String> getRequestedLogins(AgencyClientsSelectionCriteria selectionCriteria) {
        return selectionCriteria.getLogins()
                .stream()
                .map(PassportUtils::normalizeLogin)
                .distinct()
                .collect(toList());
    }

    private List<Long> getRequestedSubclientUids(ApiUser agencyRep, List<String> requestedLogins) {
        if (requestedLogins.isEmpty()) {
            // выдавать всех клиентов, видимых оператору агенства
            return rbacService.getAgencySubclients(agencyRep.getUid());
        } else {
            // выдавать клиентов только с запрошенными логинами и видимых агенству-опреатору
            List<Long> requestedUids = filterList(shardHelper.getUidsByLogin(requestedLogins), Objects::nonNull);
            return rbacService.getAccessibleAgencySubclients(agencyRep.getUid(), requestedUids);
        }
    }

    private List<Long> getRequestedRelatedClientUids(ApiUser freelancer, List<String> requestedLogins) {
        if (requestedLogins.isEmpty()) {
            // выдавать всех клиентов, видимых фрилансеру
            return rbacService.getRelatedClientsChiefs(freelancer.getClientId());
        } else {
            // выдавать клиентов только с запрошенными логинами и имеющих отношения с фрилансером
            List<Long> requestedUids = filterList(shardHelper.getUidsByLogin(requestedLogins), Objects::nonNull);
            return rbacService.getAccessibleRelatedClients(freelancer.getUid(), requestedUids);
        }
    }

    private void extraCheckOperatorRights(ApiUser operator, ApiUser agencyRep) {
        if (operator.isAgency() && !userIsChief(operator) && !Objects.equals(operator.getUid(), agencyRep.getUid())) {
            // оператор с ролью "Агентство", не являющийся главным представителем, может получать данные
            // субклиентов только от своего имени
            throw new AccessDeniedException(ApiFaultTranslations.INSTANCE.detailedNoRightsForUser());
        }
    }

    private boolean userIsChief(User user) {
        return Objects.equals(user.getUid(), user.getChiefUid());
    }
}
