package ru.yandex.direct.api.v5.entity.agencyclients.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.account.score.model.AccountScore;
import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.model.ClientSpecialLimits;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.Representative;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.utils.CommonUtils.memoize;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Component
public class AgencyClientDataFetcher {
    private static final Logger logger = LoggerFactory.getLogger(AgencyClientDataFetcher.class);

    private final AgencyClientRelationService agencyClientRelationService;
    private final BaseClientDataFetcher dataFetcher;
    private final ClientService clientService;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;
    private final UserService userService;
    private final ClientGetItemWriters writers;
    private final ClientNdsService clientNdsService;

    @Autowired
    public AgencyClientDataFetcher(AgencyClientRelationService agencyClientRelationService,
                                   ClientService clientService, BaseClientDataFetcher dataFetcher,
                                   RbacService rbacService, ShardHelper shardHelper, UserService userService, ClientGetItemWriters writers,
                                   ClientNdsService clientNdsService) {
        this.agencyClientRelationService = agencyClientRelationService;
        this.clientService = clientService;
        this.dataFetcher = dataFetcher;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
        this.userService = userService;
        this.writers = writers;
        this.clientNdsService = clientNdsService;
    }

    public List<ClientGetItem> getData(Set<RequestedField> fieldNames, List<Long> requestedUids, User agencyChief) {
        ClientId agencyClientId = agencyChief.getClientId();

        Map<Long, User> usersByUid = listToMap(userService.massGetUser(requestedUids), User::getUid);

        // пользователи могут быть представителями одного клиента: уникализируем клиентов
        List<ClientId> clientIds = usersByUid.values().stream()
                .map(User::getClientId)
                .distinct()
                .collect(toList());

        List<Client> clients = clientService.massGetClient(clientIds);
        Supplier<Map<ClientId, Client>> clientsSupplier = memoize(
                () -> listToMap(clients, c -> ClientId.fromLong(c.getId())));

        Supplier<Map<ClientId, ClientLimits>> clientsLimitsSupplier =
                memoize(() -> dataFetcher.getClientLimits(clientIds));

        Supplier<Map<ClientId, ClientSpecialLimits>> clientsSpecialLimitsSupplier =
                memoize(() -> dataFetcher.getClientSpecialLimits(clientIds));

        Supplier<Map<ClientId, Percent>> clientsNdsSupplier =
                memoize(() -> getClientsNds(clientsSupplier, clientIds, agencyClientId));

        Supplier<Map<ClientId, Currency>> clientsCurrenciesSupplier =
                memoize(() -> dataFetcher.getClientsCurrencies(clientIds));

        Supplier<Map<ClientId, BigDecimal>> overdraftRestSupplier = memoize(() -> dataFetcher
                .getClientsOverdraftRest(clientsCurrenciesSupplier, clientsNdsSupplier, clientIds));

        Supplier<Map<ClientId, BigDecimal>> awaitingBonusesSupplier =
                memoize(() -> dataFetcher.getAwaitingBonuses(clients));

        Supplier<Map<ClientId, BigDecimal>> awaitingBonusesWithoutNdsSupplier = memoize(() -> dataFetcher
                .getAwaitingBonusesWithoutNds(clients, clientsNdsSupplier, clientsCurrenciesSupplier));

        Supplier<Map<ClientId, Collection<AgencyClientRelation>>> agencyClientRelationsSupplier =
                memoize(() -> dataFetcher.getAgencyClientRelation(clientIds));

        Supplier<Map<ClientId, Boolean>> sharedAccountInfoSupplier =
                memoize(() -> dataFetcher.getSharedAccountInfo(
                        clientsCurrenciesSupplier, agencyClientRelationsSupplier, agencyChief.getClientId(),
                        clientIds));

        List<ClientGetItemWriter> fieldWriters = asList(
                writers.getClientIdWriter(),
                writers.getLoginWriter(),
                writers.getClientInfoWriter(),
                writers.getPhoneWriter(),
                writers.getCountryIdWriter(clientsSupplier),
                writers.getCreatedAtWriter(),
                writers.getAccountQualityWriter(accountScoreSupplier(clientIds)),
                writers.getClientTypeWriter(clientTypesSupplier(agencyChief, usersByUid)),
                writers.getArchivedWriter(unarchivedSupplier(agencyChief, usersByUid)),
                writers.getNotificationWriter(smsPhoneSupplier(requestedUids)),
                writers.getSettingsWriter(clientsSupplier, sharedAccountInfoSupplier),
                writers.getRestrictionsWriter(clientsLimitsSupplier, clientsSpecialLimitsSupplier, null),
                writers.getGrantsWriter(subclientsGrantsSupplier(agencyChief, clientIds)),
                writers.getBonusesWriter(awaitingBonusesSupplier, awaitingBonusesWithoutNdsSupplier),
                writers.getRepresentativesWriter(representativesSupplier(clientIds)),
                writers.getCurrencyWriter(clientsCurrenciesSupplier),
                writers.getVatRateWriter(clientsNdsSupplier),
                writers.getOverdraftSumAvailableWriter(overdraftRestSupplier)
        );

        List<ClientGetItem> clientItems = new ArrayList<>(usersByUid.size());
        for (Long uid : requestedUids) {
            User user = usersByUid.get(uid);

            ClientGetItem clientGetItem = new ClientGetItem();
            fieldWriters.forEach(w -> w.write(fieldNames, clientGetItem, user));
            clientItems.add(clientGetItem);
        }
        return clientItems;
    }

    private Map<ClientId, Percent> getClientsNds(Supplier<Map<ClientId, Client>> clientsSupplier,
                                                 List<ClientId> clientIds,
                                                 ClientId agencyClientId) {
        // Для субклиентов-резидентов НДС равен НДС агенства
        ClientNds agencyNds = clientNdsService.getClientNds(agencyClientId);
        if (agencyNds == null) {
            logger.warn("Could not get NDS of agency (agency client id: {}", agencyClientId);
            return Collections.emptyMap();
        }

        Map<ClientId, Client> clientMap = clientsSupplier.get();
        // НДС нерезидентов нужно получать вне зависимости от агенства
        Set<ClientId> nonResidents = clientIds.stream()
                .filter(id -> clientMap.containsKey(id) && clientMap.get(id).getNonResident())
                .collect(toSet());
        Map<ClientId, Percent> nonResidentsNds = clientNdsService.massGetClientNds(nonResidents).stream()
                .collect(toMap((ClientNds c) -> ClientId.fromLong(c.getClientId()), ClientNds::getNds));

        return StreamEx.of(clientIds).mapToEntry(subclientId -> {
            if (nonResidentsNds.containsKey(subclientId)) {
                return nonResidentsNds.get(subclientId);
            } else {
                return agencyNds.getNds();
            }
        }).toMap();
    }


    private Supplier<Map<ClientId, List<RepresentativeInfo>>> representativesSupplier(List<ClientId> clientIds) {
        return memoize(() -> getRepresentatives(clientIds));
    }

    private Supplier<Map<ClientId, List<AgencySubclientGrants>>> subclientsGrantsSupplier(User chiefOperator,
                                                                                          List<ClientId> clientIds) {
        return memoize(() -> getGrants(chiefOperator, clientIds));
    }

    private Supplier<Map<Long, ClientType>> clientTypesSupplier(User agencyChief, Map<Long, User> usersByUid) {
        return memoize(() -> getClientTypes(agencyChief, usersByUid));
    }

    private Supplier<Map<Long, String>> smsPhoneSupplier(List<Long> requestedUids) {
        return memoize(() -> dataFetcher.getSmsPhones(requestedUids));
    }

    private Supplier<Map<ClientId, AccountScore>> accountScoreSupplier(List<ClientId> clientIds) {
        return memoize(() -> dataFetcher.getAccountScore(clientIds));
    }

    private Supplier<Set<Long>> unarchivedSupplier(User chiefOperator, Map<Long, User> usersByUid) {
        return memoize(() -> getUnarchived(chiefOperator, usersByUid));
    }

    /**
     * Возвращает ClientId не заархивированных пользователей
     */
    private Set<Long> getUnarchived(User chiefOperator, Map<Long, User> usersByUid) {
        if (isAgency(chiefOperator)) {
            Map<Long, Long> uidToClientId = shardHelper.getClientIdsByUids(usersByUid.keySet());

            Set<Long> uniqueClientIds = new HashSet<>(uidToClientId.values());
            return agencyClientRelationService.getUnArchivedAgencyClients(
                    chiefOperator.getClientId().asLong(), uniqueClientIds);
        } else {
            // вызов не агентством -- запрос фрилансером своих клиентов
            // знаем, что заказчики фрилансера -- самоходные клиенты, поэтому смотрим на statusArch
            return EntryStream.of(usersByUid)
                    .removeValues(User::getStatusArch)
                    .values()
                    .map(User::getClientId)
                    .map(ClientId::asLong)
                    .toSet();
        }
    }

    private Map<Long, ClientType> getClientTypes(User agencyChief, Map<Long, User> usersByUid) {
        if (isAgency(agencyChief)) {
            return usersByUid.keySet().stream().collect(toMap(Function.identity(), uid -> ClientType.SUBCLIENT));
        } else {
            // вызов не агентством -- запрос фрилансером своих клиентов
            // знаем, что заказчики фрилансера -- самоходные клиенты
            return StreamEx.ofKeys(usersByUid).mapToEntry(uid -> ClientType.CLIENT).toMap();
        }
    }

    private Map<ClientId, List<AgencySubclientGrants>> getGrants(User agencyChief, List<ClientId> clientIds) {
        if (isAgency(agencyChief)) {

            String agencyName = Optional.ofNullable(clientService.getClient(agencyChief.getClientId()))
                    .map(Client::getName)
                    .orElse(null);

            return rbacService.getSubclientsGrants(agencyChief.getUid(), clientIds).stream()
                    .map(e -> new AgencySubclientGrants(agencyName, e))
                    .collect(groupingBy(e -> e.getSubclientGrants().getClientId()));
        } else {
            // вызов не агентством -- запрос фрилансером своих клиентов
            // знаем, что заказчики фрилансера -- самоходные клиенты
            return StreamEx.of(clientIds)
                    .mapToEntry(rbacService::getClientGrantsWithoutAgency)
                    .mapValues(e -> mapList(e, item -> new AgencySubclientGrants(null, item)))
                    .toMap();
        }
    }

    @Nonnull
    private Map<ClientId, List<RepresentativeInfo>> getRepresentatives(List<ClientId> clientIds) {
        Collection<Representative> representatives = rbacService.massGetClientRepresentatives(clientIds);
        List<Long> uids = mapList(representatives, Representative::getUserId);
        Map<Long, User> userByUid = listToMap(userService.massGetUser(uids), User::getUid);

        return dataFetcher.constructRepresentativesResponse(representatives, userByUid);
    }
}
