package ru.yandex.direct.api.v5.entity.agencyclients.service;

import java.math.BigDecimal;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.CurrencyEnum;
import com.yandex.direct.api.v5.general.RepresentativeRoleEnum;
import com.yandex.direct.api.v5.generalclients.BonusesItem;
import com.yandex.direct.api.v5.generalclients.ClientRestrictionEnum;
import com.yandex.direct.api.v5.generalclients.ClientRestrictionItem;
import com.yandex.direct.api.v5.generalclients.ClientSettingGetEnum;
import com.yandex.direct.api.v5.generalclients.ClientSettingGetItem;
import com.yandex.direct.api.v5.generalclients.EmailSubscriptionEnum;
import com.yandex.direct.api.v5.generalclients.EmailSubscriptionItem;
import com.yandex.direct.api.v5.generalclients.GrantGetItem;
import com.yandex.direct.api.v5.generalclients.NotificationGet;
import com.yandex.direct.api.v5.generalclients.ObjectFactory;
import com.yandex.direct.api.v5.generalclients.PrivilegeEnum;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.account.score.model.AccountScore;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.model.ClientSpecialLimits;
import ru.yandex.direct.core.entity.client.repository.ClientMapping;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.model.Representative;

import static ru.yandex.direct.api.v5.ApiConstants.MONEY_MULTIPLIER;
import static ru.yandex.direct.api.v5.common.GeneralUtil.langFromString;
import static ru.yandex.direct.api.v5.common.GeneralUtil.stripZeros;
import static ru.yandex.direct.api.v5.common.GeneralUtil.yesNoFromBool;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Component
@ParametersAreNonnullByDefault
@SuppressWarnings("CodeBlock2Expr")
public class ClientGetItemWriters {
    private static final ObjectFactory FACTORY = new ObjectFactory();

    @Nonnull
    public ClientGetItemWriter getClientIdWriter() {
        return ClientGetItemWriter.forField(RequestedField.CLIENT_ID, (getItem, user) -> {
            getItem.setClientId(user.getClientId().asLong());
        });
    }

    @Nonnull
    public ClientGetItemWriter getLoginWriter() {
        return ClientGetItemWriter.forField(RequestedField.LOGIN, (getItem, user) -> {
            getItem.setLogin(user.getLogin());
        });
    }

    @Nonnull
    public ClientGetItemWriter getClientInfoWriter() {
        return ClientGetItemWriter.forField(RequestedField.CLIENT_INFO, (getItem, user) -> {
            getItem.setClientInfo(user.getFio());
        });
    }

    @Nonnull
    public ClientGetItemWriter getPhoneWriter() {
        return ClientGetItemWriter.forField(RequestedField.PHONE, (getItem, user) -> {
            // NB: возвращаем пустую строку если телефона нет в БД
            getItem.setPhone(user.getPhone() == null ? "" : user.getPhone());
        });
    }

    @Nonnull
    public ClientGetItemWriter getCountryIdWriter(Supplier<Map<ClientId, Client>> clientsSupplier) {
        return ClientGetItemWriter.forField(RequestedField.COUNTRY_ID, (getItem, user) -> {
            Map<ClientId, Client> clientMap = clientsSupplier.get();
            Client client = clientMap.get(user.getClientId());
            if (client != null) {
                getItem.setCountryId(client.getCountryRegionId().intValue());
            }
        });
    }

    @Nonnull
    public ClientGetItemWriter getCreatedAtWriter() {
        return ClientGetItemWriter.forField(RequestedField.CREATED_AT, (getItem, user) -> {
            String createdAt = "";
            if (user.getCreateTime() != null) {
                createdAt = user.getCreateTime().format(DateTimeFormatter.ISO_LOCAL_DATE);
            }
            getItem.setCreatedAt(createdAt);
        });
    }

    @Nonnull
    public ClientGetItemWriter getAccountQualityWriter(
            Supplier<Map<ClientId, AccountScore>> accountScoreSupplier) {
        return ClientGetItemWriter.forField(RequestedField.ACCOUNT_QUALITY, (getItem, user) -> {
            Map<ClientId, AccountScore> accountScoreMap = accountScoreSupplier.get();
            AccountScore accountScore = accountScoreMap.get(user.getClientId());
            getItem.setAccountQuality(FACTORY.createClientGetItemAccountQuality(
                    accountScore == null ? null : stripZeros(accountScore.getScore())));
        });
    }

    @Nonnull
    public ClientGetItemWriter getClientTypeWriter(
            Supplier<Map<Long, ClientType>> clientTypesSupplier) {
        return ClientGetItemWriter.forField(RequestedField.TYPE, (getItem, user) -> {
            Map<Long, ClientType> clientTypeMap = clientTypesSupplier.get();
            ClientType clientType = clientTypeMap.get(user.getUid());
            getItem.setType(clientType.toString().toUpperCase());
        });
    }

    @Nonnull
    public ClientGetItemWriter getClientSubtypeWriter(
            Supplier<Map<Long, ClientSubtype>> clientTypesSupplier) {
        return ClientGetItemWriter.forField(RequestedField.SUBTYPE, (getItem, user) -> {
            Map<Long, ClientSubtype> clientTypeMap = clientTypesSupplier.get();
            ClientSubtype clientSubtype = clientTypeMap.get(user.getUid());
            getItem.setSubtype(clientSubtype.toString().toUpperCase());
        });
    }

    @Nonnull
    public ClientGetItemWriter getRepresentativesWriter(
            Supplier<Map<ClientId, List<RepresentativeInfo>>> representativesSupplier) {
        return ClientGetItemWriter.forField(RequestedField.REPRESENTATIVES, (getItem, user) -> {
            Map<ClientId, List<RepresentativeInfo>> clientsReps = representativesSupplier.get();
            List<RepresentativeInfo> representativeInfos = clientsReps.get(user.getClientId());
            List<com.yandex.direct.api.v5.generalclients.Representative> reps = representativeInfos.stream()
                    .map(ri -> new com.yandex.direct.api.v5.generalclients.Representative()
                            .withLogin(ri.getUser().getLogin())
                            .withEmail(ri.getUser().getEmail())
                            .withRole(getRepresentativeRole(ri.getRepresentative())))
                    .collect(Collectors.toList());
            getItem.setRepresentatives(reps);
        });
    }

    private RepresentativeRoleEnum getRepresentativeRole(Representative representative) {
        if (representative.isChief()) {
            return RepresentativeRoleEnum.CHIEF;
        } else if (representative.isLimited()) {
            return RepresentativeRoleEnum.LIMITED;
        } else if (representative.isReadonly()) {
            return RepresentativeRoleEnum.READONLY;
        } else {
            return RepresentativeRoleEnum.DELEGATE;
        }
    }

    @Nonnull
    public ClientGetItemWriter getVatRateWriter(Supplier<Map<ClientId, Percent>> clientsNdsSupplier) {
        return ClientGetItemWriter.forField(RequestedField.VAT_RATE, (getItem, user) -> {
            Map<ClientId, Percent> clientsNds = clientsNdsSupplier.get();
            Percent nds = clientsNds.get(user.getClientId());
            getItem.setVatRate(FACTORY.createClientGetItemVatRate(
                    nds == null ? null : stripZeros(nds.asPercent().setScale(2, BigDecimal.ROUND_HALF_UP))));
        });
    }

    @Nonnull
    public ClientGetItemWriter getManagedLoginsWriter(Supplier<List<String>> managedLoginsSupplier) {
        return ClientGetItemWriter.forField(RequestedField.MANAGED_LOGINS, (getItem, user) -> {
            List<String> managedLogins = managedLoginsSupplier.get();
            getItem.setManagedLogins(managedLogins);
        });
    }

    @Nonnull
    public ClientGetItemWriter getCurrencyWriter(Supplier<Map<ClientId, Currency>> clientsCurrenciesSupplier) {
        return ClientGetItemWriter.forField(RequestedField.CURRENCY, (getItem, user) -> {
            Map<ClientId, Currency> currencyMap = clientsCurrenciesSupplier.get();
            Currency currency = currencyMap.get(user.getClientId());
            if (currency != null) {
                getItem.setCurrency(CurrencyEnum.fromValue(currency.getCode().name()));
            }
        });
    }

    @Nonnull
    public ClientGetItemWriter getOverdraftSumAvailableWriter(
            Supplier<Map<ClientId, BigDecimal>> overdraftRestSupplier) {
        return ClientGetItemWriter.forField(RequestedField.OVERDRAFT_SUM_AVAILABLE, (getItem, user) -> {
            Map<ClientId, BigDecimal> overdraftRestMap = overdraftRestSupplier.get();
            BigDecimal overdraftRest = overdraftRestMap.get(user.getClientId());
            if (overdraftRest != null) {
                getItem.setOverdraftSumAvailable(
                        overdraftRest.multiply(BigDecimal.valueOf(MONEY_MULTIPLIER)).longValue());
            }
        });
    }

    @Nonnull
    public ClientGetItemWriter getSettingsWriter(Supplier<Map<ClientId, Client>> clientsSupplier,
                                                 Supplier<Map<ClientId, Boolean>> sharedAccountInfoSupplier) {
        return ClientGetItemWriter.forField(RequestedField.SETTINGS, (getItem, user) -> {
            Map<ClientId, Client> clientMap = clientsSupplier.get();
            Map<ClientId, Boolean> sharedAccountEnabledMap = sharedAccountInfoSupplier.get();

            Client client = clientMap.getOrDefault(user.getClientId(), ClientMapping.postProcess(new Client()));
            Boolean sharedAccountEnabled = sharedAccountEnabledMap.getOrDefault(user.getClientId(), false);

            List<ClientSettingGetItem> clientSettings = new ArrayList<>();
            clientSettings.add(
                    new ClientSettingGetItem()
                            .withOption(ClientSettingGetEnum.DISPLAY_STORE_RATING)
                            .withValue(yesNoFromBool(!client.getHideMarketRating())));
            clientSettings.add(
                    new ClientSettingGetItem()
                            .withOption(ClientSettingGetEnum.CORRECT_TYPOS_AUTOMATICALLY)
                            .withValue(yesNoFromBool(!client.getNoTextAutocorrection())));
            clientSettings.add(
                    new ClientSettingGetItem()
                            .withOption(ClientSettingGetEnum.SHARED_ACCOUNT_ENABLED)
                            .withValue(yesNoFromBool(sharedAccountEnabled)));

            getItem.setSettings(clientSettings);
        });
    }

    @Nonnull
    public ClientGetItemWriter getArchivedWriter(Supplier<Set<Long>> unarchivedSupplier) {
        return ClientGetItemWriter.forField(RequestedField.ARCHIVED, (getItem, user) -> {
            Set<Long> unarchived = unarchivedSupplier.get();
            getItem.setArchived(yesNoFromBool(!unarchived.contains(user.getClientId().asLong())));
        });
    }

    @Nonnull
    public ClientGetItemWriter getNotificationWriter(Supplier<Map<Long, String>> smsPhoneSupplier) {
        return ClientGetItemWriter.forField(RequestedField.NOTIFICATION, (getItem, user) -> {
            Map<Long, String> smsPhoneMap = smsPhoneSupplier.get();
            String smsPhone = smsPhoneMap.getOrDefault(user.getUid(), "");
            NotificationGet notification = new NotificationGet()
                    .withLang(langFromString(user.getLang()))
                    .withEmail(user.getEmail())
                    .withSmsPhoneNumber(smsPhone);
            if (!isAgency(user)) {
                notification.withEmailSubscriptions(
                        new EmailSubscriptionItem()
                                .withOption(EmailSubscriptionEnum.RECEIVE_RECOMMENDATIONS)
                                .withValue(yesNoFromBool(user.getSendNews())),
                        new EmailSubscriptionItem()
                                .withOption(EmailSubscriptionEnum.TRACK_MANAGED_CAMPAIGNS)
                                .withValue(yesNoFromBool(user.getSendAccNews())),
                        new EmailSubscriptionItem()
                                .withOption(EmailSubscriptionEnum.TRACK_POSITION_CHANGES)
                                .withValue(yesNoFromBool(user.getSendWarn())));
            }
            getItem.setNotification(notification);
        });
    }

    @Nonnull
    public ClientGetItemWriter getRestrictionsWriter(Supplier<Map<ClientId, ClientLimits>> clientsLimitsSupplier,
                                                     Supplier<Map<ClientId, ClientSpecialLimits>> clientsSpecialLimitsSupplier,
                                                     @Nullable Supplier<Map<ClientId, Integer>> apiUnitsLimitSupplier) {
        return ClientGetItemWriter.forField(RequestedField.RESTRICTIONS, (getItem, user) -> {
            Map<ClientId, ClientLimits> clientLimitsMap = clientsLimitsSupplier.get();
            Map<ClientId, ClientSpecialLimits> clientSpecialLimitsMap = clientsSpecialLimitsSupplier.get();

            ClientLimits clientLimits = clientLimitsMap.get(user.getClientId());
            ClientSpecialLimits clientSpecialLimits = clientSpecialLimitsMap.get(user.getClientId());
            Integer apiUnitsLimit =
                    apiUnitsLimitSupplier == null ? null : apiUnitsLimitSupplier.get().get(user.getClientId());

            List<ClientRestrictionItem> restrictions = new ArrayList<>();

            if (clientLimits != null) {
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.CAMPAIGNS_TOTAL_PER_CLIENT)
                        .withValue(clientLimits.getCampsCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.CAMPAIGNS_UNARCHIVED_PER_CLIENT)
                        .withValue(clientLimits.getUnarcCampsCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.ADGROUPS_TOTAL_PER_CAMPAIGN)
                        .withValue(clientLimits.getBannersCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.ADS_TOTAL_PER_ADGROUP)
                        .withValue(clientLimits.getCreativesCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.KEYWORDS_TOTAL_PER_ADGROUP)
                        .withValue(clientLimits.getKeywordsCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.AD_EXTENSIONS_TOTAL)
                        .withValue(clientLimits.getCalloutsCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.GENERAL_DOMAIN_BLACKLIST_SIZE)
                        .withValue(clientLimits.getGeneralBlacklistSizeLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.VIDEO_DOMAIN_BLACKLIST_SIZE)
                        .withValue(clientLimits.getVideoBlacklistSizeLimitOrDefault().intValue()));
            }

            if (clientSpecialLimits != null) {
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.STAT_REPORTS_TOTAL_IN_QUEUE)
                        .withValue(clientSpecialLimits.getStatReportsCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.FORECAST_REPORTS_TOTAL_IN_QUEUE)
                        .withValue(clientSpecialLimits.getForecastReportsCountLimitOrDefault().intValue()));
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.WORDSTAT_REPORTS_TOTAL_IN_QUEUE)
                        .withValue(clientSpecialLimits.getWordstatReportsCountLimitOrDefault().intValue()));
            }

            if (apiUnitsLimit != null) { // NB: не реализовано для агентского сервиса (пока?)
                restrictions.add(new ClientRestrictionItem()
                        .withElement(ClientRestrictionEnum.API_POINTS)
                        .withValue(apiUnitsLimit));
            }

            getItem.setRestrictions(restrictions);
        });
    }

    @Nonnull
    public ClientGetItemWriter getGrantsWriter(
            Supplier<Map<ClientId, List<AgencySubclientGrants>>> subclientsGrantsSupplier) {
        return ClientGetItemWriter.forField(RequestedField.GRANTS, (getItem, user) -> {
            Map<ClientId, List<AgencySubclientGrants>> grantsMap = subclientsGrantsSupplier.get();
            List<AgencySubclientGrants> subclientGrants = grantsMap.get(user.getClientId());
            List<GrantGetItem> grants = new ArrayList<>();
            for (AgencySubclientGrants el : subclientGrants) {
                grants.add(new GrantGetItem().withPrivilege(PrivilegeEnum.EDIT_CAMPAIGNS)
                        .withValue(yesNoFromBool(el.getSubclientGrants().canCreateCampaign()))
                        .withAgency(el.getAgencyName()));
                grants.add(new GrantGetItem().withPrivilege(PrivilegeEnum.IMPORT_XLS)
                        .withValue(yesNoFromBool(el.getSubclientGrants().canImportXLS()))
                        .withAgency(el.getAgencyName()));
                grants.add(new GrantGetItem().withPrivilege(PrivilegeEnum.TRANSFER_MONEY)
                        .withValue(yesNoFromBool(el.getSubclientGrants().canToTransferMoney()))
                        .withAgency(el.getAgencyName()));
            }
            getItem.setGrants(grants);
        });
    }

    public ClientGetItemWriter getBonusesWriter(
            Supplier<Map<ClientId, BigDecimal>> awaitingBonusSupplier,
            Supplier<Map<ClientId, BigDecimal>> awaitingBonusWithoutNdsSupplier) {
        return ClientGetItemWriter.forField(RequestedField.BONUSES, (getItem, user) -> {
            Map<ClientId, BigDecimal> awaitingBonuses = awaitingBonusSupplier.get();
            Map<ClientId, BigDecimal> awaitingBonusesWithoutNds = awaitingBonusWithoutNdsSupplier.get();
            var clientAwaitingBonus = nvl(awaitingBonuses.get(user.getClientId()), BigDecimal.ZERO);
            var clientAwaitingBonusWithoutNds =
                    nvl(awaitingBonusesWithoutNds.get(user.getClientId()), BigDecimal.ZERO);
            var bonuses = new BonusesItem()
                    .withAwaitingBonus(clientAwaitingBonus.multiply(BigDecimal.valueOf(MONEY_MULTIPLIER)).longValue())
                    .withAwaitingBonusWithoutNds(
                            clientAwaitingBonusWithoutNds.multiply(BigDecimal.valueOf(MONEY_MULTIPLIER)).longValue());
            getItem.setBonuses(bonuses);
        });
    }

}
