package ru.yandex.direct.api.v5.entity.audiencetargets.delegate;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;

import com.yandex.direct.api.v5.audiencetargets.AddRequest;
import com.yandex.direct.api.v5.audiencetargets.AddResponse;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.audiencetargets.converter.AudienceTargetsHelperConverter;
import ru.yandex.direct.api.v5.entity.audiencetargets.validation.AudienceTargetsDefectPresentations;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.audiencetargets.Constants.MAX_RETARGETINGS_PER_REQUEST;
import static ru.yandex.direct.api.v5.entity.audiencetargets.validation.AudienceTargetsAdGroupTypeConstraints.allowedAdGroupTypeForAdd;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequestAddRetargetings;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;

@Service
public class AddAudienceTargetsDelegate
        extends OperationOnListDelegate<AddRequest, AddResponse, TargetInterest, Long> {
    private static final Logger logger = LoggerFactory.getLogger(AddAudienceTargetsDelegate.class);

    private final AudienceTargetsHelperConverter audienceTargetsHelperConverter;
    private final RetargetingService retargetingService;
    private final ResultConverter resultConverter;
    private final AdGroupService adGroupService;

    @Autowired
    public AddAudienceTargetsDelegate(ApiAuthenticationSource auth,
                                      AudienceTargetsHelperConverter audienceTargetsHelperConverter,
                                      RetargetingService retargetingService,
                                      ResultConverter resultConverter,
                                      AdGroupService adGroupService,
                                      PpcPropertiesSupport ppcPropertiesSupport,
                                      FeatureService featureService) {
        super(ApiPathConverter.forAudienceTargets(), auth, ppcPropertiesSupport, featureService);
        this.audienceTargetsHelperConverter = audienceTargetsHelperConverter;
        this.retargetingService = retargetingService;
        this.resultConverter = resultConverter;
        this.adGroupService = adGroupService;
    }

    @Override
    public ValidationResult<AddRequest, DefectType> validateRequest(AddRequest request) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getAudienceTargets(), "AudienceTargets")
                .check(eachNotNull())
                .check(maxListSize(MAX_RETARGETINGS_PER_REQUEST),
                        maxElementsPerRequestAddRetargetings(MAX_RETARGETINGS_PER_REQUEST), When.isValid());
        return vb.getResult();
    }

    @Override
    public List<TargetInterest> convertRequest(AddRequest externalRequest) {
        logger.debug("convertRequest({})", externalRequest);
        return audienceTargetsHelperConverter.convertAddRequest(externalRequest.getAudienceTargets());
    }

    @Nonnull
    @Override
    public ValidationResult<List<TargetInterest>, DefectType> validateInternalRequest(
            List<TargetInterest> internalRequest) {
        logger.debug("validateInternalRequest({})", internalRequest);
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Set<Long> adGroupIds = StreamEx.of(internalRequest)
                .map(Retargeting::getAdGroupId)
                .filter(Objects::nonNull)
                .toSet();
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, adGroupIds);
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        return ListValidationBuilder.<TargetInterest, DefectType>of(internalRequest)
                .checkEach(allowedAdGroupTypeForAdd(adGroupTypes, allowedAdGroupTypes))
                .getResult();
    }

    @Override
    public ApiMassResult<Long> processList(List<TargetInterest> targetInterests) {
        logger.debug("processList({})", targetInterests);
        Long operatorUid = auth.getOperator().getUid();
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Long clientUid = auth.getChiefSubclient().getChiefUid();
        MassResult<Long> result = retargetingService
                .createAddOperation(Applicability.PARTIAL, targetInterests, operatorUid, clientId, clientUid)
                .prepareAndApply();
        return resultConverter.toApiMassResult(result, AudienceTargetsDefectPresentations.HOLDER);
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        logger.debug("convertResponse({})", result);
        return audienceTargetsHelperConverter.convertAddResponse(result, apiPathConverter);
    }
}
