package ru.yandex.direct.api.v5.entity.audiencetargets.delegate;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;

import com.yandex.direct.api.v5.audiencetargets.DeleteRequest;
import com.yandex.direct.api.v5.audiencetargets.DeleteResponse;
import com.yandex.direct.api.v5.general.ActionResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.audiencetargets.AudienceTargetsTranslations;
import ru.yandex.direct.api.v5.entity.audiencetargets.converter.AudienceTargetsDeleteRequestConverter;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.types.Identity;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.audiencetargets.Constants.MAX_IDS_PER_REQUEST;
import static ru.yandex.direct.api.v5.entity.audiencetargets.validation.AudienceTargetsAdGroupTypeConstraints.allowedAdGroupType;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class DeleteAudienceTargetsDelegate
        extends OperationOnListDelegate<DeleteRequest, DeleteResponse, Long, Long> {
    private static final DefectType DELETE_MAX_IDS_DETAILED = DefectTypes.maxIdsInSelection().withDetailedMessage(
            AudienceTargetsTranslations.INSTANCE.deleteMaxIdsDetailed(new Identity(MAX_IDS_PER_REQUEST)));

    private final AudienceTargetsDeleteRequestConverter deleteRequestConverter;
    private final RetargetingService retargetingService;
    private final ResultConverter resultConverter;
    private final AdGroupService adGroupService;

    @Autowired
    public DeleteAudienceTargetsDelegate(
            ApiAuthenticationSource auth,
            AudienceTargetsDeleteRequestConverter deleteRequestConverter,
            RetargetingService retargetingService,
            ResultConverter resultConverter,
            AdGroupService adGroupService,
            PpcPropertiesSupport ppcPropertiesSupport,
            FeatureService featureService) {
        super(ApiPathConverter.forAudienceTargets(), auth, ppcPropertiesSupport, featureService);
        this.retargetingService = retargetingService;
        this.deleteRequestConverter = deleteRequestConverter;
        this.resultConverter = resultConverter;
        this.adGroupService = adGroupService;
    }

    @Override
    public ValidationResult<DeleteRequest, DefectType> validateRequest(DeleteRequest request) {
        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getSelectionCriteria(), "SelectionCriteria")
                .item(request.getSelectionCriteria().getIds(), "Ids")
                .check(maxListSize(MAX_IDS_PER_REQUEST), DELETE_MAX_IDS_DETAILED);
        return vb.getResult();
    }

    @Override
    public List<Long> convertRequest(DeleteRequest externalRequest) {
        return deleteRequestConverter.convert(externalRequest);
    }

    @Nonnull
    @Override
    public ValidationResult<List<Long>, DefectType> validateInternalRequest(List<Long> internalRequest) {
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        return allowedAdGroupType(internalRequest, retargetingService, auth, adGroupService, allowedAdGroupTypes);
    }

    @Override
    public ApiMassResult<Long> processList(List<Long> validItems) {
        Long operatorUid = auth.getOperator().getUid();
        ClientId clientId = auth.getChiefSubclient().getClientId();
        return resultConverter
                .toApiMassResult(
                        retargetingService.deleteRetargetings(validItems, clientId, operatorUid));
    }

    @Override
    public DeleteResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        List<ActionResult> deleteResults =
                mapList(result.getResult(), r -> resultConverter.convertToActionResult(r, apiPathConverter));
        return new DeleteResponse().withDeleteResults(deleteResults);
    }
}
