package ru.yandex.direct.api.v5.entity.audiencetargets.delegate;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.audiencetargets.AudienceTargetFieldEnum;
import com.yandex.direct.api.v5.audiencetargets.AudienceTargetGetItem;
import com.yandex.direct.api.v5.audiencetargets.AudienceTargetSelectionCriteria;
import com.yandex.direct.api.v5.audiencetargets.GetRequest;
import com.yandex.direct.api.v5.audiencetargets.GetResponse;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.audiencetargets.converter.AudienceTargetsGetRequestConverter;
import ru.yandex.direct.api.v5.entity.audiencetargets.converter.TargetInterestConverter;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelection;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.entity.audiencetargets.Constants.GET_AD_GROUP_IDS_LIMIT;
import static ru.yandex.direct.api.v5.entity.audiencetargets.Constants.GET_CAMPAIGN_IDS_LIMIT;
import static ru.yandex.direct.api.v5.entity.audiencetargets.Constants.GET_INTEREST_IDS_LIMIT;
import static ru.yandex.direct.api.v5.entity.audiencetargets.Constants.GET_RETARGETING_LIST_IDS_LIMIT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.noNeededParamsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class GetAudienceTargetsDelegate extends
        GetApiServiceDelegate<GetRequest, GetResponse, AudienceTargetFieldEnum, RetargetingSelection, TargetInterest> {
    private static final Set<AdGroupType> CPM_ADGROUP_TYPES = ImmutableSet.<AdGroupType>builder()
            .add(AdGroupType.CPM_BANNER)
            .add(AdGroupType.CPM_GEOPRODUCT)
            .add(AdGroupType.CPM_GEO_PIN)
            .add(AdGroupType.CPM_VIDEO)
            .add(AdGroupType.CPM_OUTDOOR)
            .add(AdGroupType.CPM_INDOOR)
            .add(AdGroupType.CPM_YNDX_FRONTPAGE)
            .add(AdGroupType.CPM_PRICE)
            .add(AdGroupType.CPM_AUDIO)
            .build();

    private final AudienceTargetsGetRequestConverter getRequestConverter;
    private final RetargetingService retargetingService;
    private final ClientService clientService;
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;
    private final EnumPropertyFilter<AudienceTargetFieldEnum> propertyFilter;
    private final TargetInterestConverter targetInterestConverter;
    private final AdGroupService adGroupService;

    @Autowired
    public GetAudienceTargetsDelegate(ApiAuthenticationSource auth,
                                      AudienceTargetsGetRequestConverter getRequestConverter,
                                      RetargetingService retargetingService,
                                      ClientService clientService,
                                      AdGroupRepository adGroupRepository,
                                      ShardHelper shardHelper, PropertyFilter propertyFilter,
                                      TargetInterestConverter targetInterestConverter,
                                      AdGroupService adGroupService) {
        super(ApiPathConverter.forAudienceTargets(), auth);
        this.getRequestConverter = getRequestConverter;
        this.retargetingService = retargetingService;
        this.clientService = clientService;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
        this.propertyFilter = EnumPropertyFilter.from(AudienceTargetFieldEnum.class, propertyFilter);
        this.targetInterestConverter = targetInterestConverter;
        this.adGroupService = adGroupService;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes);

        vb.item(externalRequest.getSelectionCriteria(), "SelectionCriteria")
                .checkBy(this::validateSelectionCriteria, When.notNull());

        return vb.getResult();
    }

    private ValidationResult<AudienceTargetSelectionCriteria, DefectType> validateSelectionCriteria(
            AudienceTargetSelectionCriteria selection) {
        ItemValidationBuilder<AudienceTargetSelectionCriteria, DefectType> vb =
                ItemValidationBuilder.of(selection);

        vb.item(selection.getIds(), "Ids")
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection());

        vb.item(selection.getAdGroupIds(), "AdGroupIds")
                .check(maxListSize(GET_AD_GROUP_IDS_LIMIT), maxIdsInSelection());

        vb.item(selection.getCampaignIds(), "CampaignIds")
                .check(maxListSize(GET_CAMPAIGN_IDS_LIMIT), maxIdsInSelection());

        vb.item(selection.getRetargetingListIds(), "RetargetingListIds")
                .check(maxListSize(GET_RETARGETING_LIST_IDS_LIMIT), maxIdsInSelection());

        vb.item(selection.getInterestIds(), "InterestIds")
                .check(maxListSize(GET_INTEREST_IDS_LIMIT), maxIdsInSelection());

        Constraint<AudienceTargetSelectionCriteria, DefectType> someIdsMustPresent =
                s -> !isEmpty(s.getIds()) || !isEmpty(s.getAdGroupIds()) || !isEmpty(s.getCampaignIds()) ||
                        !isEmpty(s.getRetargetingListIds()) || !isEmpty(s.getInterestIds())
                        ? null
                        : noNeededParamsInSelection();

        vb.check(someIdsMustPresent);

        return vb.getResult();
    }

    @Override
    public RetargetingSelection extractSelectionCriteria(GetRequest externalRequest) {
        return getRequestConverter.convert(externalRequest);
    }

    @Override
    public Set<AudienceTargetFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public List<TargetInterest> get(GenericGetRequest<AudienceTargetFieldEnum, RetargetingSelection> getRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Long operatorUid = auth.getOperator().getUid();
        List<TargetInterest> targetInterests = retargetingService.getRetargetings(getRequest.getSelectionCriteria(),
                clientId, operatorUid, getRequest.getLimitOffset());

        Set<Long> adGroupIds = StreamEx.of(targetInterests)
                .map(Retargeting::getAdGroupId)
                .filter(Objects::nonNull)
                .toSet();
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, adGroupIds);

        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        targetInterests.removeIf(ti -> adGroupTypes.get(ti.getAdGroupId()) != null
                && !allowedAdGroupTypes.contains(adGroupTypes.get(ti.getAdGroupId())));

        fillAbsentPriceContext(clientId, targetInterests);
        return targetInterests;
    }

    private void fillAbsentPriceContext(ClientId clientId, List<TargetInterest> targetInterests) {
        // см. DIRECT-77192
        // решили, что "клиент нулевую ставку поставить не может (т.е. если клиент выполнит запрос на установку
        // ставки со значением, которое мы возвращаем, то запрос завершится ошибкой), то надо возвращать вместо
        // нулевой минимальную."
        if (targetInterests.stream().anyMatch(ti -> ti.getPriceContext() == null)) {
            Currency currency = clientService.getWorkCurrency(clientId);
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            Map<Long, AdGroupWithType> adGroupWithTypeMap = adGroupRepository
                    .getAdGroupsWithType(shard, clientId, mapList(targetInterests, TargetInterest::getAdGroupId));
            targetInterests.stream()
                    .filter(ti -> ti.getPriceContext() == null)
                    .forEach(ti -> ti.setPriceContext(calcDefaultPriceContext(
                            adGroupWithTypeMap.getOrDefault(ti.getAdGroupId(), new AdGroup()).getType(), currency)));
        }
    }

    private BigDecimal calcDefaultPriceContext(@Nullable AdGroupType adGroupType, Currency currency) {
        return adGroupType != null && CPM_ADGROUP_TYPES.contains(adGroupType) ?
                currency.getMinCpmPrice() : currency.getMinPrice();
    }

    @Override
    public GetResponse convertGetResponse(List<TargetInterest> result, Set<AudienceTargetFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        GetResponse response = new GetResponse().withLimitedBy(limitedBy);

        List<AudienceTargetGetItem> items = mapList(result, targetInterestConverter::convert);
        propertyFilter.filterProperties(items, requestedFields);
        if (!items.isEmpty()) {
            response.withAudienceTargets(items);
        }
        return response;
    }
}
