package ru.yandex.direct.api.v5.entity.audiencetargets.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelection;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.audiencetargets.AudienceTargetsDefectTypes.audienceTargetsNotSupportedInThisAdGroup;

public class AudienceTargetsAdGroupTypeConstraints {

    public static Constraint<TargetInterest, DefectType> allowedAdGroupTypeForAdd(Map<Long, AdGroupType> adGroupTypes,
                                                                                  Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(item -> adGroupTypes.get(item.getAdGroupId()) == null
                        || allowedAdGroupTypes.contains(adGroupTypes.get(item.getAdGroupId())),
                audienceTargetsNotSupportedInThisAdGroup());
    }

    public static Constraint<SetBidItem, DefectType> allowedAdGroupTypeForSetBids(
            Map<Long, TargetInterest> targetInterestById,
            Map<Long, AdGroupType> adGroupTypes,
            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(item -> {
                    AdGroupType adGroupTypeByTargetInterestId =
                            getAdGroupTypeByTargetInterestId(item.getId(), targetInterestById, adGroupTypes);
                    AdGroupType adGroupType = adGroupTypes.get(item.getAdGroupId());

                    return (adGroupTypeByTargetInterestId == null || allowedAdGroupTypes.contains(adGroupTypeByTargetInterestId))
                            && (adGroupType == null || allowedAdGroupTypes.contains(adGroupType));
                },
                audienceTargetsNotSupportedInThisAdGroup());
    }

    public static ValidationResult<List<Long>, DefectType> allowedAdGroupType(
            List<Long> internalRequest,
            RetargetingService retargetingService,
            ApiAuthenticationSource auth,
            AdGroupService adGroupService,
            Set<AdGroupType> allowedAdGroupTypes) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Long operatorUid = auth.getOperator().getUid();
        List<TargetInterest> targetInterests = retargetingService.getRetargetings(
                new RetargetingSelection().withIds(internalRequest), clientId, operatorUid, LimitOffset.maxLimited());

        Map<Long, TargetInterest> targetInterestById = StreamEx.of(targetInterests)
                .mapToEntry(Retargeting::getId, ti -> ti)
                .distinctKeys()
                .toMap();

        Set<Long> adGroupIds = StreamEx.of(targetInterests)
                .map(Retargeting::getAdGroupId)
                .filter(Objects::nonNull)
                .toSet();
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, adGroupIds);

        return ListValidationBuilder.<Long, DefectType>of(internalRequest)
                .checkEach(allowedAdGroupType(targetInterestById, adGroupTypes, allowedAdGroupTypes))
                .getResult();
    }

    private static Constraint<Long, DefectType> allowedAdGroupType(
            Map<Long, TargetInterest> targetInterestById,
            Map<Long, AdGroupType> adGroupTypes,
            Set<AdGroupType> allowedAdGroupTypes) {
        return Constraint.fromPredicate(id -> {
                    AdGroupType adGroupTypeByTargetInterestId =
                            getAdGroupTypeByTargetInterestId(id, targetInterestById, adGroupTypes);
                    return (adGroupTypeByTargetInterestId == null || allowedAdGroupTypes.contains(adGroupTypeByTargetInterestId));
                },
                audienceTargetsNotSupportedInThisAdGroup());
    }

    private static AdGroupType getAdGroupTypeByTargetInterestId(
            Long interestId,
            Map<Long, TargetInterest> targetInterestById,
            Map<Long, AdGroupType> adGroupTypes) {
        if (interestId == null) {
            return null;
        }
        TargetInterest targetInterest = targetInterestById.get(interestId);
        if (targetInterest == null) {
            return null;
        }
        return adGroupTypes.get(targetInterest.getAdGroupId());
    }
}
