package ru.yandex.direct.api.v5.entity.bidmodifiers.delegate;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.bidmodifiers.AddRequest;
import com.yandex.direct.api.v5.bidmodifiers.AddResponse;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierAddBase;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierAddItem;
import com.yandex.direct.api.v5.bidmodifiers.IncomeGradeAdjustmentAdd;
import com.yandex.direct.api.v5.bidmodifiers.OperatingSystemTypeEnum;
import com.yandex.direct.api.v5.general.AgeRangeEnum;
import com.yandex.direct.api.v5.general.GenderEnum;
import com.yandex.direct.api.v5.general.IncomeGradeEnum;
import com.yandex.direct.api.v5.general.SerpLayoutEnum;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnlyAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgoAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPrismaIncomeGrade;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPrismaIncomeGradeAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTVAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPositionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideoAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.bidmodifier.TabletOsType;
import ru.yandex.direct.core.entity.bidmodifier.TrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionOperator;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionParameter;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.converter.Converter;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.BidModifiersDefectTypes.addItemsLimitExceeded;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.BID_MODIFIERS_ADD_ITEMS_LIMIT;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.INCOME_GRADE_LITERAL_VALUE_BY_INCOME_GRADE_ENUM;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.getApi5AllowedAdjustments;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.validation.BidModifiersAdGroupTypeConstraints.allowedAdGroupType;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.validation.BidModifiersDefectPresentations.HOLDER;
import static ru.yandex.direct.api.v5.validation.DefectTypes.possibleOnlyOneField;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredAtLeastOneOfFields;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.mappingValueConverter;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class AddBidModifiersDelegate
        extends OperationOnListDelegate<AddRequest, AddResponse, BidModifierAddItem, List<Long>> {

    private final BidModifierService bidModifierService;
    private final ResultConverter resultConverter;
    private final AdGroupService adGroupService;

    @Autowired
    public AddBidModifiersDelegate(BidModifierService bidModifierService,
                                   ResultConverter resultConverter,
                                   ApiAuthenticationSource auth,
                                   AdGroupService adGroupService,
                                   PpcPropertiesSupport ppcPropertiesSupport,
                                   FeatureService featureService) {
        super(PathConverter.identity(), auth, ppcPropertiesSupport, featureService);
        this.bidModifierService = bidModifierService;
        this.resultConverter = resultConverter;
        this.adGroupService = adGroupService;
    }

    private static GenderType genderTypeFromExternal(@Nullable GenderEnum gender) {
        if (gender == null) {
            return null;
        }
        switch (gender) {
            case GENDER_MALE:
                return GenderType.MALE;
            case GENDER_FEMALE:
                return GenderType.FEMALE;
            default:
                throw new IllegalStateException("Invalid value: " + gender);
        }
    }

    private static final Map<OperatingSystemTypeEnum, OsType> OS_TYPE_MAPPING = ImmutableMap.of(
            OperatingSystemTypeEnum.IOS, OsType.IOS,
            OperatingSystemTypeEnum.ANDROID, OsType.ANDROID
    );
    private static final Converter<OperatingSystemTypeEnum, OsType>
            OS_TYPE_CONVERTER = mappingValueConverter(OS_TYPE_MAPPING);

    private static final Map<OperatingSystemTypeEnum, TabletOsType> TABLET_OS_TYPE_MAPPING = ImmutableMap.of(
            OperatingSystemTypeEnum.IOS, TabletOsType.IOS,
            OperatingSystemTypeEnum.ANDROID, TabletOsType.ANDROID
    );
    private static final Converter<OperatingSystemTypeEnum, TabletOsType>
            TABLET_OS_TYPE_CONVERTER = mappingValueConverter(TABLET_OS_TYPE_MAPPING);


    private static AgeType ageTypeFromExternal(@Nullable AgeRangeEnum age) {
        if (age == null) {
            return null;
        }
        switch (age) {
            case AGE_0_17:
                return AgeType._0_17;
            case AGE_18_24:
                return AgeType._18_24;
            case AGE_25_34:
                return AgeType._25_34;
            case AGE_35_44:
                return AgeType._35_44;
            case AGE_45_54:
                return AgeType._45_54;
            case AGE_45:
                return AgeType._45_;
            case AGE_55:
                return AgeType._55_;
            default:
                throw new IllegalStateException("Invalid value: " + age);
        }
    }

    private static TrafaretPosition positionFromExternal(@Nullable SerpLayoutEnum serpLayout) {
        if (serpLayout == null) {
            return null;
        }
        switch (serpLayout) {
            case ALONE:
                return TrafaretPosition.ALONE;
            case SUGGEST:
                return TrafaretPosition.SUGGEST;
            default:
                throw new IllegalStateException("Invalid value: " + serpLayout);
        }
    }

    static String incomeGradeEnumFromExternal(IncomeGradeEnum gradeEnum) {
        if (INCOME_GRADE_LITERAL_VALUE_BY_INCOME_GRADE_ENUM.containsKey(gradeEnum)) {
            return INCOME_GRADE_LITERAL_VALUE_BY_INCOME_GRADE_ENUM.get(gradeEnum);
        }
        throw new IllegalStateException("Unknown income grade " + gradeEnum.name());
    }

    /**
     * Валидация исходного запроса в целом. Если в результатах будут ошибки, обработка на этом завершится.
     */
    @Nullable
    @Override
    public ValidationResult<AddRequest, DefectType> validateRequest(AddRequest externalRequest) {
        return validateRequestCore(externalRequest);
    }

    public ValidationResult<AddRequest, DefectType> validateRequestCore(AddRequest externalRequest) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getBidModifiers(), "BidModifiers")
                .check(fromPredicate(allItems -> allItems.size() <= BID_MODIFIERS_ADD_ITEMS_LIMIT,
                        addItemsLimitExceeded(BID_MODIFIERS_ADD_ITEMS_LIMIT)));
        return vb.getResult();
    }

    @Override
    public List<BidModifierAddItem> convertRequest(AddRequest externalRequest) {
        return externalRequest.getBidModifiers();
    }

    @Nonnull
    @Override
    public ValidationResult<List<BidModifierAddItem>, DefectType> validateInternalRequest(
            List<BidModifierAddItem> addItems) {
        Set<AdGroupType> allowedAdGroupTypes = getAllowedAdGroupTypes();
        return validateInternalRequestCore(addItems, allowedAdGroupTypes);
    }

    private ValidationResult<List<BidModifierAddItem>, DefectType> validateInternalRequestCore(
            List<BidModifierAddItem> bidModifiers,
            Set<AdGroupType> allowedAdGroupTypes) {
        ListValidationBuilder<BidModifierAddItem, DefectType> vb = ListValidationBuilder.of(bidModifiers);
        String adjustments = getApi5AllowedAdjustments();
        vb.checkEach(fromPredicate(it -> getAdjustmentsCount(it) != 0, requiredAtLeastOneOfFields(adjustments)));
        vb.checkEach(fromPredicate(it -> getAdjustmentsCount(it) == 1,
                possibleOnlyOneField(adjustments)), When.isValid());

        List<Long> adGroupIds = StreamEx.of(bidModifiers)
                .map(BidModifierAddBase::getAdGroupId)
                .filter(Objects::nonNull)
                .toList();
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(auth.getSubclient().getClientId(),
                adGroupIds);
        vb.checkEach(allowedAdGroupType(adGroupTypes, allowedAdGroupTypes));

        return vb.getResult();
    }

    private BidModifierExpression toPrismaIncomeGradeBidModifier(List<IncomeGradeAdjustmentAdd> listOfIncomeGradeAdjustmentAdd) {
        var bidModifierExpression = new BidModifierPrismaIncomeGrade()
                .withType(BidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER);

        return bidModifierExpression
                .withExpressionAdjustments(mapList(listOfIncomeGradeAdjustmentAdd,
                        this::toPrismaIncomeGradeAdjustment));
    }

    private BidModifierExpressionAdjustment toPrismaIncomeGradeAdjustment(IncomeGradeAdjustmentAdd incomeGradeAdjustmentAdd) {
        var condition = new BidModifierExpressionLiteral()
                .withOperation(BidModifierExpressionOperator.EQ)
                .withParameter(BidModifierExpressionParameter.PRISMA_INCOME_GRADE)
                .withValueString(incomeGradeEnumFromExternal(incomeGradeAdjustmentAdd.getGrade()));
        return new BidModifierPrismaIncomeGradeAdjustment()
                .withPercent(incomeGradeAdjustmentAdd.getBidModifier())
                .withCondition(Collections.singletonList(Collections.singletonList(condition)));
    }

    private List<BidModifier> convert(List<BidModifierAddItem> items) {
        return items.stream().map(item -> {
            BidModifier result;
            if (item.getMobileAdjustment() != null) {
                result = new BidModifierMobile()
                        .withType(BidModifierType.MOBILE_MULTIPLIER)
                        .withMobileAdjustment(
                                new BidModifierMobileAdjustment()
                                        .withOsType(OS_TYPE_CONVERTER.convert(
                                                item.getMobileAdjustment().getOperatingSystemType()))
                                        .withPercent(item.getMobileAdjustment().getBidModifier()));
            } else if (item.getTabletAdjustment() != null) {
                result = new BidModifierTablet()
                        .withType(BidModifierType.TABLET_MULTIPLIER)
                        .withTabletAdjustment(
                                new BidModifierTabletAdjustment()
                                        .withOsType(TABLET_OS_TYPE_CONVERTER.convert(
                                                item.getTabletAdjustment().getOperatingSystemType()))
                                        .withPercent(item.getTabletAdjustment().getBidModifier()));
            } else if (!item.getDemographicsAdjustments().isEmpty()) {
                result = new BidModifierDemographics()
                        .withType(BidModifierType.DEMOGRAPHY_MULTIPLIER)
                        .withDemographicsAdjustments(
                                item.getDemographicsAdjustments().stream().map(
                                        adj -> new BidModifierDemographicsAdjustment()
                                                .withAge(ageTypeFromExternal(adj.getAge()))
                                                .withGender(genderTypeFromExternal(adj.getGender()))
                                                .withPercent(adj.getBidModifier())).collect(toList()));
            } else if (!item.getRetargetingAdjustments().isEmpty()) {
                result = new BidModifierRetargeting()
                        .withType(BidModifierType.RETARGETING_MULTIPLIER)
                        .withRetargetingAdjustments(
                                item.getRetargetingAdjustments().stream().map(
                                        adj -> new BidModifierRetargetingAdjustment()
                                                .withRetargetingConditionId(adj.getRetargetingConditionId())
                                                .withPercent(adj.getBidModifier())).collect(toList()));
            } else if (!item.getRegionalAdjustments().isEmpty()) {
                result = new BidModifierGeo()
                        .withType(BidModifierType.GEO_MULTIPLIER)
                        .withRegionalAdjustments(
                                item.getRegionalAdjustments().stream().map(
                                        adj -> new BidModifierRegionalAdjustment()
                                                .withRegionId(adj.getRegionId())
                                                .withHidden(false)
                                                .withPercent(adj.getBidModifier())).collect(toList()));
            } else if (item.getVideoAdjustment() != null) {
                result = new BidModifierVideo()
                        .withType(BidModifierType.VIDEO_MULTIPLIER)
                        .withVideoAdjustment(
                                new BidModifierVideoAdjustment()
                                        .withPercent(item.getVideoAdjustment().getBidModifier()));
            } else if (item.getDesktopAdjustment() != null) {
                result = new BidModifierDesktop()
                        .withType(BidModifierType.DESKTOP_MULTIPLIER)
                        .withDesktopAdjustment(
                                new BidModifierDesktopAdjustment()
                                        .withPercent(item.getDesktopAdjustment().getBidModifier()));
            } else if (item.getDesktopOnlyAdjustment() != null) {
                result = new BidModifierDesktopOnly()
                        .withType(BidModifierType.DESKTOP_ONLY_MULTIPLIER)
                        .withDesktopOnlyAdjustment(
                                new BidModifierDesktopOnlyAdjustment()
                                        .withPercent(item.getDesktopOnlyAdjustment().getBidModifier()));
            } else if (item.getSmartTvAdjustment() != null) {
                result = new BidModifierSmartTV()
                        .withType(BidModifierType.SMARTTV_MULTIPLIER)
                        .withSmartTVAdjustment(
                                new BidModifierSmartTVAdjustment()
                                        .withPercent(item.getSmartTvAdjustment().getBidModifier()));
            } else if (item.getSmartAdAdjustment() != null) {
                result = new BidModifierPerformanceTgo()
                        .withType(BidModifierType.PERFORMANCE_TGO_MULTIPLIER)
                        .withPerformanceTgoAdjustment(
                                new BidModifierPerformanceTgoAdjustment()
                                        .withPercent(item.getSmartAdAdjustment().getBidModifier()));
            } else if (!item.getIncomeGradeAdjustments().isEmpty()) {
                result = toPrismaIncomeGradeBidModifier(item.getIncomeGradeAdjustments());
            } else if (!item.getSerpLayoutAdjustments().isEmpty()) {
                result = new BidModifierTrafaretPosition()
                        .withType(BidModifierType.TRAFARET_POSITION_MULTIPLIER)
                        .withTrafaretPositionAdjustments(
                                item.getSerpLayoutAdjustments().stream().map(
                                        adj -> new BidModifierTrafaretPositionAdjustment()
                                                .withTrafaretPosition(positionFromExternal(adj.getSerpLayout()))
                                                .withPercent(adj.getBidModifier())).collect(toList()));
            } else {
                throw new IllegalStateException("Unsupported BidModifierAddItem");
            }

            return result.withCampaignId(item.getCampaignId())
                    .withAdGroupId(item.getAdGroupId())
                    .withEnabled(true);
        }).collect(toList());
    }

    @Override
    public ApiMassResult<List<Long>> processList(List<BidModifierAddItem> validItems) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();

        // Пробуем добавить, результат может содержать как элементы с успешным результатом, так и элементы с ошибками
        List<BidModifier> convertedBidModifiers = convert(validItems);
        MassResult<List<Long>> massResult = bidModifierService.add(convertedBidModifiers, clientId, operatorUid);

        return resultConverter.toApiMassResult(massResult, HOLDER);
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<List<Long>>>> results) {
        return new AddResponse()
                .withAddResults(resultConverter.convertToMultiIdsActionResults(results, this.apiPathConverter));
    }

    /**
     * Возвращает количество установленных adjustment'ов на элементе.
     */
    private long getAdjustmentsCount(BidModifierAddItem addItem) {
        long singleAdjustmentCount = Stream
                .of(addItem.getMobileAdjustment(),
                        addItem.getTabletAdjustment(),
                        addItem.getVideoAdjustment(),
                        addItem.getDesktopAdjustment(),
                        addItem.getDesktopOnlyAdjustment(),
                        addItem.getSmartTvAdjustment(),
                        addItem.getSmartAdAdjustment())
                .filter(Objects::nonNull)
                .count();
        long noSingleAdjustmentsCount = Stream
                .of(addItem.getRegionalAdjustments(),
                        addItem.getDemographicsAdjustments(),
                        addItem.getRetargetingAdjustments(),
                        addItem.getSerpLayoutAdjustments(),
                        addItem.getIncomeGradeAdjustments())
                .filter(Objects::nonNull)
                .filter(adjs -> !adjs.isEmpty())
                .count();
        return singleAdjustmentCount + noSingleAdjustmentsCount;
    }
}
