package ru.yandex.direct.api.v5.entity.bidmodifiers.delegate;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.bidmodifiers.DeleteRequest;
import com.yandex.direct.api.v5.bidmodifiers.DeleteResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.BID_MODIFIERS_DELETE_ITEMS_LIMIT;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.validation.BidModifiersDefectPresentations.HOLDER;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;

@Component
@ParametersAreNonnullByDefault
public class DeleteBidModifiersDelegate
        extends BaseApiServiceDelegate<DeleteRequest, DeleteResponse, List<Long>, ApiResult<Long>> {
    private static final DefectType DELETE_MAX_IDS_DETAILED = maxElementsInSelection(BID_MODIFIERS_DELETE_ITEMS_LIMIT);

    private final BidModifierService bidModifierService;
    private final ResultConverter resultConverter;

    @Autowired
    public DeleteBidModifiersDelegate(BidModifierService bidModifierService,
                                      ResultConverter resultConverter,
                                      ApiAuthenticationSource auth,
                                      PpcPropertiesSupport ppcPropertiesSupport,
                                      FeatureService featureService) {
        super(PathConverter.identity(), auth);
        this.bidModifierService = bidModifierService;
        this.resultConverter = resultConverter;
    }

    @Override
    public List<Long> convertRequest(DeleteRequest externalRequest) {
        return externalRequest.getSelectionCriteria().getIds();
    }

    @Override
    public ValidationResult<DeleteRequest, DefectType> validateRequest(DeleteRequest request) {
        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getSelectionCriteria(), "SelectionCriteria")
                .item(request.getSelectionCriteria().getIds(), "Ids")
                .check(Constraints.maxListSize(BID_MODIFIERS_DELETE_ITEMS_LIMIT), DELETE_MAX_IDS_DETAILED);
        return vb.getResult();
    }

    @Override
    public ApiResult<List<ApiResult<Long>>> processRequest(List<Long> ids) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();
        MassResult<Long> massResult = bidModifierService.delete(ids, clientId, operatorUid);

        return resultConverter.toApiMassResult(massResult, HOLDER);
    }

    @Override
    public DeleteResponse convertResponse(ApiResult<List<ApiResult<Long>>> results) {
        return new DeleteResponse()
                .withDeleteResults(resultConverter.toActionResults(results, this.apiPathConverter));
    }
}
