package ru.yandex.direct.api.v5.entity.bidmodifiers.delegate;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.yandex.direct.api.v5.bidmodifiers.BidModifierSetItem;
import com.yandex.direct.api.v5.bidmodifiers.SetRequest;
import com.yandex.direct.api.v5.bidmodifiers.SetResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.bidmodifiers.Constants;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.UntypedBidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.api.v5.entity.bidmodifiers.BidModifiersDefectTypes.adjustmentNotFound;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.BidModifiersDefectTypes.setItemsLimitExceeded;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.validation.BidModifiersDefectPresentations.HOLDER;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidId;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class SetBidModifiersDelegate
        extends OperationOnListDelegate<SetRequest, SetResponse, ModelChanges<BidModifierAdjustment>, Long> {
    private final BidModifierService bidModifierService;
    private final ResultConverter resultConverter;
    private final BidModifierTypeSupportDispatcher typeSupportDispatcher;

    @Autowired
    public SetBidModifiersDelegate(BidModifierService bidModifierService, ResultConverter resultConverter,
                                   BidModifierTypeSupportDispatcher typeSupportDispatcher,
                                   ApiAuthenticationSource auth,
                                   PpcPropertiesSupport ppcPropertiesSupport,
                                   FeatureService featureService) {
        super(PathConverter.identity(), auth, ppcPropertiesSupport, featureService);
        this.bidModifierService = bidModifierService;
        this.resultConverter = resultConverter;
        this.typeSupportDispatcher = typeSupportDispatcher;
    }

    /**
     * Валидация всего запроса целиком -- если она не проходит без ошибок, пользователю сразу возвращается ответ.
     */
    @Nullable
    @Override
    public ValidationResult<SetRequest, DefectType> validateRequest(SetRequest externalRequest) {
        ItemValidationBuilder<SetRequest, DefectType> vb = ModelItemValidationBuilder.of(externalRequest);
        vb.list(externalRequest.getBidModifiers(), "bidModifiers").check(
                fromPredicate(allItems -> allItems.size() <= Constants.BID_MODIFIERS_SET_ITEMS_LIMIT,
                        setItemsLimitExceeded(Constants.BID_MODIFIERS_SET_ITEMS_LIMIT)));
        return vb.getResult();
    }

    @Override
    public List<ModelChanges<BidModifierAdjustment>> convertRequest(SetRequest externalRequest) {
        return externalRequest.getBidModifiers().stream().map(this::convert).collect(Collectors.toList());
    }

    private ModelChanges<BidModifierAdjustment> convert(BidModifierSetItem setItem) {
        Long externalId = setItem.getId();

        if (externalId <= 0) {
            // Если переданный id <= 0, возвращаем пустой ModelChanges с id = -1
            // Это будет учтено в дальнейшей валидации internalRequest'а
            ModelChanges<UntypedBidModifierAdjustment> dummy =
                    new ModelChanges<>(-1L, UntypedBidModifierAdjustment.class);
            return dummy.castModelUp(BidModifierAdjustment.class);
        }

        BidModifierType type = BidModifierService.getRealType(externalId);
        if (type == null) {
            // Если по префиксу невозможно определить тип корректировки, то возвращаем
            // пустой ModelChanges с id = 0
            // Это будет учтено в дальнейшей валидации internalRequest'а
            ModelChanges<UntypedBidModifierAdjustment> dummy =
                    new ModelChanges<>(0L, UntypedBidModifierAdjustment.class);
            return dummy.castModelUp(BidModifierAdjustment.class);
        }

        Long realId = BidModifierService.getRealId(externalId);
        Preconditions.checkState(realId != null);

        ModelChanges<BidModifierAdjustment> modelChanges = new ModelChanges<>(realId,
                typeSupportDispatcher.getTypeSupport(type).getAdjustmentClass());
        modelChanges.process(setItem.getBidModifier(), BidModifierAdjustment.PERCENT);
        return modelChanges;
    }

    @Nonnull
    @Override
    public ValidationResult<List<ModelChanges<BidModifierAdjustment>>, DefectType> validateInternalRequest(
            List<ModelChanges<BidModifierAdjustment>> modelChanges) {
        ListValidationBuilder<ModelChanges<BidModifierAdjustment>, DefectType> vb =
                ListValidationBuilder.of(modelChanges);

        // Если на этапе конвертации мы встретили неположительный id или id с неизвестным префиксом,
        // то здесь мы это учитываем и возвращаем на таких элементах соответствующие дефекты
        vb.checkEachBy(mc -> {
            ItemValidationBuilder<ModelChanges<BidModifierAdjustment>, DefectType> ivb =
                    ItemValidationBuilder.of(mc);
            ivb.item(mc.getId(), "Id")
                    .check(fromPredicate(id -> id >= 0, invalidId()))
                    .check(fromPredicate(id -> id != 0, adjustmentNotFound()), When.isValid());
            return ivb.getResult();
        });

        return vb.getResult();
    }

    @Override
    public ApiMassResult<Long> processList(List<ModelChanges<BidModifierAdjustment>> validModelChanges) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();

        // Выполняем изменения, результат может содержать как элементы с успешным результатом, так и элементы с ошибками
        MassResult<Long> massResult = bidModifierService.set(
                validModelChanges, clientId, operatorUid);

        return resultConverter.toApiMassResult(massResult, HOLDER);
    }

    @Override
    public SetResponse convertResponse(ApiResult<List<ApiResult<Long>>> results) {
        return new SetResponse()
                .withSetResults(resultConverter.toActionResults(results, this.apiPathConverter));
    }
}
