package ru.yandex.direct.api.v5.entity.bidmodifiers.delegate;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum;
import com.yandex.direct.api.v5.bidmodifiers.ToggleRequest;
import com.yandex.direct.api.v5.bidmodifiers.ToggleResponse;
import com.yandex.direct.api.v5.bidmodifiers.ToggleResult;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.UntypedBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum.DEMOGRAPHICS_ADJUSTMENT;
import static com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum.INCOME_GRADE_ADJUSTMENT;
import static com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum.REGIONAL_ADJUSTMENT;
import static com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum.RETARGETING_ADJUSTMENT;
import static com.yandex.direct.api.v5.bidmodifiers.BidModifierToggleTypeEnum.SERP_LAYOUT_ADJUSTMENT;
import static com.yandex.direct.api.v5.general.YesNoEnum.YES;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.BidModifiersDefectTypes.toggleItemsLimitExceeded;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.BID_MODIFIERS_TOGGLE_ITEMS_LIMIT;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.validation.BidModifiersDefectPresentations.HOLDER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DEMOGRAPHY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.GEO_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.RETARGETING_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.TRAFARET_POSITION_MULTIPLIER;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class ToggleBidModifiersDelegate
        extends BaseApiServiceDelegate<ToggleRequest, ToggleResponse, List<UntypedBidModifier>,
        ApiResult<UntypedBidModifier>> {

    private final BidModifierService bidModifierService;
    private final ResultConverter resultConverter;

    @Autowired
    public ToggleBidModifiersDelegate(BidModifierService bidModifierService, ResultConverter resultConverter,
                                      ApiAuthenticationSource auth) {
        super(PathConverter.identity(), auth);
        this.bidModifierService = bidModifierService;
        this.resultConverter = resultConverter;
    }

    @Override
    public List<UntypedBidModifier> convertRequest(ToggleRequest externalRequest) {
        return externalRequest.getBidModifierToggleItems().stream()
                .map(e -> (UntypedBidModifier) new UntypedBidModifier()
                        .withCampaignId(e.getCampaignId())
                        .withAdGroupId(e.getAdGroupId())
                        .withType(convertTypeToInternal(e.getType()))
                        .withEnabled(e.getEnabled() == YES))
                .collect(Collectors.toList());
    }

    private BidModifierType convertTypeToInternal(BidModifierToggleTypeEnum type) {
        switch (type) {
            case DEMOGRAPHICS_ADJUSTMENT:
                return DEMOGRAPHY_MULTIPLIER;

            case RETARGETING_ADJUSTMENT:
                return RETARGETING_MULTIPLIER;

            case REGIONAL_ADJUSTMENT:
                return GEO_MULTIPLIER;

            case SERP_LAYOUT_ADJUSTMENT:
                return TRAFARET_POSITION_MULTIPLIER;

            case INCOME_GRADE_ADJUSTMENT:
                return PRISMA_INCOME_GRADE_MULTIPLIER;

            default:
                throw new IllegalStateException("Invalid value: " + type);
        }
    }

    private BidModifierToggleTypeEnum convertTypeToExternal(BidModifierType type) {
        switch (type) {
            case DEMOGRAPHY_MULTIPLIER:
                return DEMOGRAPHICS_ADJUSTMENT;

            case RETARGETING_MULTIPLIER:
                return RETARGETING_ADJUSTMENT;

            case GEO_MULTIPLIER:
                return REGIONAL_ADJUSTMENT;

            case TRAFARET_POSITION_MULTIPLIER:
                return SERP_LAYOUT_ADJUSTMENT;

            case PRISMA_INCOME_GRADE_MULTIPLIER:
                return INCOME_GRADE_ADJUSTMENT;

            default:
                throw new IllegalStateException("Invalid value: " + type);
        }
    }

    @Override
    public ValidationResult<ToggleRequest, DefectType> validateRequest(ToggleRequest externalRequest) {
        return validateRequestCore(externalRequest);
    }

    public ValidationResult<ToggleRequest, DefectType> validateRequestCore(ToggleRequest externalRequest) {

        ItemValidationBuilder<ToggleRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getBidModifierToggleItems(), "ToggleItems")
                .check(fromPredicate(allItems -> allItems.size() <= BID_MODIFIERS_TOGGLE_ITEMS_LIMIT,
                        toggleItemsLimitExceeded(BID_MODIFIERS_TOGGLE_ITEMS_LIMIT)));
        return vb.getResult();
    }

    @Override
    public ApiResult<List<ApiResult<UntypedBidModifier>>> processRequest(List<UntypedBidModifier> items) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();
        MassResult<UntypedBidModifier> massResult = bidModifierService.toggle(items, clientId, operatorUid);

        return resultConverter.toApiMassResult(massResult, HOLDER);
    }

    @Override
    public ToggleResponse convertResponse(ApiResult<List<ApiResult<UntypedBidModifier>>> results) {

        List<ToggleResult> result =
                StreamEx.of(results.getResult())
                        .mapToEntry(r -> resultConverter.convertToGenericActionResult(r, this.apiPathConverter))
                        .mapKeyValue((r, actionResult) -> {
                            ToggleResult rr = new ToggleResult();
                            if (r.isSuccessful()) {
                                BidModifier bidItem = r.getResult();
                                rr.withAdGroupId(bidItem.getAdGroupId())
                                        .withCampaignId(bidItem.getCampaignId())
                                        .withType(convertTypeToExternal(bidItem.getType()))
                                ;
                            }
                            // getErrors() и getWarnings() под капотом лениво инициализируют списки ошибок и
                            // предупреждений
                            // проверяем списки на наличие элементов, чтобы не отдавать "Errors" и "Warnings", если
                            // ошибок не было
                            // See also ResultConverter.toBidActionResult()
                            if (!actionResult.getErrors().isEmpty()) {
                                rr.setErrors(actionResult.getErrors());
                            }
                            if (!actionResult.getWarnings().isEmpty()) {
                                rr.setWarnings(actionResult.getWarnings());
                            }
                            return rr;
                        })
                        .toList();
        return new ToggleResponse().withToggleResults(result);
    }
}
