package ru.yandex.direct.api.v5.entity.bidmodifiers.validation;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.bidmodifiers.BidModifiersSelectionCriteria;
import com.yandex.direct.api.v5.bidmodifiers.GetRequest;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Arrays.asList;
import static ru.yandex.direct.api.v5.common.ConverterUtils.getLimit;
import static ru.yandex.direct.api.v5.common.ConverterUtils.getOffset;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.ADGROUP_IDS_LIMIT;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.CAMPAIGN_IDS_LIMIT;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.GET_HARD_LIMIT;
import static ru.yandex.direct.api.v5.entity.bidmodifiers.Constants.IDS_LIMIT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.missedParamsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.tooBroadSelectionClause;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;

@ParametersAreNonnullByDefault
@Service
public class GetBidModifiersValidationService {
    public static final String SELECTION_CRITERIA_PATH_KEY = "SelectionCriteria";

    public static final String CAMPAIGN_IDS_PATH_KEY = "CampaignIds";
    public static final String AD_GROUP_IDS_PATH_KEY = "AdGroupIds";
    public static final String IDS_PATH_KEY = "Ids";

    public static final String PAGE_PATH_KEY = "Page";

    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> ivb = ItemValidationBuilder.of(externalRequest);

        BidModifiersSelectionCriteria selectionCriteria = externalRequest.getSelectionCriteria();

        // Проверка offset, limit
        ivb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes);

        // Проверка offset+limit
        ivb.checkBy(request -> {
            ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(request);
            vb.item(request.getPage(), PAGE_PATH_KEY)
                    .check(
                            Constraint.fromPredicate(
                                    page -> getLimit(page) + getOffset(page) <= GET_HARD_LIMIT,
                                    tooBroadSelectionClause(GET_HARD_LIMIT)),
                            When.notNull());
            return vb.getResult();
        });

        ItemValidationBuilder<BidModifiersSelectionCriteria, DefectType> selectionCriteriaValidation =
                ivb.item(selectionCriteria, SELECTION_CRITERIA_PATH_KEY);

        selectionCriteriaValidation.check(atLeastOneRequiredParamPresent());

        selectionCriteriaValidation
                .item(selectionCriteria.getCampaignIds(), CAMPAIGN_IDS_PATH_KEY)
                .check(maxListSize(CAMPAIGN_IDS_LIMIT), maxElementsInSelection(CAMPAIGN_IDS_LIMIT));
        selectionCriteriaValidation
                .item(selectionCriteria.getAdGroupIds(), AD_GROUP_IDS_PATH_KEY)
                .check(maxListSize(ADGROUP_IDS_LIMIT), maxElementsInSelection(ADGROUP_IDS_LIMIT));
        selectionCriteriaValidation
                .item(selectionCriteria.getIds(), IDS_PATH_KEY)
                .check(maxListSize(IDS_LIMIT), maxElementsInSelection(IDS_LIMIT));

        return ivb.getResult();
    }

    private static Constraint<BidModifiersSelectionCriteria, DefectType> atLeastOneRequiredParamPresent() {
        return Constraint.fromPredicate(
                bsc -> !bsc.getIds().isEmpty() || !bsc.getAdGroupIds().isEmpty() || !bsc.getCampaignIds().isEmpty(),
                missedParamsInSelection(String.join(
                        ", ", asList(CAMPAIGN_IDS_PATH_KEY, AD_GROUP_IDS_PATH_KEY, IDS_PATH_KEY))));
    }
}
