package ru.yandex.direct.api.v5.entity.bids.converter.get;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.bids.AuctionBidItem;
import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.SearchPrices;
import com.yandex.direct.api.v5.general.PositionEnum;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.auction.container.bs.Block;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.Position;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidDynamicData;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.MoneyUtils;

import static ru.yandex.direct.api.v5.entity.bids.converter.get.BidGetItemWriterComposition.writerCompositionOf;

@ParametersAreNonnullByDefault
public class BsAuctionBidFieldWriter implements BidGetItemWriter {

    private final Set<BidFieldEnum> requiredFields;
    private final Map<Long, KeywordBidBsAuctionData> bsResultsById;

    private final BidGetItemWriter writer;

    private BsAuctionBidFieldWriter(Set<BidFieldEnum> requiredFields,
                                    Map<Long, KeywordBidBsAuctionData> bsResultsById) {
        this.requiredFields = requiredFields;
        this.bsResultsById = bsResultsById;
        writer = writerCompositionOf(
                getMinSearchPriceWriter(),
                getCurrentSearchPriceWriter(),
                getAuctionBidsWriter(),
                getSearchPricesWriter(),
                getCompetitorsBidsWriter()
        );
    }

    static BsAuctionBidFieldWriter createBsAuctionBidFieldWriter(Set<BidFieldEnum> requiredFields,
                                                                 Collection<CompleteBidData<KeywordBidBsAuctionData>> completeBidData) {
        Map<Long, KeywordBidBsAuctionData> bsAuctionDataByBidId =
                StreamEx.of(completeBidData)
                        .mapToEntry(CompleteBidData::getBidId)
                        .invert()
                        .mapValues(CompleteBidData::getDynamicData)
                        .nonNullValues()
                        .mapValues(KeywordBidDynamicData::getBsAuctionData)
                        .nonNullValues()
                        .toMap();
        return new BsAuctionBidFieldWriter(requiredFields, bsAuctionDataByBidId);
    }


    @Override
    public void write(BidGetItem bid, Long bidId) {
        writer.write(bid, bidId);
    }

    @Nullable
    private BidGetItemWriter getBidGetItemWriter(BidFieldEnum keywordIdField, BidGetItemWriter keywordIdWriter) {
        return !requiredFields.contains(keywordIdField) ? null : keywordIdWriter;
    }

    private Optional<KeywordBidBsAuctionData> getBsResultFor(Long id) {
        return Optional.ofNullable(bsResultsById.get(id));
    }

    private BidGetItemWriter getMinSearchPriceWriter() {
        return getBidGetItemWriter(BidFieldEnum.MIN_SEARCH_PRICE, (item, id) ->
                getBsResultFor(id).map(KeywordBidBsAuctionData::getMinPrice).map(Money::micros).ifPresent(
                        // если от Торгов не пришла минимальная ставка, в minSearchPrice сохранится значение, проставленное из БД в DirectBidFieldWriter
                        minPrice -> item.setMinSearchPrice(
                                new JAXBElement<>(new QName("", "MinSearchPrice"), Long.class, minPrice))));
    }

    private BidGetItemWriter getCurrentSearchPriceWriter() {
        return getBidGetItemWriter(BidFieldEnum.CURRENT_SEARCH_PRICE, (item, id) ->
                item.setCurrentSearchPrice(getCurrentSearchPrice(id)));
    }

    @Nonnull
    private JAXBElement<Long> getCurrentSearchPrice(Long id) {
        return new JAXBElement<>(new QName("", "CurrentSearchPrice"), Long.class,
                getBsResultFor(id).map(KeywordBidBsAuctionData::getBroker)
                        .map(Money::micros)
                        .orElse(null)
        );
    }

    private BidGetItemWriter getCompetitorsBidsWriter() {
        return getBidGetItemWriter(BidFieldEnum.COMPETITORS_BIDS, (item, id) ->
                item.setCompetitorsBids(getBsResultFor(id).map(this::calcCompetitorsBidsFromBsResult).orElse(null))
        );
    }

    private List<Long> calcCompetitorsBidsFromBsResult(KeywordBidBsAuctionData bsResult) {
        // distinct() отсутствует специально. Дубли из ответа Торгов отдаются в CompetitorsBids. See also CompetitorsBidsWithAuctionBidsTest
        return StreamEx.of(bsResult.getPremium().allPositions())
                .append(StreamEx.of(bsResult.getGuarantee().allPositions()))
                .map(Position::getBidPrice)
                .map(BsAuctionBidFieldWriter::getShowValue)
                .map(Money::micros)
                .reverseSorted()
                .toList();
    }

    private BidGetItemWriter getAuctionBidsWriter() {
        return getBidGetItemWriter(BidFieldEnum.AUCTION_BIDS, (item, id) ->
                item.setAuctionBids(getBsResultFor(id).map(this::convertAuctionDataToAuctionBidItems).orElse(null))
        );
    }

    private List<AuctionBidItem> convertAuctionDataToAuctionBidItems(KeywordBidBsAuctionData bsResult) {
        List<AuctionBidItem> result = new ArrayList<>();

        List<AuctionBidItem> premiumItems = transformPositionsToAuctionBidItems("P1", bsResult.getPremium());
        result.addAll(premiumItems);

        List<AuctionBidItem> guaranteeItems = transformPositionsToAuctionBidItems("P2", bsResult.getGuarantee());
        result.addAll(guaranteeItems);
        return result;
    }

    @Nonnull
    private List<AuctionBidItem> transformPositionsToAuctionBidItems(String positionPrefix,
                                                                     Block block) {
        List<AuctionBidItem> res = new ArrayList<>();
        for (int i = 0; i < block.size(); i++) {
            Position pos = block.get(i);
            String positionId = positionPrefix + (i + 1);
            AuctionBidItem item = new AuctionBidItem()
                    .withPosition(positionId)
                    .withBid(getShowValue(pos.getBidPrice()).micros())
                    .withPrice(getShowValue(pos.getAmnestyPrice()).micros());
            res.add(item);
        }
        return res;
    }

    /**
     * в торгах подняли лимит для максимального значения ставки, на время переходного этапа (DIRECT-74109)
     * ограничиваем значение торгов до MaxShowBid (для РФ это 5 000 рублей)
     */
    public static Money getShowValue(Money money) {
        return MoneyUtils.min(
                money,
                Money.valueOf(money.getCurrencyCode().getCurrency().getMaxShowBid(), money.getCurrencyCode())
        );
    }

    private BidGetItemWriter getSearchPricesWriter() {
        return getBidGetItemWriter(BidFieldEnum.SEARCH_PRICES, (item, id) ->
                item.setSearchPrices(
                        getBsResultFor(id).map(this::convertAuctionDataToSearchPrices).orElse(null))
        );
    }

    private List<SearchPrices> convertAuctionDataToSearchPrices(KeywordBidBsAuctionData bsResult) {
        return EntryStream.of(ImmutableMap.of(
                PositionEnum.PREMIUMFIRST,
                (Function<KeywordBidBsAuctionData, Position>) (res) -> res.getPremium().first(),
                PositionEnum.PREMIUMBLOCK, (res) -> res.getPremium().last(),
                PositionEnum.FOOTERFIRST, (res) -> res.getGuarantee().first(),
                PositionEnum.FOOTERBLOCK, (res) -> res.getGuarantee().last()
        ))
                .mapKeyValue(((positionEnum, selector) -> new SearchPrices()
                        .withPosition(positionEnum)
                        .withPrice(getShowValue(selector.apply(bsResult).getBidPrice()).micros())
                )).toList();
    }

}
