package ru.yandex.direct.api.v5.entity.bids.converter.get;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import com.yandex.direct.api.v5.bids.BidActionResult;
import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.general.ServingStatusEnum;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertStrategyPriority;
import static ru.yandex.direct.api.v5.entity.bids.converter.get.BidGetItemWriterComposition.writerCompositionOf;

/**
 * Реализация {@link BidGetItemWriter}, заполняющая поля {@link BidGetItem} на основе данных, хранящихся в Директе
 */
@ParametersAreNonnullByDefault
public class DirectBidFieldWriter implements BidGetItemWriter {

    private final Set<BidFieldEnum> requiredFields;
    private final Map<Long, CompleteBidData<KeywordBidBsAuctionData>> completeBidDataById;

    private final BidGetItemWriter writer;

    private DirectBidFieldWriter(Set<BidFieldEnum> requiredFields,
                                 Map<Long, CompleteBidData<KeywordBidBsAuctionData>> completeBidDataById) {
        this.requiredFields = requiredFields;
        this.completeBidDataById = completeBidDataById;
        writer = writerCompositionOf(
                getKeywordIdWriter(),
                getAdGroupIdWriter(),
                getCampaignIdWriter(),
                getBidWriter(),
                getContextBidWriter(),
                getStrategyPriorityWriter(),
                getServingStatusWriter(),
                getMinSearchPriceWriter());
    }

    /**
     * @see DirectBidFieldWriter
     */
    static DirectBidFieldWriter createDirectBidFieldWriter(Set<BidFieldEnum> requiredFields,
                                                           Collection<CompleteBidData<KeywordBidBsAuctionData>> completeBidData) {
        Map<Long, CompleteBidData<KeywordBidBsAuctionData>> completeBidDataById = StreamEx.of(completeBidData)
                .mapToEntry(CompleteBidData::getBidId)
                .invert()
                .toMap();
        return new DirectBidFieldWriter(requiredFields, completeBidDataById);
    }


    @Override
    public void write(BidGetItem bid, Long bidId) {
        writer.write(bid, bidId);
    }

    private BidGetItemWriter getKeywordIdWriter() {
        return getBidGetItemWriter(BidFieldEnum.KEYWORD_ID, BidActionResult::setKeywordId);
    }

    private BidGetItemWriter getAdGroupIdWriter() {
        return getBidGetItemWriter(BidFieldEnum.AD_GROUP_ID,
                (item, id) -> item.setAdGroupId(getPhrase(id).getAdGroupId()));
    }

    private BidGetItemWriter getCampaignIdWriter() {
        return getBidGetItemWriter(BidFieldEnum.CAMPAIGN_ID,
                (item, id) -> item.setCampaignId(getPhrase(id).getCampaignId()));
    }

    private BidGetItemWriter getBidWriter() {
        return getBidGetItemWriter(BidFieldEnum.BID,
                (item, id) -> item.setBid(priceToExternalBid(id, getPhrase(id).getPrice())));
    }

    private BidGetItemWriter getContextBidWriter() {
        return getBidGetItemWriter(BidFieldEnum.CONTEXT_BID,
                (item, id) -> item.setContextBid(priceToExternalBid(id, getPhrase(id).getPriceContext())));
    }

    /**
     * Переводит ставку из внутреннего формата во внешний.
     * Нулевые ставки превращаются в минимальные для данной валюты
     *
     * @param id    id ключевого слова
     * @param price ставка
     * @return ставка во внешнем формате в микроденьгах
     */
    private Long priceToExternalBid(Long id, BigDecimal price) {
        return (price.compareTo(BigDecimal.ZERO) == 0) ?
                getBidCurrencyMinPriceMicro(id) :
                price.multiply(Money.MICRO_MULTIPLIER).longValue();
    }

    private BidGetItemWriter getStrategyPriorityWriter() {
        return getBidGetItemWriter(BidFieldEnum.STRATEGY_PRIORITY,
                (item, id) -> item.setStrategyPriority(new JAXBElement<>(
                        new QName("", BidFieldEnum.STRATEGY_PRIORITY.value()),
                        PriorityEnum.class,
                        convertStrategyPriority(getPhrase(id).getAutobudgetPriority()))));
    }

    private BidGetItemWriter getServingStatusWriter() {
        return getBidGetItemWriter(BidFieldEnum.SERVING_STATUS, (item, id) -> item
                .setServingStatus(getAdGroupByPhraseId(id).getBsRarelyLoaded()
                        ? ServingStatusEnum.RARELY_SERVED
                        : ServingStatusEnum.ELIGIBLE));
    }

    @Nullable
    private BidGetItemWriter getMinSearchPriceWriter() {
        return getBidGetItemWriter(BidFieldEnum.MIN_SEARCH_PRICE, (item, id) -> {
            boolean bsRarelyLoaded = Boolean.TRUE.equals(getAdGroupByPhraseId(id).getBsRarelyLoaded());
            // не устанавливаем MinSearchPrice если "мало показов"
            Long minPrice;
            if (bsRarelyLoaded) {
                minPrice = null;
            } else {
                // заполняем минимальной ставкой
                // это значение может быть переписано более точными данными из торгов в соседнем writer'е
                minPrice = getBidCurrencyMinPriceMicro(id);
            }
            item.setMinSearchPrice(new JAXBElement<>(new QName("", "MinSearchPrice"), Long.class,
                    minPrice));
        });
    }

    private AdGroup getAdGroupByPhraseId(Long id) {
        return completeBidDataById.get(id).getAdGroup();
    }

    private Bid getPhrase(Long id) {
        return completeBidDataById.get(id).getBid();
    }

    /**
     * Возвращает минимальную ставку в валюте кампании, которой принадлежит ключевое слово
     *
     * @param id id ключевого слова
     * @return минимальная ставка в микроденьгах
     */
    private Long getBidCurrencyMinPriceMicro(Long id) {
        Campaign campaign = completeBidDataById.get(id).getCampaign();
        Currency currency = campaign.getCurrency().getCurrency();
        AdGroup adGroup = completeBidDataById.get(id).getAdGroup();
        BigDecimal defaultPrice =
                (adGroup.getType() == AdGroupType.CPM_BANNER) ? currency.getMinCpmPrice() : currency.getMinPrice();
        return defaultPrice.multiply(Money.MICRO_MULTIPLIER).longValue();
    }

    @Nullable
    private BidGetItemWriter getBidGetItemWriter(BidFieldEnum keywordIdField, BidGetItemWriter keywordIdWriter) {
        return !requiredFields.contains(keywordIdField) ? null : keywordIdWriter;
    }
}
