package ru.yandex.direct.api.v5.entity.bids.converter.get;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.ContextCoverage;
import com.yandex.direct.api.v5.bids.ContextCoverageItem;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidDynamicData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.pokazometer.PhraseResponse;

import static ru.yandex.direct.api.v5.entity.bids.converter.get.BidGetItemWriterComposition.writerCompositionOf;

@ParametersAreNonnullByDefault
public class PokazometerBidFieldWriter implements BidGetItemWriter {


    private final Set<BidFieldEnum> requiredFields;
    private final Map<Long, KeywordBidPokazometerData> pokazometerDataByBidId;

    private final BidGetItemWriter writer;

    private PokazometerBidFieldWriter(Set<BidFieldEnum> requiredFields,
                                      Map<Long, KeywordBidPokazometerData> pokazometerDataByBidId) {
        this.requiredFields = requiredFields;
        // используем memoize, чтобы не вызывать supplier лишний раз
        this.pokazometerDataByBidId = pokazometerDataByBidId;
        writer = writerCompositionOf(getContextCoverageWriter());
    }

    static PokazometerBidFieldWriter createPokazometerBidFieldWriter(Set<BidFieldEnum> requiredFields,
                                                                     Collection<CompleteBidData<KeywordBidBsAuctionData>> completeBidData) {
        Map<Long, KeywordBidPokazometerData> pokazometerDataByBidId =
                StreamEx.of(completeBidData)
                        .mapToEntry(CompleteBidData::getBidId)
                        .invert()
                        .mapValues(CompleteBidData::getDynamicData)
                        .nonNullValues()
                        .mapValues(KeywordBidDynamicData::getPokazometerData)
                        .nonNullValues()
                        .toMap();
        return new PokazometerBidFieldWriter(requiredFields, pokazometerDataByBidId);
    }


    @Override
    public void write(BidGetItem bid, Long bidId) {
        writer.write(bid, bidId);
    }

    @Nullable
    private BidGetItemWriter getBidGetItemWriter(BidFieldEnum keywordIdField, BidGetItemWriter keywordIdWriter) {
        return !requiredFields.contains(keywordIdField) ? null : keywordIdWriter;
    }

    @Nullable
    private BidGetItemWriter getContextCoverageWriter() {
        return getBidGetItemWriter(BidFieldEnum.CONTEXT_COVERAGE, (item, id) -> {
                    KeywordBidPokazometerData pokazometerData = pokazometerDataByBidId.get(id);
                    if (pokazometerData == null) {
                        // заполняем ContextCoverage = null, если параметр был запрошен, но от Показометра не пришли данные
                        item.setContextCoverage(
                                new JAXBElement<>(new QName("", "ContextCoverage"), ContextCoverage.class, null));
                        return;
                    }
                    item.setContextCoverage(
                            new JAXBElement<>(new QName("", "ContextCoverage"), ContextCoverage.class,
                                    new ContextCoverage().withItems(
                                            EntryStream.of(pokazometerData.getCoverageWithPrices())
                                                    .mapKeys(PhraseResponse.Coverage::getPercentage)
                                                    .reverseSorted(Comparator.comparingInt(Map.Entry::getKey))
                                                    .mapKeyValue((probability, price) -> new ContextCoverageItem()
                                                            .withPrice(price.micros())
                                                            .withProbability(BigDecimal.valueOf(probability)))
                                                    .collect(Collectors.toList())
                                    )
                            ));
                }
        );
    }
}
