package ru.yandex.direct.api.v5.entity.bids.delegate;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.bids.BidFieldEnum;
import com.yandex.direct.api.v5.bids.BidGetItem;
import com.yandex.direct.api.v5.bids.BidsSelectionCriteria;
import com.yandex.direct.api.v5.bids.GetRequest;
import com.yandex.direct.api.v5.bids.GetResponse;
import com.yandex.direct.api.v5.general.ServingStatusEnum;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.bids.converter.get.GetBidsConverter;
import ru.yandex.direct.api.v5.entity.bids.validation.GetBidsValidationService;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.bids.service.KeywordBidDynamicDataService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessibiltyChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.keyword.model.ServingStatus;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class GetBidsDelegate extends
        GetApiServiceDelegate<GetRequest, GetResponse, BidFieldEnum, ShowConditionSelectionCriteria, CompleteBidData<KeywordBidBsAuctionData>> {

    private static final Set<BidFieldEnum> AUCTION_FIELDS = EnumSet.of(BidFieldEnum.SEARCH_PRICES,
            BidFieldEnum.COMPETITORS_BIDS,
            BidFieldEnum.MIN_SEARCH_PRICE,
            BidFieldEnum.CURRENT_SEARCH_PRICE,
            BidFieldEnum.AUCTION_BIDS);

    private final GetBidsValidationService getBidsValidationService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final BidService bidService;
    private final KeywordBidDynamicDataService keywordBidDynamicDataService;
    private final ResultConverter resultConverter;
    private final ApiContextHolder apiContextHolder;

    @Autowired
    public GetBidsDelegate(ApiAuthenticationSource auth,
                           GetBidsValidationService getBidsValidationService,
                           CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                           BidService bidService,
                           KeywordBidDynamicDataService keywordBidDynamicDataService,
                           ResultConverter resultConverter,
                           ApiContextHolder apiContextHolder) {
        // так как все валидации делаются на уровне API, можем PathConverter брать прямо из ValidationService
        super(getBidsValidationService.pathConverter(), auth);
        this.getBidsValidationService = getBidsValidationService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.bidService = bidService;
        this.keywordBidDynamicDataService = keywordBidDynamicDataService;
        this.resultConverter = resultConverter;
        this.apiContextHolder = apiContextHolder;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        return getBidsValidationService.validate(externalRequest);
    }

    @Override
    public List<Long> returnedCampaignIds(ApiResult<List<CompleteBidData<KeywordBidBsAuctionData>>> result) {
        return mapList(result.getResult(), data -> data.getCampaign().getId());
    }

    @Override
    public ShowConditionSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        BidsSelectionCriteria bidsSelectionCriteria = externalRequest.getSelectionCriteria();
        return new ShowConditionSelectionCriteria()
                .withShowConditionIds(bidsSelectionCriteria.getKeywordIds())
                .withAdGroupIds(bidsSelectionCriteria.getAdGroupIds())
                .withCampaignIds(bidsSelectionCriteria.getCampaignIds())
                .withServingStatuses(Collections.singleton(toServingStatus(bidsSelectionCriteria.getServingStatuses())));
    }

    @Override
    public Set<BidFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public List<CompleteBidData<KeywordBidBsAuctionData>> get(
            GenericGetRequest<BidFieldEnum, ShowConditionSelectionCriteria> getRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Long operatorUid = auth.getOperator().getUid();

        CampaignAccessibiltyChecker campaignAccessibiltyChecker = getCampaignAccessibiltyChecker();
        ShowConditionSelectionCriteria selection = getRequest.getSelectionCriteria();
        if (!selection.getShowConditionIds().isEmpty()) {
            CampaignSubObjectAccessChecker checker =
                    campaignSubObjectAccessCheckerFactory.newBidChecker(operatorUid, clientId, selection.getShowConditionIds());
            selection.getShowConditionIds().removeAll(checker.getUnsupported(campaignAccessibiltyChecker));
        }
        if (!selection.getAdGroupIds().isEmpty()) {
            CampaignSubObjectAccessChecker checker =
                    campaignSubObjectAccessCheckerFactory.newAdGroupChecker(operatorUid, clientId, selection.getAdGroupIds());
            selection.getAdGroupIds().removeAll(checker.getUnsupported(campaignAccessibiltyChecker));
        }
        if (!selection.getCampaignIds().isEmpty()) {
            CampaignSubObjectAccessChecker checker =
                    campaignSubObjectAccessCheckerFactory.newCampaignChecker(operatorUid, clientId, selection.getCampaignIds());
            selection.getCampaignIds().removeAll(checker.getUnsupported(campaignAccessibiltyChecker));
        }

        List<Bid> bids = bidService.getBids(clientId, operatorUid, selection, getRequest.getLimitOffset());

        Set<BidFieldEnum> requestedFields = getRequest.getRequestedFields();
        boolean hasPokazometerFields = hasPokazometerFields(requestedFields);
        boolean hasBsAuctionFields = hasBsAuctionFields(requestedFields);

        boolean safePokazometer = true;
        //noinspection ConstantConditions
        Collection<CompleteBidData<KeywordBidBsAuctionData>> dynamicData = keywordBidDynamicDataService
                .getCompleteBidData(clientId, bids, hasPokazometerFields, hasBsAuctionFields, safePokazometer);

        checkState(dynamicData.size() == bids.size(),
                "Mismatch collections of bids (size: %d) and dynamicBidData (size: %d)",
                bids.size(), dynamicData.size());

        // todo maxlog: сервисы ядра должны возвращать упорядоченный список
        return StreamEx.of(dynamicData)
                .sortedBy(dynData -> dynData.getBid().getId())
                .toList();
    }

    @Override
    public void executeAdditionalActionsOverFetchedItems(List<CompleteBidData<KeywordBidBsAuctionData>> dynamicData) {
        apiContextHolder.get().getApiLogRecord().withCampaignIds(
                StreamEx.of(dynamicData)
                        .map(dd -> dd.getCampaign().getId())
                        .distinct()
                        .toList()
        );
    }

    @Override
    public GetResponse convertGetResponse(List<CompleteBidData<KeywordBidBsAuctionData>> dynamicBidData,
                                          Set<BidFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        List<BidGetItem> items = GetBidsConverter
                .convertCompleteBidDataToBidGetItems(requestedFields, dynamicBidData);

        GetResponse getResponse = new GetResponse();
        if (!items.isEmpty()) {
            getResponse.withBids(items);
        }
        return getResponse.withLimitedBy(limitedBy);
    }

    @Nullable
    private static ServingStatus toServingStatus(List<ServingStatusEnum> servingStatusEnums) {
        List<ServingStatus> servingStatuses = StreamEx.of(servingStatusEnums)
                .map(ServingStatusEnum::value)
                .map(ServingStatus::valueOf)
                .toList();

        // Сейчас возможны два взаимоисключающих статуса: ELIGIBLE и RARELY_SERVED.
        // Если заданы оба или не задан ни один, то возвращаем все ставки
        if (servingStatuses.size() == 1) {
            return servingStatuses.get(0);
        } else {
            return null;
        }
    }

    private boolean hasPokazometerFields(Set<BidFieldEnum> requestedFields) {
        return requestedFields.contains(BidFieldEnum.CONTEXT_COVERAGE);
    }

    private boolean hasBsAuctionFields(Set<BidFieldEnum> requestedFields) {
        return StreamEx.of(requestedFields).findAny(AUCTION_FIELDS::contains).isPresent();
    }
}
