package ru.yandex.direct.api.v5.entity.bids.delegate;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import com.yandex.direct.api.v5.bids.BidSetAutoItem;
import com.yandex.direct.api.v5.bids.SetAutoRequest;
import com.yandex.direct.api.v5.bids.SetAutoResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.bids.converter.BidsHelperConverter;
import ru.yandex.direct.api.v5.entity.bids.service.validation.SetAutoBidsDefectPresentations;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Arrays.asList;
import static ru.yandex.direct.api.v5.entity.OperationOnListUtils.notCountErrorsWithCodes;
import static ru.yandex.direct.api.v5.entity.bids.Constants.MAX_BID_ADGROUPIDS_PER_REQUEST;
import static ru.yandex.direct.api.v5.entity.bids.Constants.MAX_BID_CAMPAIGNIDS_PER_REQUEST;
import static ru.yandex.direct.api.v5.entity.bids.Constants.MAX_BID_IDS_PER_REQUEST;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.bidChangeNotAllowed;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.fieldDoesNotMatchStrategy;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.maxAdGroupsBidsPerRequest;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.maxCampBidsPerRequest;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.maxKeywordBidsPerRequest;
import static ru.yandex.direct.api.v5.validation.SetBidsConstraints.bidsListSizeMaxLimit;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
public class SetAutoBidsDelegate
        extends OperationOnListDelegate<SetAutoRequest, SetAutoResponse, SetAutoBidItem, SetAutoBidItem> {
    /**
     * Пути, ошибки на которых игнорируем, так как при вызове из Bids.setAuto дублируем соответствующие поля в запросе к core
     *
     * @see #filterExtraSetAutoErrors
     */
    private static final Set<Path> SET_AUTO_ERROR_IGNORED_PATH = new HashSet<>(asList(
            path(field("searchByPosition"), field("increasePercent")),
            path(field("searchByPosition"), field("maxBid"))
    ));
    private static final Set<Integer> NO_WITHDRAWAL_UNITS_ERROR_CODES = Set.of(fieldDoesNotMatchStrategy().getCode(),
            bidChangeNotAllowed().getCode());

    private final BidsHelperConverter bidsHelperConverter;
    private final BidService bidService;
    private final ResultConverter resultConverter;

    @Autowired
    public SetAutoBidsDelegate(BidsHelperConverter bidsHelperConverter,
                               ApiAuthenticationSource auth,
                               BidService bidService,
                               ResultConverter resultConverter) {
        super(ApiPathConverter.forSetAutoBids(), auth);
        this.bidsHelperConverter = bidsHelperConverter;
        this.bidService = bidService;
        this.resultConverter = resultConverter;
    }

    @Nonnull
    @Override
    public ValidationResult<SetAutoRequest, DefectType> validateRequest(SetAutoRequest request) {
        ItemValidationBuilder<SetAutoRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getBids(), "Bids")
                .check(eachNotNull())
                .check(bidsListSizeMaxLimit(
                        BidSetAutoItem::getKeywordId, MAX_BID_IDS_PER_REQUEST,
                        t -> maxKeywordBidsPerRequest(),
                        BidSetAutoItem::getAdGroupId, MAX_BID_ADGROUPIDS_PER_REQUEST,
                        t -> maxAdGroupsBidsPerRequest(),
                        BidSetAutoItem::getCampaignId, MAX_BID_CAMPAIGNIDS_PER_REQUEST,
                        t -> maxCampBidsPerRequest()), When.isValid());

        return vb.getResult();
    }

    @Override
    public List<SetAutoBidItem> convertRequest(SetAutoRequest externalRequest) {
        return bidsHelperConverter.convertFromSetAutoRequest(externalRequest);
    }

    @Override
    public ApiMassResult<SetAutoBidItem> processList(List<SetAutoBidItem> internalRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();
        MassResult<SetAutoBidItem> setAutoResult = bidService.setAutoBids(clientId, operatorUid, internalRequest,
                false);
        return resultConverter.toApiMassResult(setAutoResult, SetAutoBidsDefectPresentations.HOLDER);

    }

    @Override
    public SetAutoResponse convertResponse(ApiResult<List<ApiResult<SetAutoBidItem>>> result) {
        ApiResult<List<ApiResult<SetAutoBidItem>>> newResult = filterExtraSetAutoErrors(result);
        return bidsHelperConverter.convertToSetAutoResponse(newResult, apiPathConverter);
    }

    /**
     * При трансформации внешнего запроса во внутренний поле {@code increasePercent} раскладывается в два:
     * {@code searchByPosition.increasePercent} и {@code networkByCoverage.increasePercent}.
     * Ошибки валидации этого поля дублируются. Чтобы отдавать пользователю ответ без дублей, отфильтровываем ошибки
     * привязнные к пути {@code searchByPosition.increasePercent}
     */
    ApiResult<List<ApiResult<SetAutoBidItem>>> filterExtraSetAutoErrors(
            ApiResult<List<ApiResult<SetAutoBidItem>>> result) {
        if (result.getResult() == null) {
            return result;
        }
        List<ApiResult<SetAutoBidItem>> filteredResult = result.getResult().stream()
                .map(r -> {
                    List<DefectInfo<DefectType>> errors = r.getErrors();
                    if (errors == null || errors.isEmpty()) {
                        return r;
                    }
                    List<DefectInfo<DefectType>> filteredErrors = errors.stream()
                            .filter(d -> !SET_AUTO_ERROR_IGNORED_PATH.contains(d.getPath()))
                            .collect(Collectors.toList());
                    return new ApiResult<>(r.getResult(), filteredErrors, r.getWarnings(), r.getState());
                })
                .collect(Collectors.toList());
        return new ApiResult<>(filteredResult, result.getErrors(), result.getWarnings(), result.getState());
    }

    /**
     * Не вычитаем баллы в случае, когда невозможно указать ставку из-за несоответствия стратегии.
     */
    @Override
    public OperationSummary correctOperationSummary(List<SetAutoBidItem> internalRequest,
                                                    ApiResult<List<ApiResult<SetAutoBidItem>>> apiResult) {
        return notCountErrorsWithCodes(apiResult, NO_WITHDRAWAL_UNITS_ERROR_CODES);
    }
}
