package ru.yandex.direct.api.v5.entity.bids.delegate;

import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.bids.SetRequest;
import com.yandex.direct.api.v5.bids.SetResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.bids.converter.BidsHelperConverter;
import ru.yandex.direct.api.v5.entity.bids.service.validation.SetBidsRequestValidationService;
import ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectPresentations;
import ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes;
import ru.yandex.direct.api.v5.entity.bids.validation.BidsInternalValidationService;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessibiltyChecker;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.OperationOnListUtils.notCountErrorsWithCodes;

@SuppressWarnings("unused")
@ParametersAreNonnullByDefault
@Component
public class SetBidsDelegate
        extends OperationOnListDelegate<SetRequest, SetResponse, SetBidItem, SetBidItem> {
    private static final int FIELD_DOES_NOT_MATCH_STRATEGY = BidsDefectTypes.fieldDoesNotMatchStrategy().getCode();

    private final SetBidsRequestValidationService setBidsRequestValidationService;
    private final BidsHelperConverter bidsHelperConverter;
    private final BidService bidService;
    private final BidsInternalValidationService bidsInternalValidationService;
    private final ResultConverter resultConverter;

    @Autowired
    public SetBidsDelegate(
            SetBidsRequestValidationService setBidsRequestValidationService,
            BidsHelperConverter bidsHelperConverter,
            ApiAuthenticationSource auth,
            BidService bidService,
            BidsInternalValidationService bidsInternalValidationService,
            ResultConverter resultConverter,
            PpcPropertiesSupport ppcPropertiesSupport,
            FeatureService featureService) {
        super(ApiPathConverter.forSetBids(), auth, ppcPropertiesSupport, featureService);
        this.setBidsRequestValidationService = setBidsRequestValidationService;
        this.bidsHelperConverter = bidsHelperConverter;
        this.bidService = bidService;
        this.bidsInternalValidationService = bidsInternalValidationService;
        this.resultConverter = resultConverter;
    }

    @Nullable
    @Override
    public ValidationResult<SetRequest, DefectType> validateRequest(SetRequest externalRequest) {
        return setBidsRequestValidationService.validate(externalRequest);
    }

    @Override
    public List<SetBidItem> convertRequest(SetRequest externalRequest) {
        return bidsHelperConverter.convertSetBidsRequest(externalRequest);
    }

    @Nonnull
    @Override
    public ValidationResult<List<SetBidItem>, DefectType> validateInternalRequest(List<SetBidItem> internalRequest) {
        Long operatorUid = auth.getOperator().getUid();
        ClientId clientId = auth.getChiefSubclient().getClientId();
        CampaignAccessibiltyChecker campaignAccessibiltyChecker = getCampaignAccessibiltyChecker();
        return bidsInternalValidationService.validateInternalRequest(internalRequest,
                operatorUid,
                clientId,
                campaignAccessibiltyChecker);
    }

    @Override
    public ApiMassResult<SetBidItem> processList(List<SetBidItem> validItems) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        long operatorUid = auth.getOperator().getUid();
        MassResult<SetBidItem> setBidMassResult = bidService.setBids(clientId, operatorUid, validItems);
        return resultConverter.toApiMassResult(setBidMassResult, BidsDefectPresentations.HOLDER);
    }

    @Override
    public SetResponse convertResponse(ApiResult<List<ApiResult<SetBidItem>>> result) {
        return bidsHelperConverter.convertSetBidsResponse(result, apiPathConverter);
    }

    /**
     * Не вычитаем баллы в случае, когда невозможно указать ставку из-за несоответствия стратегии.
     */
    @Override
    public OperationSummary correctOperationSummary(List<SetBidItem> internalRequest,
                                                    ApiResult<List<ApiResult<SetBidItem>>> apiResult) {
        return notCountErrorsWithCodes(apiResult, Set.of(FIELD_DOES_NOT_MATCH_STRATEGY));
    }
}
