package ru.yandex.direct.api.v5.entity.campaigns;

import java.util.Collections;

import com.yandex.direct.api.v5.campaigns.AddRequest;
import com.yandex.direct.api.v5.campaigns.AddResponse;
import com.yandex.direct.api.v5.campaigns.ArchiveRequest;
import com.yandex.direct.api.v5.campaigns.ArchiveResponse;
import com.yandex.direct.api.v5.campaigns.CampaignsPort;
import com.yandex.direct.api.v5.campaigns.DeleteRequest;
import com.yandex.direct.api.v5.campaigns.DeleteResponse;
import com.yandex.direct.api.v5.campaigns.GetRequest;
import com.yandex.direct.api.v5.campaigns.GetResponse;
import com.yandex.direct.api.v5.campaigns.ResumeRequest;
import com.yandex.direct.api.v5.campaigns.ResumeResponse;
import com.yandex.direct.api.v5.campaigns.SuspendRequest;
import com.yandex.direct.api.v5.campaigns.SuspendResponse;
import com.yandex.direct.api.v5.campaigns.UnarchiveRequest;
import com.yandex.direct.api.v5.campaigns.UnarchiveResponse;
import com.yandex.direct.api.v5.campaigns.UpdateRequest;
import com.yandex.direct.api.v5.campaigns.UpdateResponse;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolderRespectingPath;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolderRespectingPathImpl;
import ru.yandex.direct.api.v5.entity.GenericApiService;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.AddCampaignsDelegate;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.ArchiveCampaignsDelegate;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.GetCampaignsDelegate;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.ResumeCampaignsDelegate;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.SuspendCampaignsDelegate;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.UnarchiveCampaignsDelegate;
import ru.yandex.direct.api.v5.entity.campaigns.delegate.UpdateCampaignsDelegate;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.api.v5.ws.annotation.ApiMethod;
import ru.yandex.direct.api.v5.ws.annotation.ApiRequest;
import ru.yandex.direct.api.v5.ws.annotation.ApiResponse;
import ru.yandex.direct.api.v5.ws.annotation.ApiServiceEndpoint;
import ru.yandex.direct.api.v5.ws.annotation.ApiServiceType;
import ru.yandex.direct.api.v5.ws.annotation.ServiceType;
import ru.yandex.direct.core.entity.bids.validation.BidsDefects;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectTranslations;
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefectIds;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.validation.defect.ids.CollectionDefectIds;
import ru.yandex.direct.validation.defect.ids.DateDefectIds;
import ru.yandex.direct.validation.defect.ids.NumberDefectIds;
import ru.yandex.direct.validation.result.DefectIds;

import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.avgBidAndBidTogetherAreProhibited;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsCpmPriceIsNotGreaterThanMin;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsCpmPriceIsNotSmallerThanMax;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsIncorrectSetOfMobileGoals;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsStrategyEndDateMustBeLessThanOrEqualToCampaignEndDate;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsStrategyPeriodDaysCountGreaterThanMax;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsStrategyPeriodDaysCountLessThanMin;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsStrategyStartDateMustBeGreaterThanOrEqualToCampaignStartDate;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.badParamsToOptimizeForMeaningfulGoalAtLeastOneGoalExceptEngagedSessionIsRequired;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.campaignTranslations;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.commonTranslations;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.dayBudgetCanBeUsedWithManualStrategiesOnly;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.inconsistentStateEndDateMustBeGreaterThanOrEqualToStartDate;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.requireServicingWillBeIgnored;
import static ru.yandex.direct.api.v5.validation.DefectTypes.badParams;
import static ru.yandex.direct.api.v5.validation.DefectTypes.invalidValue;
import static ru.yandex.direct.api.v5.validation.DefectTypes.notFound;
import static ru.yandex.direct.api.v5.validation.DefectTypes.restricted;
import static ru.yandex.direct.api.v5.validation.DefectTypes.translations;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Сервис по работе с кампаниями
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/campaigns/campaigns-docpage/
 */
@ApiServiceEndpoint
@ApiServiceType(type = ServiceType.CLIENT)
public class CampaignsEndpoint implements CampaignsPort {

    private static final String SERVICE_NAME = "campaigns";

    private static final DefectPresentationsHolder DEFECT_PRESENTATIONS = DefectPresentationsHolder.builder()
            .register(CampaignDefectIds.Subset.MUST_NOT_CONTAIN_DUPLICATED_STRINGS,
                    t -> DefectTypes.duplicatedElements(t.getSubset()))
            .register(CollectionDefectIds.Gen.MUST_NOT_CONTAIN_DUPLICATED_ELEMENTS,
                    t -> DefectTypes.duplicatedElements(Collections.emptyList()))
            .register(StrategyDefectIds.Gen.AVG_BID_AND_BID_TOGETHER_PROHIBITED, t ->
                    avgBidAndBidTogetherAreProhibited())
            .register(CampaignDefectIds.Gen.DAY_BUDGET_NOT_SUPPORTED_WITH_STRATEGY, t ->
                    dayBudgetCanBeUsedWithManualStrategiesOnly())
            .register(CampaignDefectIds.Gen.REQUIRE_SERVICING_WILL_BE_IGNORED, t ->
                    requireServicingWillBeIgnored())

            // ExcludedSitesInternalPagesTest
            // .add_ExcludedSitesFieldContainsValidAndInvalidInternalPages_ErrorsAndWarningsReturned ожидает 5004
            // ядро возвращает
            // DefectInfo{path=[0].disabledDomains[2], defect=Defect{defectId=INVALID_DOMAIN_OR_SSP, params=lego},
            // value=lego}

            // ExcludedSitesValidationNegativeTest ожидает 5006
            // ядро возвращает
            // DefectInfo{path=[0].disabledDomains[0], defect=Defect{defectId=INVALID_DOMAIN_OR_SSP, params=spb.ru},
            // value=spb.ru}

            // Вышло так, что на один и тот же путь, на один и тот же дефект, ожидается разная конвертация. Выбрали
            // 5006, а во втором тесте ошибки будут не совпадает между перл и java
            .register(CampaignDefectIds.Places.INVALID_DOMAIN_OR_SSP,
                    t -> DefectTypes.invalidUseOfField())

            .register(CampaignDefectIds.IpAddresses.INVALID_IP_FORMAT,
                    t -> DefectTypes.invalidFormat())
            .register(CampaignDefectIds.IpAddresses.IP_CANT_BE_FROM_PRIVATE_NETWORK,
                    t -> DefectTypes.invalidUseOfField()
                            .withDetailedMessage((path, value) ->
                                    translations().ipCantBeFromPrivateNetwork(path.getNodes().get(0).toString(),
                                            value)))

            // делаем так чтобы свести коды ошибок для этого запроса
            // https://st.yandex-team.ru/DIRECT-152061#61289b20252a4c0bac88e6f3
            .register(NumberDefectIds.MUST_BE_GREATER_THAN_OR_EQUAL_TO_MIN, t ->
                    DefectTypes.badParamsNotLessThan(t.getMin()))
            // делаем так чтобы свести коды ошибок для этого запроса
            // https://st.yandex-team.ru/DIRECT-155663#61773bb46fa1cc4d8baac79b
            .register(NumberDefectIds.MUST_BE_LESS_THEN_OR_EQUAL_TO_MAX, t ->
                    DefectTypes.badParamsNotGreaterThan(t.getMax()))

            .register(DateDefectIds.END_DATE_MUST_BE_GREATER_THAN_OR_EQUAL_TO_START_DATE, t ->
                    inconsistentStateEndDateMustBeGreaterThanOrEqualToStartDate())

            // для этого запроса
            // https://st.yandex-team.ru/DIRECT-155663#61773eac67f373742a071c72
            .register(StrategyDefectIds.Gen.WEEK_BUDGET_LESS_THAN, t -> badParams().withDetailedMessage(
                    (path, value) -> CampaignDefectTranslations.INSTANCE.weekBudgetLessThanDetailed(path)))

            .register(StrategyDefectIds.Gen.STRATEGY_START_DATE_IS_BEFORE_CAMPAIGN_START_DATE,
                    t -> badParamsStrategyStartDateMustBeGreaterThanOrEqualToCampaignStartDate())

            .register(StrategyDefectIds.Gen.STRATEGY_END_DATE_IS_AFTER_CAMPAIGN_END_DATE,
                    t -> badParamsStrategyEndDateMustBeLessThanOrEqualToCampaignEndDate())

            .register(StrategyDefectIds.Num.STRATEGY_PERIOD_DAYS_COUNT_LESS_THAN_MIN,
                    t -> badParamsStrategyPeriodDaysCountLessThanMin(t.intValue()))

            .register(StrategyDefectIds.Num.STRATEGY_PERIOD_DAYS_COUNT_GREATER_THAN_MAX,
                    t -> badParamsStrategyPeriodDaysCountGreaterThanMax(t.intValue()))

            .register(BidsDefects.CurrencyAmountDefects.CPM_PRICE_IS_NOT_GREATER_THAN_MIN,
                    t -> badParamsCpmPriceIsNotGreaterThanMin(t.getMoneyPriceValue()))

            // ru.yandex.autotests.direct.api.campaigns.add.cpmbannercampaign.strategy.cp
            // .StrategiesCustomPeriodSpendLimitTest
            // "Значение больше максимального"
            .register(BidsDefects.CurrencyAmountDefects.CPM_PRICE_IS_NOT_SMALLER_THAN_MAX,
                    t -> badParamsCpmPriceIsNotSmallerThanMax(t.getMoneyPriceValue()))

            .register(StrategyDefectIds.Gen.INCORRECT_SET_OF_MOBILE_GOALS,
                    t -> badParamsIncorrectSetOfMobileGoals())

            .register(CollectionDefectIds.Gen.MUST_BE_IN_COLLECTION, t -> notFound())

            .register(StrategyDefectIds.Gen.UNABLE_TO_USE_CURRENT_MEANINGFUL_GOALS_FOR_OPTIMIZATION,
                    t -> badParamsToOptimizeForMeaningfulGoalAtLeastOneGoalExceptEngagedSessionIsRequired())

            // StrategyPayForConversionAccessErrorTest
            .register(StrategyDefectIds.Gen.PAY_FOR_CONVERSION_NOT_SUPPORTED, t -> restricted()
                    .withDetailedMessage(CampaignDefectTranslations.INSTANCE.payForConversionNotSupported()))

            // StrategyPayForConversionGoalIdNegativeTest
            .register(StrategyDefectIds.Gen.PAY_FOR_CONVERSION_DOES_NOT_ALLOW_ALL_GOALS,
                    t -> badParams().withDetailedMessage(
                            CampaignDefectTranslations.INSTANCE.payForConversionDoesNotAllowAllGoals()))

            // ExcludedSitesValidationNegativeTest, "Количество доменов, превышающее максимальную длину"
            // дефект прилетает на корень, т.к. относится сразу к 2ум элементам, поэтому не можем повесить на path
            .register(CollectionDefectIds.Size.SIZE_CANNOT_BE_MORE_THAN_MAX,
                    t -> DefectTypes.maxElementsExceeded(t.getMaxSize()))

            .register(CampaignDefectIds.Gen.DAY_BUDGET_OVERRIDEN_BY_WALLET,
                    t -> DefectTypes.warningNoEffect()
                            .withDetailedMessage(CampaignDefectTranslations.INSTANCE.walletDayBudget())
            )

            .register(CampaignDefectIds.Gen.DAY_BUDGET_SHOW_MODE_OVERRIDEN_BY_WALLET,
                    t -> DefectTypes.warningNoEffect()
                            .withDetailedMessage(CampaignDefectTranslations.INSTANCE.walletDayBudgetShowMode())
            )

            .register(CampaignDefectIds.Gen.VIDEO_TYPE_NOT_SUPPORTED_WITH_STRATEGY,
                    t -> CampaignDefectTypes.parameterInconsistentWithStrategy("VideoTarget")
            )

            .register(CampaignDefectIds.Nums.TOO_MANY_DAY_BUDGET_DAILY_CHANGES,
                    t -> DefectTypes.сantPerform()
                            .withDetailedMessage(CampaignDefectTranslations.INSTANCE.changeDayBudgetNotMoreThan3Times())
            )

            // StrategyAverageRoiFieldCombinationsNegativeTest
            .register(StrategyDefectIds.Gen.INCORRECT_RESERVE_RETURN, t -> badParams()
                    .withDetailedMessage(CampaignDefectTranslations.INSTANCE.incorrectReserveReturn()))

            .build();

    public static final DefectPresentationsHolderRespectingPath DEFECT_PRESENTATIONS_RESPECTING_PATH =
            DefectPresentationsHolderRespectingPathImpl.builderWithFallback(DEFECT_PRESENTATIONS)

                    .register(DefectIds.OBJECT_NOT_FOUND,
                            path(field("strategy"), field("strategyData"), field("goalId")),
                            badParams().withDetailedMessage(
                                    commonTranslations().goalNotFound()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("goalId")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().goalIdNotSet()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("sum")),
                            badParams().withDetailedMessage(
                                    commonTranslations().weeklyBudgetNotSpecified()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("roiCoef")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().enterRoiCoefValue()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("avgBid")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().averageCpcValueNotSpecified()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("avgCpa")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().averageCpaNotSpecified()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("avgCpi")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().averageCpiNotSpecified()))
                    .register(DefectIds.CANNOT_BE_NULL,
                            path(field("strategy"), field("strategyData"), field("reserveReturn")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().reserveReturnNotSpecified()))
                    .register(NumberDefectIds.MUST_BE_GREATER_THAN_MIN,
                            path(field("strategy"), field("strategyData"), field("roiCoef")),
                            t -> badParams().withDetailedMessage(
                                    campaignTranslations().roiCoefMustBeGreaterThanMin(t.getMin().intValue())))
                    .register(DateDefectIds.MUST_BE_GREATER_THAN_OR_EQUAL_TO_MIN,
                            path(field("strategy"), field("strategyData"), field("start")),
                            t -> badParams().withDetailedMessage(
                                    commonTranslations().dateCantBeEarlierThanNow(path(field("StartDate")))))
                    .register(DateDefectIds.MUST_BE_GREATER_THAN_OR_EQUAL_TO_MIN,
                            path(field("startDate")),
                            t -> invalidValue().withDetailedMessage(
                                    commonTranslations().dateCantBeEarlierThanNow(path(field("StartDate")))))
                    .register(DateDefectIds.MUST_BE_GREATER_THAN_OR_EQUAL_TO_MIN,
                            path(field("endDate")),
                            t -> invalidValue().withDetailedMessage(
                                    commonTranslations().dateCantBeEarlierThanNow(path(field("EndDate")))))
                    .register(DefectIds.OBJECT_NOT_FOUND,
                            path(field("broadMatch"), field("broadMatchGoalId")),
                            t -> notFound("OptimizeGoalId"))
                    .register(StrategyDefectIds.Gen.INCONSISTENT_STATE_STRATEGY_TYPE_AND_AD_GROUP_TYPES,
                            path(field("strategy"), field("strategyName")),
                            badParams().withDetailedMessage(
                                    campaignTranslations().averageCpvStrategyIsOnlyApplicableToCampaignsWithVideoGroups()))
                    .build();

    private final GenericApiService genericApiService;
    private final AddCampaignsDelegate addCampaignsDelegate;
    private final UpdateCampaignsDelegate updateCampaignsDelegate;
    private final SuspendCampaignsDelegate suspendCampaignsDelegate;
    private final ResumeCampaignsDelegate resumeCampaignsDelegate;
    private final GetCampaignsDelegate getCampaignsDelegate;
    private final ArchiveCampaignsDelegate archiveCampaignsDelegate;
    private final UnarchiveCampaignsDelegate unarchiveCampaignsDelegate;

    @Autowired
    public CampaignsEndpoint(
            GenericApiService genericApiService,
            AddCampaignsDelegate addCampaignsDelegate,
            UpdateCampaignsDelegate updateCampaignsDelegate,
            SuspendCampaignsDelegate suspendCampaignsDelegate,
            ResumeCampaignsDelegate resumeCampaignsDelegate,
            GetCampaignsDelegate getCampaignsDelegate,
            ArchiveCampaignsDelegate archiveCampaignsDelegate,
            UnarchiveCampaignsDelegate unarchiveCampaignsDelegate) {
        this.genericApiService = genericApiService;
        this.addCampaignsDelegate = addCampaignsDelegate;
        this.updateCampaignsDelegate = updateCampaignsDelegate;
        this.suspendCampaignsDelegate = suspendCampaignsDelegate;
        this.resumeCampaignsDelegate = resumeCampaignsDelegate;
        this.getCampaignsDelegate = getCampaignsDelegate;
        this.archiveCampaignsDelegate = archiveCampaignsDelegate;
        this.unarchiveCampaignsDelegate = unarchiveCampaignsDelegate;
    }

    @PreAuthorizeRead
    @ApiMethod(service = SERVICE_NAME, operation = "get")
    @ApiResponse
    @Override
    public GetResponse get(@ApiRequest GetRequest parameters) {
        return genericApiService.doAction(getCampaignsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "add")
    @ApiResponse
    @Override
    public AddResponse add(@ApiRequest AddRequest parameters) {
        return genericApiService.doAction(addCampaignsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "update")
    @ApiResponse
    @Override
    public UpdateResponse update(@ApiRequest UpdateRequest parameters) {
        return genericApiService.doAction(updateCampaignsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "delete")
    @ApiResponse
    @Override
    public DeleteResponse delete(@ApiRequest DeleteRequest parameters) {
        throw notImplementedYet();
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "archive")
    @ApiResponse
    @Override
    public ArchiveResponse archive(@ApiRequest ArchiveRequest parameters) {
        return genericApiService.doAction(archiveCampaignsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "unarchive")
    @ApiResponse
    @Override
    public UnarchiveResponse unarchive(@ApiRequest UnarchiveRequest parameters) {
        return genericApiService.doAction(unarchiveCampaignsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "suspend")
    @ApiResponse
    @Override
    public SuspendResponse suspend(@ApiRequest SuspendRequest parameters) {
        return genericApiService.doAction(suspendCampaignsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "resume")
    @ApiResponse
    @Override
    public ResumeResponse resume(@ApiRequest ResumeRequest parameters) {
        return genericApiService.doAction(resumeCampaignsDelegate, parameters);
    }

    private static RuntimeException notImplementedYet() {
        return new UnsupportedOperationException("Not implemented yet.");
    }
}
