package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.CampaignAssistant
import com.yandex.direct.api.v5.campaigns.CampaignFundsEnum
import com.yandex.direct.api.v5.campaigns.CampaignFundsParam
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSettingGet
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSettingsGetEnum
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSettingGet
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSettingsGetEnum
import com.yandex.direct.api.v5.campaigns.FundsParam
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSettingGet
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSettingsGetEnum
import com.yandex.direct.api.v5.campaigns.SharedAccountFundsParam
import com.yandex.direct.api.v5.campaigns.SmartCampaignSettingGet
import com.yandex.direct.api.v5.campaigns.SmartCampaignSettingsGetEnum
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingGet
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingsGetEnum
import ru.yandex.direct.api.v5.common.ConverterUtils.convertToMicros
import ru.yandex.direct.api.v5.common.toYesNoEnum
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import java.math.BigDecimal
import java.math.RoundingMode

fun createApiFundsParam(
    campaign: CampaignWithStrategy,
    ndsRatio: BigDecimal,
    sumForTransfer: BigDecimal?,
): FundsParam =
    FundsParam().apply {
        mode = campaign.extractFundsMode()

        if (mode == CampaignFundsEnum.SHARED_ACCOUNT_FUNDS) {
            sharedAccountFunds = createSharedAccountFundsParam(campaign, ndsRatio)
        }

        if (mode == CampaignFundsEnum.CAMPAIGN_FUNDS) {
            campaignFunds = createCampaignFundsParam(campaign, ndsRatio, sumForTransfer!!)
        }
    }

private fun CommonCampaign.extractFundsMode(): CampaignFundsEnum =
    when (walletId) {
        null, 0L -> CampaignFundsEnum.CAMPAIGN_FUNDS
        else -> CampaignFundsEnum.SHARED_ACCOUNT_FUNDS
    }

private fun createSharedAccountFundsParam(
    campaign: CampaignWithStrategy,
    ndsRatio: BigDecimal,
): SharedAccountFundsParam =
    SharedAccountFundsParam().apply {
        refund = (campaign.sum - campaign.sumSpent)
            .toApiMoney(ndsRatio)
            .coerceAtLeast(0)

        spend = campaign.sumSpent.toApiMoney(nds = BigDecimal.ZERO)
    }

private fun createCampaignFundsParam(
    campaign: CampaignWithStrategy,
    ndsRatio: BigDecimal,
    sumForTransfer: BigDecimal,
): CampaignFundsParam =
    CampaignFundsParam().apply {
        sum = campaign.sum.toApiMoney(nds = BigDecimal.ZERO)
        balance = (campaign.sum - campaign.sumSpent).toApiMoney(ndsRatio)
        balanceBonus = 0 // не используется, DIRECT-70057
        // для поля SumAvailableForTransfer НДС уже учтён в расчётах
        sumAvailableForTransfer = sumForTransfer.toApiMoney(nds = BigDecimal.ZERO)
    }

/**
 * Переводит деньги в формат API, c возможностью вычета НДС
 *
 * Аналог функции `API::Converter::ConvertSubs::convert_to_long` из перла:
 * https://a.yandex-team.ru/arc_vcs/direct/perl/api/lib/API/Converter/ConvertSubs.pm?rev=r5435964#L47
 */
private fun BigDecimal.toApiMoney(
    nds: BigDecimal,
): Long {
    val withSubtractedNds = this.divide(BigDecimal.ONE + nds, RoundingMode.FLOOR)
    return convertToMicros(withSubtractedNds)!!
}

fun createApiRepresentedBySettings(
    managerFio: String?,
    agencyName: String?,
): CampaignAssistant =
    CampaignAssistant().apply {
        manager = FACTORY.createCampaignAssistantManager(managerFio)
        agency = FACTORY.createCampaignAssistantAgency(agencyName)
    }

/**
 * Аналог функции API::Service::Campaigns::ConvertSubs::convert_settings_to_external из перла
 *
 * https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns/ConvertSubs.pm?rev=r8833935#L415
 */
fun createCpmBannerCampaignSettings(
    campaign: CpmBannerCampaign,
    clientUid: Long,
    advancedGeoTargeting: Boolean,
): List<CpmBannerCampaignSettingGet> = listOfNotNull(
    CpmBannerCampaignSettingsGetEnum.ADD_METRICA_TAG
        .enabledIf(campaign.hasAddMetrikaTagToUrl),
    CpmBannerCampaignSettingsGetEnum.ADD_OPENSTAT_TAG
        .enabledIf(campaign.hasAddOpenstatTagToUrl),
    CpmBannerCampaignSettingsGetEnum.ADD_TO_FAVORITES
        .enabledIf(clientUid in campaign.favoriteForUids.orEmpty()),
    CpmBannerCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED
        .enabledIf(true), // всегда включена
    CpmBannerCampaignSettingsGetEnum.ENABLE_AREA_OF_INTEREST_TARGETING
        .enabledIf(campaign.hasExtendedGeoTargeting),
    CpmBannerCampaignSettingsGetEnum.ENABLE_SITE_MONITORING
        .enabledIf(campaign.hasSiteMonitoring),
    CpmBannerCampaignSettingsGetEnum.REQUIRE_SERVICING
        .enabledIf(campaign.isServiced()),
    CpmBannerCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED
        .enabledIf((campaign.walletId ?: 0L) != 0L),
    CpmBannerCampaignSettingsGetEnum.ENABLE_CURRENT_AREA_TARGETING
        .enabledIf(campaign.isEnabledCurrentAreaTargeting())
        .takeIf { advancedGeoTargeting },
    CpmBannerCampaignSettingsGetEnum.ENABLE_REGULAR_AREA_TARGETING
        .enabledIf(campaign.isEnabledRegularAreaTargeting())
        .takeIf { advancedGeoTargeting }
)

private fun CpmBannerCampaignSettingsGetEnum.enabledIf(condition: Boolean?): CpmBannerCampaignSettingGet =
    CpmBannerCampaignSettingGet()
        .withOption(this)
        .withValue((condition ?: false).toYesNoEnum())

/**
 * Аналог функции API::Service::Campaigns::ConvertSubs::convert_settings_to_external из перла
 *
 * https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns/ConvertSubs.pm?rev=r8833935#L415
 */
fun createDynamicTextCampaignSettings(
    campaign: DynamicCampaign,
    clientUid: Long,
    advancedGeoTargeting: Boolean,
): List<DynamicTextCampaignSettingGet> = listOfNotNull(
    DynamicTextCampaignSettingsGetEnum.ADD_METRICA_TAG
        .enabledIf(campaign.hasAddMetrikaTagToUrl),
    DynamicTextCampaignSettingsGetEnum.ADD_OPENSTAT_TAG
        .enabledIf(campaign.hasAddOpenstatTagToUrl),
    DynamicTextCampaignSettingsGetEnum.ADD_TO_FAVORITES
        .enabledIf(clientUid in campaign.favoriteForUids.orEmpty()),
    DynamicTextCampaignSettingsGetEnum.CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED
        .enabledIf(campaign.isOrderPhraseLengthPrecedenceEnabled),
    DynamicTextCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED
        .enabledIf(true), // всегда включена
    DynamicTextCampaignSettingsGetEnum.ENABLE_AREA_OF_INTEREST_TARGETING
        .enabledIf(campaign.hasExtendedGeoTargeting),
    DynamicTextCampaignSettingsGetEnum.ENABLE_COMPANY_INFO
        .enabledIf(campaign.enableCompanyInfo),
    DynamicTextCampaignSettingsGetEnum.ENABLE_EXTENDED_AD_TITLE
        .enabledIf(campaign.hasTitleSubstitution),
    DynamicTextCampaignSettingsGetEnum.ENABLE_SITE_MONITORING
        .enabledIf(campaign.hasSiteMonitoring),
    DynamicTextCampaignSettingsGetEnum.REQUIRE_SERVICING
        .enabledIf(campaign.isServiced()),
    DynamicTextCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED
        .enabledIf((campaign.walletId ?: 0L) != 0L),
    DynamicTextCampaignSettingsGetEnum.ENABLE_CURRENT_AREA_TARGETING
        .enabledIf(campaign.isEnabledCurrentAreaTargeting())
        .takeIf { advancedGeoTargeting },
    DynamicTextCampaignSettingsGetEnum.ENABLE_REGULAR_AREA_TARGETING
        .enabledIf(campaign.isEnabledRegularAreaTargeting())
        .takeIf { advancedGeoTargeting }
)

private fun DynamicTextCampaignSettingsGetEnum.enabledIf(condition: Boolean?): DynamicTextCampaignSettingGet =
    DynamicTextCampaignSettingGet()
        .withOption(this)
        .withValue((condition ?: false).toYesNoEnum())

/**
 * Аналог функции API::Service::Campaigns::ConvertSubs::convert_settings_to_external из перла
 *
 * https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns/ConvertSubs.pm?rev=r8833935#L415
 */
fun createMobileAppCampaignSettings(
    campaign: MobileContentCampaign,
    clientUid: Long,
    advancedGeoTargeting: Boolean,
): List<MobileAppCampaignSettingGet> = listOfNotNull(
    MobileAppCampaignSettingsGetEnum.ADD_TO_FAVORITES
        .enabledIf(clientUid in campaign.favoriteForUids.orEmpty()),
    MobileAppCampaignSettingsGetEnum.CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED
        .enabledIf(campaign.isOrderPhraseLengthPrecedenceEnabled),
    MobileAppCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED
        .enabledIf(true), // всегда включена
    MobileAppCampaignSettingsGetEnum.ENABLE_AREA_OF_INTEREST_TARGETING
        .enabledIf(campaign.hasExtendedGeoTargeting),
    MobileAppCampaignSettingsGetEnum.MAINTAIN_NETWORK_CPC
        .enabledIf(campaign.enableCpcHold),
    MobileAppCampaignSettingsGetEnum.REQUIRE_SERVICING
        .enabledIf(campaign.isServiced()),
    MobileAppCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED
        .enabledIf((campaign.walletId ?: 0L) != 0L),
    MobileAppCampaignSettingsGetEnum.ENABLE_CURRENT_AREA_TARGETING
        .enabledIf(campaign.isEnabledCurrentAreaTargeting())
        .takeIf { advancedGeoTargeting },
    MobileAppCampaignSettingsGetEnum.ENABLE_REGULAR_AREA_TARGETING
        .enabledIf(campaign.isEnabledRegularAreaTargeting())
        .takeIf { advancedGeoTargeting }
)

private fun MobileAppCampaignSettingsGetEnum.enabledIf(condition: Boolean?): MobileAppCampaignSettingGet =
    MobileAppCampaignSettingGet()
        .withOption(this)
        .withValue((condition ?: false).toYesNoEnum())

/**
 * Аналог функции API::Service::Campaigns::ConvertSubs::convert_settings_to_external из перла
 *
 * https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns/ConvertSubs.pm?rev=r8833935#L415
 */
fun createSmartCampaignSettings(
    campaign: SmartCampaign,
    clientUid: Long,
    advancedGeoTargeting: Boolean,
): List<SmartCampaignSettingGet> = listOfNotNull(
    SmartCampaignSettingsGetEnum.ADD_TO_FAVORITES
        .enabledIf(clientUid in campaign.favoriteForUids.orEmpty()),
    SmartCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED
        .enabledIf(true), // всегда включена
    SmartCampaignSettingsGetEnum.ENABLE_AREA_OF_INTEREST_TARGETING
        .enabledIf(campaign.hasExtendedGeoTargeting),
    SmartCampaignSettingsGetEnum.REQUIRE_SERVICING
        .enabledIf(campaign.isServiced()),
    SmartCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED
        .enabledIf((campaign.walletId ?: 0L) != 0L),
    SmartCampaignSettingsGetEnum.ENABLE_CURRENT_AREA_TARGETING
        .enabledIf(campaign.isEnabledCurrentAreaTargeting())
        .takeIf { advancedGeoTargeting },
    SmartCampaignSettingsGetEnum.ENABLE_REGULAR_AREA_TARGETING
        .enabledIf(campaign.isEnabledRegularAreaTargeting())
        .takeIf { advancedGeoTargeting }
)

private fun SmartCampaignSettingsGetEnum.enabledIf(condition: Boolean?): SmartCampaignSettingGet =
    SmartCampaignSettingGet()
        .withOption(this)
        .withValue((condition ?: false).toYesNoEnum())

/**
 * Аналог функции API::Service::Campaigns::ConvertSubs::convert_settings_to_external из перла
 *
 * https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns/ConvertSubs.pm?rev=r8833935#L415
 */
fun createTextCampaignSettings(
    campaign: TextCampaign,
    clientUid: Long,
    advancedGeoTargeting: Boolean,
): List<TextCampaignSettingGet> = listOfNotNull(
    TextCampaignSettingsGetEnum.ADD_METRICA_TAG
        .enabledIf(campaign.hasAddMetrikaTagToUrl),
    TextCampaignSettingsGetEnum.ADD_OPENSTAT_TAG
        .enabledIf(campaign.hasAddOpenstatTagToUrl),
    TextCampaignSettingsGetEnum.ADD_TO_FAVORITES
        .enabledIf(clientUid in campaign.favoriteForUids.orEmpty()),
    TextCampaignSettingsGetEnum.CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED
        .enabledIf(campaign.isOrderPhraseLengthPrecedenceEnabled),
    TextCampaignSettingsGetEnum.DAILY_BUDGET_ALLOWED
        .enabledIf(true), // всегда включена
    TextCampaignSettingsGetEnum.ENABLE_AREA_OF_INTEREST_TARGETING
        .enabledIf(campaign.hasExtendedGeoTargeting),
    TextCampaignSettingsGetEnum.ENABLE_COMPANY_INFO
        .enabledIf(campaign.enableCompanyInfo),
    TextCampaignSettingsGetEnum.ENABLE_EXTENDED_AD_TITLE
        .enabledIf(campaign.hasTitleSubstitution),
    TextCampaignSettingsGetEnum.ENABLE_SITE_MONITORING
        .enabledIf(campaign.hasSiteMonitoring),
    TextCampaignSettingsGetEnum.EXCLUDE_PAUSED_COMPETING_ADS
        .enabledIf(campaign.excludePausedCompetingAds),
    TextCampaignSettingsGetEnum.MAINTAIN_NETWORK_CPC
        .enabledIf(campaign.enableCpcHold),
    TextCampaignSettingsGetEnum.REQUIRE_SERVICING
        .enabledIf(campaign.isServiced()),
    TextCampaignSettingsGetEnum.SHARED_ACCOUNT_ENABLED
        .enabledIf((campaign.walletId ?: 0L) != 0L),
    TextCampaignSettingsGetEnum.ENABLE_CURRENT_AREA_TARGETING
        .enabledIf(campaign.isEnabledCurrentAreaTargeting())
        .takeIf { advancedGeoTargeting },
    TextCampaignSettingsGetEnum.ENABLE_REGULAR_AREA_TARGETING
        .enabledIf(campaign.isEnabledRegularAreaTargeting())
        .takeIf { advancedGeoTargeting }
)

private fun TextCampaignSettingsGetEnum.enabledIf(condition: Boolean?): TextCampaignSettingGet =
    TextCampaignSettingGet()
        .withOption(this)
        .withValue((condition ?: false).toYesNoEnum())

/**
 * Аналог условия из Campaign::mass_get_servicing
 *
 * https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9245901#L7010
 */
private fun CommonCampaign.isServiced(): Boolean =
    isServiceRequested == true || (managerUid ?: 0) > 0

private fun CommonCampaign.isEnabledCurrentAreaTargeting(): Boolean =
    useRegularRegion != true || useCurrentRegion == true

private fun CommonCampaign.isEnabledRegularAreaTargeting(): Boolean =
    useCurrentRegion != true && hasExtendedGeoTargeting == true || useRegularRegion == true
