package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.CampaignTypeEnum
import com.yandex.direct.api.v5.campaigns.CampaignUpdateItem
import com.yandex.direct.api.v5.campaigns.UpdateRequest
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.campaignNotFound
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.campaignTypeNotSupported
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsContainer
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsConvertedRequest
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsUpdateRequestValidator
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsUpdateRequestValidator.SimpleValidations.allowedCampaignSource
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsUpdateRequestValidator.SimpleValidations.allowedCampaignType
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsUpdateRequestValidator.SimpleValidations.campaignTypeNotMatchesWithTypeInRequest
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsUpdateRequestValidator.SimpleValidations.validCampaignType
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsUpdateRequestValidator.SimpleValidations.validateCampaignUpdateItemCommonDefects
import ru.yandex.direct.api.v5.entity.campaigns.validation.CpmBannerCampaignUpdateRequestValidator
import ru.yandex.direct.api.v5.entity.campaigns.validation.DynamicTextCampaignUpdateRequestValidator
import ru.yandex.direct.api.v5.entity.campaigns.validation.MobileAppCampaignUpdateRequestValidator
import ru.yandex.direct.api.v5.entity.campaigns.validation.SmartCampaignUpdateRequestValidator
import ru.yandex.direct.api.v5.entity.campaigns.validation.TextCampaignUpdateRequestValidator
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.API5_EDIT_CAMPAIGNS
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.API5_VISIBLE
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPlacementTypes
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSource
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTimeTargeting
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignClientIdFilter
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIdsFilter
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIsNotEmpty
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignTypesExceptFilter
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.multipleConditionFilter
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult

interface CampaignUpdateItemConverter<out C : BaseCampaign> {
    fun convert(
        item: CampaignUpdateItem,
        uidAndClientId: UidAndClientId,
        timeZones: Map<String, GeoTimezone>,
        knownSsps: Set<String>,
        campaignFromDb: BaseCampaign?,
    ): UpdateCampaignsConvertedRequest<C>
}

@Component
class CampaignsUpdateRequestConverter @Autowired constructor(
    private val geoTimezoneRepository: GeoTimezoneRepository,
    private val shardHelper: ShardHelper,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val sspPlatformsRepository: SspPlatformsRepository,
    private val cpmBannerCampaignUpdateItemConverter: CpmBannerCampaignUpdateItemConverter,
    private val dynamicTextCampaignUpdateItemConverter: DynamicTextCampaignUpdateItemConverter,
    private val mobileAppCampaignUpdateItemConverter: MobileAppCampaignUpdateItemConverter,
    private val smartCampaignUpdateItemConverter: SmartCampaignUpdateItemConverter,
    private val textCampaignUpdateItemConverter: TextCampaignUpdateItemConverter,
) {
    fun convertRequest(uidAndClientId: UidAndClientId, request: UpdateRequest): List<UpdateCampaignsContainer<*>> {
        val campaigns = request.campaigns
        val requestCampaignIds = campaigns.map { it.id }
        val timeZones = getTimeZones(campaigns)
        val knownSsp = sspPlatformsRepository.allSspPlatforms.toSet()

        val itemsWithStrategy = campaigns.filter(this::containsStrategy)
        val itemsWithPlacementsTypes = campaigns.filter(this::containsPlacementTypes)
        val itemsWithTimeTargeting = campaigns.filter(this::containsTimeTargeting)
        val itemsWithSmsSettings = campaigns.filter(this::containsSmsSettings)
        val itemsWithRelevantKeywords = campaigns.filter(this::containsRelevantKeywords)
        val itemsWithSettings = campaigns.filter(this::containsSettings)

        val shard = shardHelper.getShardByClientIdStrictly(uidAndClientId.clientId)

        val typesToLoad = buildList {
            add(CampaignWithCampaignType::class.java)
            add(CampaignWithSource::class.java)
            if (itemsWithStrategy.isNotEmpty() || itemsWithSettings.isNotEmpty())
                add(CampaignWithStrategy::class.java)
            if (itemsWithPlacementsTypes.isNotEmpty())
                add(CampaignWithPlacementTypes::class.java)
            if (itemsWithTimeTargeting.isNotEmpty())
                add(CampaignWithTimeTargeting::class.java)
            if (itemsWithSmsSettings.isNotEmpty())
                add(CommonCampaign::class.java)
            if (itemsWithRelevantKeywords.isNotEmpty())
                add(CampaignWithBroadMatch::class.java)
        }

        val campaignsFromDbList = campaignTypedRepository.getSafely(
            shard,
            multipleConditionFilter(
                campaignIdsFilter(requestCampaignIds),
                campaignClientIdFilter(uidAndClientId.clientId),
                campaignTypesExceptFilter(CAMPAIGN_TYPES_BLACKLIST),
                campaignIsNotEmpty()
            ), typesToLoad
        )
        val campaignsFromDb = campaignsFromDbList.associateBy { it.id }

        val typesFromDb = campaignsFromDb.mapValues { (it.value as CampaignWithCampaignType).type }

        return campaigns.map { externalItem ->
            val campaignFromDb = campaignsFromDb[externalItem.id]
            val type = typesFromDb[externalItem.id]
            val typeFromStructure = inferTypeFromStructure(externalItem)

            val validationResult = validateItem(
                externalItem,
                campaignFromDb,
                timeZones,
                type,
                typeFromStructure,
                requestCampaignIds
            )
            val errors = validationResult.flattenErrors().map { it.defect }

            val convertedItem = if (errors.isEmpty()) {
                val converter = getConverterByType(type!!)
                converter.convert(
                    externalItem, uidAndClientId, timeZones, knownSsp, campaignFromDb)
            } else {
                null
            }

            UpdateCampaignsContainer(
                externalItem,
                convertedItem,
                errors,
                warnings = validationResult.flattenWarnings().map { it.defect }
            )
        }
    }

    private fun inferTypeFromStructure(item: CampaignUpdateItem): CampaignType? =
        when {
            item.cpmBannerCampaign != null -> CampaignType.CPM_BANNER
            item.dynamicTextCampaign != null -> CampaignType.DYNAMIC
            item.mobileAppCampaign != null -> CampaignType.MOBILE_CONTENT
            item.smartCampaign != null -> CampaignType.PERFORMANCE
            item.textCampaign != null -> CampaignType.TEXT
            else -> null
        }

    private fun containsStrategy(item: CampaignUpdateItem): Boolean =
        item.cpmBannerCampaign?.biddingStrategy != null
            || item.dynamicTextCampaign?.biddingStrategy != null
            || item.mobileAppCampaign?.biddingStrategy != null
            || item.smartCampaign?.biddingStrategy != null
            || item.textCampaign?.biddingStrategy != null

    private fun containsPlacementTypes(item: CampaignUpdateItem): Boolean =
        // в smart и ТГО placementTypes пока не используются, всегда заполняются пустым значением
        item.dynamicTextCampaign?.placementTypes != null

    private fun containsTimeTargeting(item: CampaignUpdateItem): Boolean =
        item.timeTargeting != null

    private fun containsSmsSettings(item: CampaignUpdateItem): Boolean =
        item.notification?.smsSettings != null

    private fun containsRelevantKeywords(item: CampaignUpdateItem): Boolean =
        item.textCampaign?.relevantKeywords != null

    private fun containsSettings(item: CampaignUpdateItem): Boolean =
        item.cpmBannerCampaign?.settings != null
            || item.dynamicTextCampaign?.settings != null
            || item.mobileAppCampaign?.settings != null
            || item.smartCampaign?.settings != null
            || item.textCampaign?.settings != null

    private fun getTimeZones(campaigns: List<CampaignUpdateItem>): Map<String, GeoTimezone> {
        val timeZones = campaigns.mapNotNull { it.timeZone }
        return geoTimezoneRepository.getByTimeZones(timeZones)
    }

    private fun validateItem(
        item: CampaignUpdateItem,
        campaignFromDb: BaseCampaign?,
        timeZones: Map<String, GeoTimezone>,
        type: CampaignType?,
        typeFromStructure: CampaignType?,
        requestCampaignIds: List<Long>
    ): ValidationResult<CampaignUpdateItem, DefectType> {
        val vb = ItemValidationBuilder.of<CampaignUpdateItem, DefectType>(item)
        validateCampaignUpdateItemFatalDefects(vb, campaignFromDb, type, typeFromStructure)
        if (vb.result.hasAnyErrors()) {
            return vb.result
        }

        vb.checkBy { validateCampaignUpdateItemCommonDefects(it, timeZones, type, requestCampaignIds) }

        vb.item(item.cpmBannerCampaign, "CpmBannerCampaign")
            .checkBy(
                {
                    CpmBannerCampaignUpdateRequestValidator.validateCpmBannerCampaign(
                        it,
                        campaignFromDb as CpmBannerCampaign?
                    )
                }, When.notNull()
            )
        vb.item(item.dynamicTextCampaign, "DynamicTextCampaign")
            .checkBy(
                {
                    DynamicTextCampaignUpdateRequestValidator.validateDynamicTextCampaign(
                        it,
                        campaignFromDb as DynamicCampaign?)
                },
                When.notNull()
            )
        vb.item(item.mobileAppCampaign, "MobileAppCampaign")
            .checkBy(
                {
                    MobileAppCampaignUpdateRequestValidator.validateMobileAppCampaign(
                        it,
                        campaignFromDb as MobileContentCampaign?
                    )
                },
                When.notNull()
            )
        vb.item(item.textCampaign, "TextCampaign")
            .checkBy(
                {
                    TextCampaignUpdateRequestValidator.validateTextCampaign(
                        it,
                        campaignFromDb as TextCampaign?
                    )
                },
                When.notNull()
            )
        vb.item(item.smartCampaign, "SmartCampaign")
            .checkBy(
                {
                    SmartCampaignUpdateRequestValidator.validateSmartCampaign(
                        it,
                        campaignFromDb as SmartCampaign?
                    )
                },
                When.notNull()
            )

        return vb.result
    }

    /**
     * Проверяет на дефекты, которые не позволяют начать валидацию CampaignUpdateItem
     */
    private fun validateCampaignUpdateItemFatalDefects(
        vb: ItemValidationBuilder<CampaignUpdateItem, DefectType>,
        campaignFromDb: BaseCampaign?,
        type: CampaignType?,
        typeFromStructure: CampaignType?
    ) = vb.apply {
            check(CampaignsUpdateRequestValidator::campaignIdShouldBePositive)
            check(allowedCampaignType(campaignFromDb as CampaignWithCampaignType?, API5_VISIBLE, campaignNotFound()),
                When.isValid())
            check(allowedCampaignType(campaignFromDb, API5_EDIT_CAMPAIGNS, campaignTypeNotSupported()), When.isValid())
            check(allowedCampaignSource(campaignFromDb as CampaignWithSource?), When.isValid())
            check(validCampaignType(), When.isValid())
            check({ campaignTypeNotMatchesWithTypeInRequest(type, typeFromStructure) }, When.isValid())
        }

    private fun getConverterByType(type: CampaignType): CampaignUpdateItemConverter<*> =
        when (type) {
            CampaignType.CPM_BANNER -> cpmBannerCampaignUpdateItemConverter
            CampaignType.DYNAMIC -> dynamicTextCampaignUpdateItemConverter
            CampaignType.MOBILE_CONTENT -> mobileAppCampaignUpdateItemConverter
            CampaignType.PERFORMANCE -> smartCampaignUpdateItemConverter
            CampaignType.TEXT -> textCampaignUpdateItemConverter
            // ветка "else" никогда не должна вызываться,
            // так как неправильные типы кампаний отсекаются на этапе валидации
            else -> error("Invalid campaign type $type for campaigns.update")
        }

    companion object {
        private val INTERNAL_TO_EXTERNAL_TYPES = mapOf(
            CampaignType.TEXT to CampaignTypeEnum.TEXT_CAMPAIGN,
            CampaignType.MOBILE_CONTENT to CampaignTypeEnum.MOBILE_APP_CAMPAIGN,
            CampaignType.DYNAMIC to CampaignTypeEnum.DYNAMIC_TEXT_CAMPAIGN,
            CampaignType.CPM_BANNER to CampaignTypeEnum.CPM_BANNER_CAMPAIGN,
            CampaignType.PERFORMANCE to CampaignTypeEnum.SMART_CAMPAIGN,
        )

        fun convertCampaignTypeToExternal(type: CampaignType): CampaignTypeEnum? = INTERNAL_TO_EXTERNAL_TYPES[type]

        val CAMPAIGN_TYPES_BLACKLIST: Set<CampaignType> = setOf(
            CampaignType.MCB,   // старый баян, закрыт как продукт
        )
    }
}

@Suppress("UNCHECKED_CAST")
private fun <K, V : Any> Map<K, V?>.filterValuesNotNull(): Map<K, V> =
    filterValues { it != null } as Map<K, V>
