package ru.yandex.direct.api.v5.entity.campaigns.converter;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategy;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignStrategy;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignStrategyAdd;

import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;

import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromCpAverageCpv;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromCpDecreasedPriceForRepeatedImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromCpMaximumImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromManualCpm;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbAverageCpv;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbDecreasedPriceForRepeatedImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbMaximumImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.toCampaignsAutobudget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromCpAverageCpv;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromCpDecreasedPriceForRepeatedImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromCpMaximumImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbAverageCpv;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbDecreasedPriceForRepeatedImpressions;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbMaximumImpressions;

@ParametersAreNonnullByDefault
public class CpmBannerCampaignStrategyConverter {

    // Медийных кампаний нет на поиске, поэтому  на поиске нет

    private static final Map<CpmBannerCampaignNetworkStrategyTypeEnum, StrategyName> STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE =
            ImmutableMap.<CpmBannerCampaignNetworkStrategyTypeEnum, StrategyName>builder()
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.MANUAL_CPM, StrategyName.CPM_DEFAULT)
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_REACH)
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD)
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS)
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.CP_MAXIMUM_IMPRESSIONS,
                            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD)
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.WB_AVERAGE_CPV,
                            StrategyName.AUTOBUDGET_AVG_CPV)
                    .put(CpmBannerCampaignNetworkStrategyTypeEnum.CP_AVERAGE_CPV,
                            StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD)
                    .build();

    private CpmBannerCampaignStrategyConverter() {
    }

    public static DbStrategy toCampaignStrategy(CpmBannerCampaignStrategy apiStrategy) {
        var dbStrategy = new DbStrategy();

        CpmBannerCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        CpmBannerCampaignNetworkStrategyTypeEnum networkStrategyType =
                apiStrategy.getNetwork().getBiddingStrategyType();

        if (STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.containsKey(networkStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.get(networkStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromNetworkStrategyData(networkStrategyType, apiStrategy.getNetwork()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType));
        dbStrategy.withStrategy(CampOptionsStrategy.DIFFERENT_PLACES);
        return dbStrategy;
    }

    public static DbStrategy toCampaignStrategy(CpmBannerCampaignStrategyAdd apiStrategy) {
        var dbStrategy = new DbStrategy();

        CpmBannerCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        CpmBannerCampaignNetworkStrategyTypeEnum networkStrategyType =
                apiStrategy.getNetwork().getBiddingStrategyType();

        if (STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.containsKey(networkStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.get(networkStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromNetworkStrategyData(networkStrategyType, apiStrategy.getNetwork()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType));
        dbStrategy.withStrategy(CampOptionsStrategy.DIFFERENT_PLACES);
        return dbStrategy;
    }

    /**
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/Model/Campaign.pm?rev=7464032#L268
     */
    private static CampaignsPlatform toCampaignsPlatform(CpmBannerCampaignSearchStrategyTypeEnum searchStrategyType) {

        boolean isSearchStop = searchStrategyType == CpmBannerCampaignSearchStrategyTypeEnum.SERVING_OFF;
        // в сетях кампания всегда включена

        if (isSearchStop) {
            return CampaignsPlatform.CONTEXT;
        } else {
            return CampaignsPlatform.BOTH;
        }
    }

    @Nullable
    private static StrategyData fromNetworkStrategyData(CpmBannerCampaignNetworkStrategyTypeEnum networkStrategyType,
                                                        CpmBannerCampaignNetworkStrategyAdd strategy) {
        switch (networkStrategyType) {
            case MANUAL_CPM:
                return fromManualCpm();
            case WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS:
                return fromWbDecreasedPriceForRepeatedImpressions(strategy.getWbDecreasedPriceForRepeatedImpressions());
            case CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS:
                return fromCpDecreasedPriceForRepeatedImpressions(strategy.getCpDecreasedPriceForRepeatedImpressions());
            case WB_MAXIMUM_IMPRESSIONS:
                return fromWbMaximumImpressions(strategy.getWbMaximumImpressions());
            case CP_MAXIMUM_IMPRESSIONS:
                return fromCpMaximumImpressions(strategy.getCpMaximumImpressions());
            case WB_AVERAGE_CPV:
                return fromWbAverageCpv(strategy.getWbAverageCpv());
            case CP_AVERAGE_CPV:
                return fromCpAverageCpv(strategy.getCpAverageCpv());
            default:
                return null;
        }
    }

    @Nullable
    private static StrategyData fromNetworkStrategyData(CpmBannerCampaignNetworkStrategyTypeEnum networkStrategyType,
                                                        CpmBannerCampaignNetworkStrategy strategy) {
        switch (networkStrategyType) {
            case MANUAL_CPM:
                return fromManualCpm();
            case WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS:
                return fromWbDecreasedPriceForRepeatedImpressions(strategy.getWbDecreasedPriceForRepeatedImpressions());
            case CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS:
                return fromCpDecreasedPriceForRepeatedImpressions(strategy.getCpDecreasedPriceForRepeatedImpressions());
            case WB_MAXIMUM_IMPRESSIONS:
                return fromWbMaximumImpressions(strategy.getWbMaximumImpressions());
            case CP_MAXIMUM_IMPRESSIONS:
                return fromCpMaximumImpressions(strategy.getCpMaximumImpressions());
            case WB_AVERAGE_CPV:
                return fromWbAverageCpv(strategy.getWbAverageCpv());
            case CP_AVERAGE_CPV:
                return fromCpAverageCpv(strategy.getCpAverageCpv());
            default:
                return null;
        }
    }

}
