package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.CampaignUpdateItem
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSetting
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignStrategy
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignUpdateItem
import com.yandex.direct.api.v5.general.YesNoEnum
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.buildModelChanges
import ru.yandex.direct.api.v5.common.processJaxbElement
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsConvertedRequest
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTimeTargeting
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.model.ModelChanges

@Component
class CpmBannerCampaignUpdateItemConverter @Autowired constructor(
    private val featureService: FeatureService
) : CampaignUpdateItemConverter<CpmBannerCampaign> {

    override fun convert(
        item: CampaignUpdateItem,
        uidAndClientId: UidAndClientId,
        timeZones: Map<String, GeoTimezone>,
        knownSsps: Set<String>,
        campaignFromDb: BaseCampaign?,
    ): UpdateCampaignsConvertedRequest<CpmBannerCampaign> {
        check(campaignFromDb == null || campaignFromDb is CpmBannerCampaign)
        val cpmBannerItem: CpmBannerCampaignUpdateItem? = item.cpmBannerCampaign
        val settings = cpmBannerItem?.settings.orEmpty()
            .associateBy(CpmBannerCampaignSetting::getOption) { it.value == YesNoEnum.YES }

        val modelChanges = buildModelChanges<CpmBannerCampaign>(item.id) {
            val oldTimeTarget = campaignFromDb?.let { (it as CampaignWithTimeTargeting).timeTarget }
            val oldCommonCampaign = campaignFromDb?.let { (it as CommonCampaign) }
            processCommonFields(item, timeZones, oldTimeTarget, oldCommonCampaign)
            processMetrikaCounters(cpmBannerItem?.counterIds)
            processEshowSettings(cpmBannerItem?.videoTarget)
            processDailyBudget(item)
            processExcludedSites(item, knownSsps)
            processCheckPositionInterval(item)
            processFrequencyCap(cpmBannerItem)

            val apiStrategy = cpmBannerItem?.biddingStrategy
            val newStrategy = apiStrategy?.let {
                updateStrategyWithOldValues(it, campaignFromDb as CpmBannerCampaign)
                CpmBannerCampaignStrategyConverter.toCampaignStrategy(it)
            }
            processStrategy(newStrategy)

            processHasAddOpenstatTagToUrl(settings[CpmBannerCampaignSettingsEnum.ADD_OPENSTAT_TAG])
            processHasAddMetrikaTagToUrl(settings[CpmBannerCampaignSettingsEnum.ADD_METRICA_TAG])
            processHasExtendedGeoTargeting(settings[CpmBannerCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING])

            val advancedGeoTargeting = featureService.isEnabledForClientId(
                uidAndClientId.clientId,
                FeatureName.ADVANCED_GEOTARGETING
            )

            if (advancedGeoTargeting) {
                processUseCurrentRegion(settings[CpmBannerCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING])
                processUseRegularRegion(settings[CpmBannerCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING])
            }
            processHasSiteMonitoring(settings[CpmBannerCampaignSettingsEnum.ENABLE_SITE_MONITORING])
            processFavoriteForIds(uidAndClientId.uid, settings[CpmBannerCampaignSettingsEnum.ADD_TO_FAVORITES])
        }

        return UpdateCampaignsConvertedRequest(
            modelChanges,
            requireServicing = settings[CpmBannerCampaignSettingsEnum.REQUIRE_SERVICING]
        )
    }

    private fun ModelChanges<CpmBannerCampaign>.processFrequencyCap(
        cpmBannerItem: CpmBannerCampaignUpdateItem?
    ) {
        processJaxbElement(
            cpmBannerItem?.frequencyCap,
            CpmBannerCampaign.IMPRESSION_RATE_COUNT
        ) { it?.impressions }
        processJaxbElement(
            cpmBannerItem?.frequencyCap,
            CpmBannerCampaign.IMPRESSION_RATE_INTERVAL_DAYS
        ) { it?.periodDays }
    }

    /**
     * Дозаполняет apiStrategy данными стратегии из oldCampaign, при условии, что biddingStrategyType одинаковый.
     */
    private fun updateStrategyWithOldValues(
        apiStrategy: CpmBannerCampaignStrategy,
        oldCampaign: CpmBannerCampaign
    ) {
        val oldStrategy = toCpmBannerCampaignExternalStrategy(oldCampaign.strategy)
        // CpmBannerCampaign бывает только в сетях, на поиске всегда SERVING_OFF
        if (apiStrategy.search == null) {
            apiStrategy.search = oldStrategy.search
        }

        if (apiStrategy.network == null) {
            apiStrategy.network = oldStrategy.network
        } else if (apiStrategy.network.biddingStrategyType == oldStrategy.network.biddingStrategyType) {
            copyProperties(apiStrategy.network, oldStrategy.network)
            apiStrategy.network = oldStrategy.network
        }
    }

    @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
    private fun copyProperties(
        source: CpmBannerCampaignNetworkStrategy,
        dest: CpmBannerCampaignNetworkStrategy
    ) {
        check(source.biddingStrategyType == dest.biddingStrategyType)
        when (source.biddingStrategyType) {
            CpmBannerCampaignNetworkStrategyTypeEnum.MANUAL_CPM -> {
                // CPM_DEFAULT, no properties
            }
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS -> {
                copyPropertiesExceptNull(
                    source.wbDecreasedPriceForRepeatedImpressions,
                    dest.wbDecreasedPriceForRepeatedImpressions
                )
            }
            CpmBannerCampaignNetworkStrategyTypeEnum.CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS ->
                copyPropertiesExceptNull(
                    source.cpDecreasedPriceForRepeatedImpressions,
                    dest.cpDecreasedPriceForRepeatedImpressions
                )
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_IMPRESSIONS ->
                copyPropertiesExceptNull(source.wbMaximumImpressions, dest.wbMaximumImpressions)
            CpmBannerCampaignNetworkStrategyTypeEnum.CP_MAXIMUM_IMPRESSIONS ->
                copyPropertiesExceptNull(source.cpMaximumImpressions, dest.cpMaximumImpressions)
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_AVERAGE_CPV ->
                copyPropertiesExceptNull(source.wbAverageCpv, dest.wbAverageCpv)
            CpmBannerCampaignNetworkStrategyTypeEnum.CP_AVERAGE_CPV ->
                copyPropertiesExceptNull(source.cpAverageCpv, dest.cpAverageCpv)
            CpmBannerCampaignNetworkStrategyTypeEnum.UNKNOWN -> {
                // not used
            }
        }
    }
}
