package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignStrategy
import org.slf4j.LoggerFactory
import ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyConverterConstants.DEFAULT_STRATEGY_NAME
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import java.lang.invoke.MethodHandles

private val logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass())

fun createDynamicTextCampaignGetStrategy(campaign: DynamicCampaign): DynamicTextCampaignStrategy =
    toDynamicTextCampaignExternalStrategy(campaign.strategy).apply {
        if (search.biddingStrategyType == null) {
            search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.UNKNOWN
            logger.warn("Campaign ${campaign.id} has unknown strategy ${campaign.strategy}")
        }

        if (network.biddingStrategyType == null) {
            network.biddingStrategyType = DynamicTextCampaignNetworkStrategyTypeEnum.UNKNOWN
            logger.warn("Campaign ${campaign.id} has unknown strategy ${campaign.strategy}")
        }
    }

fun toDynamicTextCampaignExternalStrategy(internalStrategy: DbStrategy) = DynamicTextCampaignStrategy().apply {
    search = getSearchStrategy(internalStrategy)
    network = getNetworkStrategy(internalStrategy)
}

private fun getSearchStrategy(internalStrategy: DbStrategy): DynamicTextCampaignSearchStrategy {
    // ДО бывает только на поиске
    val search = DynamicTextCampaignSearchStrategy()
    val strategyData = internalStrategy.strategyData
    when (strategyData.name) {
        DEFAULT_STRATEGY_NAME -> {
            search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION
        }
        StrategyName.AUTOBUDGET_AVG_CPA.name.lowercase() -> {
            if (strategyData.payForConversion == true) {
                search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION
                search.payForConversion = toStrategyPayForConversion(strategyData)
            } else {
                search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPA
                search.averageCpa = toStrategyAverageCpa(strategyData)
            }
        }
        StrategyName.AUTOBUDGET_AVG_CLICK.name.lowercase() -> {
            search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPC
            search.averageCpc = toStrategyAverageCpc(strategyData)
        }
        StrategyName.AUTOBUDGET_CRR.name.lowercase() -> {
            if (strategyData.payForConversion == true) {
                search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR
                search.payForConversionCrr = toStrategyPayForConversionCrr(strategyData)
            } else {
                search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CRR
                search.averageCrr = toStrategyAverageCrr(strategyData)
            }
        }
        StrategyName.AUTOBUDGET_ROI.name.lowercase() -> {
            search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_ROI
            search.averageRoi = toStrategyAverageRoi(strategyData)
        }
        StrategyName.AUTOBUDGET.name.lowercase() -> {
            when (strategyData.goalId) {
                null -> {
                    search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS
                    search.wbMaximumClicks = toStrategyMaximumClicks(strategyData)
                }
                else -> {
                    search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE
                    search.wbMaximumConversionRate = toStrategyMaximumConversionRate(strategyData)
                }
            }
        }
        StrategyName.AUTOBUDGET_WEEK_BUNDLE.name.lowercase() -> {
            search.biddingStrategyType = DynamicTextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
            search.weeklyClickPackage = toStrategyWeeklyClickPackage(strategyData)
        }
    }
    return search
}

private fun getNetworkStrategy(internalStrategy: DbStrategy): DynamicTextCampaignNetworkStrategy {
    // Несмотря на то, что в БД есть кампании с заданной стратегией в сетях,
    // в java-версии campaigns.get поддерживается только SERVING_OFF
    return DynamicTextCampaignNetworkStrategy().apply {
        biddingStrategyType = DynamicTextCampaignNetworkStrategyTypeEnum.SERVING_OFF
    }
}
