package ru.yandex.direct.api.v5.entity.campaigns.converter;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategy;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignStrategy;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignStrategyAdd;

import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;

import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.defaultStrategyData;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpa;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpc;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageRoi;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromPayForConversion;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromPayForConversionCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbMaximumClicks;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbMaximumConversionRate;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWeeklyClickPackage;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.toCampaignsAutobudget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpa;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpc;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageRoi;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromPayForConversion;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromPayForConversionCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbMaximumClicks;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbMaximumConversionRate;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWeeklyClickPackage;

/**
 * Таблица совместимости стратегий https://yandex.ru/dev/direct/doc/dg/objects/campaign-strategies.html
 */
@ParametersAreNonnullByDefault
public class DynamicTextCampaignStrategyConverter {

    /**
     * IMPRESSIONS_BELOW_SEARCH - deprecated
     */
    private static final Map<DynamicTextCampaignSearchStrategyTypeEnum, StrategyName>
            STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE =
            ImmutableMap.<DynamicTextCampaignSearchStrategyTypeEnum, StrategyName>builder()
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPA, StrategyName.AUTOBUDGET_AVG_CPA)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPC, StrategyName.AUTOBUDGET_AVG_CLICK)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CRR, StrategyName.AUTOBUDGET_CRR)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_ROI, StrategyName.AUTOBUDGET_ROI)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION, StrategyName.DEFAULT_)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION, StrategyName.AUTOBUDGET_AVG_CPA)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR, StrategyName.AUTOBUDGET_CRR)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS, StrategyName.AUTOBUDGET)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE, StrategyName.AUTOBUDGET)
                    .put(DynamicTextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
                            StrategyName.AUTOBUDGET_WEEK_BUNDLE)
                    .build();

    // ДО в сетях нет, поэтому и STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE нет

    private DynamicTextCampaignStrategyConverter() {
    }

    public static DbStrategy toCampaignStrategy(DynamicTextCampaignStrategy apiStrategy) {
        var dbStrategy = new DbStrategy();

        DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        DynamicTextCampaignNetworkStrategyTypeEnum networkStrategyType =
                apiStrategy.getNetwork().getBiddingStrategyType();

        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType, networkStrategyType));
        dbStrategy.withStrategy(toCampOptionsStrategy(networkStrategyType));
        return dbStrategy;
    }

    public static DbStrategy toCampaignStrategy(DynamicTextCampaignStrategyAdd apiStrategy) {
        var dbStrategy = new DbStrategy();

        DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        DynamicTextCampaignNetworkStrategyTypeEnum networkStrategyType =
                apiStrategy.getNetwork().getBiddingStrategyType();

        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType, networkStrategyType));
        dbStrategy.withStrategy(toCampOptionsStrategy(networkStrategyType));
        return dbStrategy;
    }

    @Nullable
    private static CampaignsPlatform toCampaignsPlatform(
            DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType,
            DynamicTextCampaignNetworkStrategyTypeEnum networkStrategyType) {

        boolean isSearchStop = searchStrategyType == DynamicTextCampaignSearchStrategyTypeEnum.SERVING_OFF;
        boolean isNetworkStop = networkStrategyType == DynamicTextCampaignNetworkStrategyTypeEnum.SERVING_OFF;

        if (isSearchStop) {
            return isNetworkStop ? null : CampaignsPlatform.CONTEXT;
        } else {
            return isNetworkStop ? CampaignsPlatform.SEARCH : CampaignsPlatform.BOTH;
        }
    }

    @Nullable
    private static CampOptionsStrategy toCampOptionsStrategy(
            DynamicTextCampaignNetworkStrategyTypeEnum networkStrategyType) {
        boolean isDifferentPlaces = (networkStrategyType != DynamicTextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT)
                && (networkStrategyType != DynamicTextCampaignNetworkStrategyTypeEnum.SERVING_OFF);
        return isDifferentPlaces ? CampOptionsStrategy.DIFFERENT_PLACES : null;
    }

    @Nullable
    private static StrategyData fromSearchStrategyData(DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       DynamicTextCampaignSearchStrategyAdd strategy) {
        switch (searchStrategyType) {
            case HIGHEST_POSITION:
                return defaultStrategyData();

            case AVERAGE_CPA:
                return fromAverageCpa(strategy.getAverageCpa());

            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());

            case AVERAGE_CRR:
                return fromAverageCrr(strategy.getAverageCrr());

            case AVERAGE_ROI:
                return fromAverageRoi(strategy.getAverageRoi());

            case PAY_FOR_CONVERSION:
                return fromPayForConversion(strategy.getPayForConversion());

            case PAY_FOR_CONVERSION_CRR:
                return fromPayForConversionCrr(strategy.getPayForConversionCrr());

            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());

            case WB_MAXIMUM_CONVERSION_RATE:
                return fromWbMaximumConversionRate(strategy.getWbMaximumConversionRate());

            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());

            default:
                return null;
        }
    }

    @Nullable
    private static StrategyData fromSearchStrategyData(DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       DynamicTextCampaignSearchStrategy strategy) {
        switch (searchStrategyType) {
            case HIGHEST_POSITION:
                return defaultStrategyData();

            case AVERAGE_CPA:
                return fromAverageCpa(strategy.getAverageCpa());

            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());

            case AVERAGE_CRR:
                return fromAverageCrr(strategy.getAverageCrr());

            case AVERAGE_ROI:
                return fromAverageRoi(strategy.getAverageRoi());

            case PAY_FOR_CONVERSION:
                return fromPayForConversion(strategy.getPayForConversion());

            case PAY_FOR_CONVERSION_CRR:
                return fromPayForConversionCrr(strategy.getPayForConversionCrr());

            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());

            case WB_MAXIMUM_CONVERSION_RATE:
                return fromWbMaximumConversionRate(strategy.getWbMaximumConversionRate());

            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());

            default:
                return null;
        }
    }

}
