package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.CampaignUpdateItem
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSetting
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignStrategy
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignUpdateItem
import com.yandex.direct.api.v5.general.YesNoEnum
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.buildModelChanges
import ru.yandex.direct.api.v5.common.processJaxbElement
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsConvertedRequest
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBannerHrefParams
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTimeTargeting
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.feature.FeatureName

@Component
class DynamicTextCampaignUpdateItemConverter @Autowired constructor(
    private val featureService: FeatureService
) : CampaignUpdateItemConverter<DynamicCampaign> {

    override fun convert(
        item: CampaignUpdateItem,
        uidAndClientId: UidAndClientId,
        timeZones: Map<String, GeoTimezone>,
        knownSsps: Set<String>,
        campaignFromDb: BaseCampaign?
    ): UpdateCampaignsConvertedRequest<DynamicCampaign> {
        check(campaignFromDb == null || campaignFromDb is DynamicCampaign)
        val dynamicItem: DynamicTextCampaignUpdateItem? = item.dynamicTextCampaign
        val settings = dynamicItem?.settings.orEmpty()
            .associateBy(DynamicTextCampaignSetting::getOption) { it.value == YesNoEnum.YES }

        val modelChanges = buildModelChanges<DynamicCampaign>(item.id) {
            val oldTimeTarget = campaignFromDb?.let { (it as CampaignWithTimeTargeting).timeTarget }
            val oldCommonCampaign = campaignFromDb?.let { (it as CommonCampaign) }
            processCommonFields(item, timeZones, oldTimeTarget, oldCommonCampaign)
            processMetrikaCounters(dynamicItem?.counterIds)
            processAttributionModel(dynamicItem?.attributionModel)
            processDailyBudget(item)
            processJaxbElement(item.negativeKeywords, DynamicCampaign.MINUS_KEYWORDS) { it!!.items }
            processExcludedSites(item, knownSsps)
            processCheckPositionInterval(item)
            processPriorityGoals(dynamicItem?.priorityGoals)
            processJaxbElement(dynamicItem?.strategyId, CampaignWithPackageStrategy.STRATEGY_ID)
            processPlacementTypes(dynamicItem?.placementTypes,
                campaignFromDb?.let { (it as DynamicCampaign).placementTypes })
            processJaxbElement(dynamicItem?.trackingParams, CampaignWithBannerHrefParams.BANNER_HREF_PARAMS)

            val apiStrategy = dynamicItem?.biddingStrategy
            val newStrategy = apiStrategy?.let {
                updateStrategyWithOldValues(it, campaignFromDb as CampaignWithStrategy)
                DynamicTextCampaignStrategyConverter.toCampaignStrategy(it)
            }
            processStrategy(newStrategy)

            // ENABLE_BEHAVIORAL_TARGETING - deprecated
            processCommonSettings(
                enableCompanyInfo = settings[DynamicTextCampaignSettingsEnum.ENABLE_COMPANY_INFO],
                // Динамическая кампания не реализует интерфейс CampaignWithNetworkSettings
                // Поэтому для неё в ядре не проставится опция enableCpcHold автоматически
                enableCpcHold = newStrategy?.isAutoBudget?.not(),
                hasTitleSubstitution = settings[DynamicTextCampaignSettingsEnum.ENABLE_EXTENDED_AD_TITLE],
                isOrderPhraseLengthPrecedenceEnabled = settings[DynamicTextCampaignSettingsEnum.CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED],
            )
            processHasAddOpenstatTagToUrl(settings[DynamicTextCampaignSettingsEnum.ADD_OPENSTAT_TAG])
            processHasAddMetrikaTagToUrl(settings[DynamicTextCampaignSettingsEnum.ADD_METRICA_TAG])
            processHasExtendedGeoTargeting(settings[DynamicTextCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING])

            val advancedGeoTargeting = featureService.isEnabledForClientId(
                uidAndClientId.clientId,
                FeatureName.ADVANCED_GEOTARGETING
            )
            if (advancedGeoTargeting) {
                processUseCurrentRegion(settings[DynamicTextCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING])
                processUseRegularRegion(settings[DynamicTextCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING])
            }
            processHasSiteMonitoring(settings[DynamicTextCampaignSettingsEnum.ENABLE_SITE_MONITORING])
            processFavoriteForIds(uidAndClientId.uid, settings[DynamicTextCampaignSettingsEnum.ADD_TO_FAVORITES])
        }

        return UpdateCampaignsConvertedRequest(
            modelChanges,
            requireServicing = settings[DynamicTextCampaignSettingsEnum.REQUIRE_SERVICING]
        )
    }

    /**
     * Дозаполняет apiStrategy данными стратегии из oldCampaign, при условии, что biddingStrategyType одинаковый.
     */
    private fun updateStrategyWithOldValues(
        apiStrategy: DynamicTextCampaignStrategy,
        campaignFromDb: CampaignWithStrategy
    ) {
        val oldStrategy = toDynamicTextCampaignExternalStrategy(campaignFromDb.strategy)
        if (apiStrategy.search == null) {
            apiStrategy.search = oldStrategy.search
        } else if (apiStrategy.search.biddingStrategyType == oldStrategy.search.biddingStrategyType) {
            copyProperties(apiStrategy.search, oldStrategy.search)
            apiStrategy.search = oldStrategy.search
        }

        // ДО бывает только на поиске, в сетях всегда SERVING_OFF
        if (apiStrategy.network == null) {
            apiStrategy.network = oldStrategy.network
        }
    }

    @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
    private fun copyProperties(
        source: DynamicTextCampaignSearchStrategy,
        dest: DynamicTextCampaignSearchStrategy
    ) {
        check(source.biddingStrategyType == dest.biddingStrategyType)
        when (source.biddingStrategyType) {
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPC ->
                copyPropertiesExceptNull(source.averageCpc, dest.averageCpc)
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPA ->
                copyPropertiesExceptNull(source.averageCpa, dest.averageCpa)
            DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION ->
                copyPropertiesExceptNull(source.payForConversion, dest.payForConversion)
            DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE ->
                copyPropertiesExceptNull(source.wbMaximumConversionRate, dest.wbMaximumConversionRate)
            DynamicTextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION -> {
                // DEFAULT_STRATEGY_NAME, no properties
            }
            DynamicTextCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH -> {
                // not used
            }
            DynamicTextCampaignSearchStrategyTypeEnum.UNKNOWN -> {
                // not used
            }
            DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS ->
                copyPropertiesExceptNull(source.wbMaximumClicks, dest.wbMaximumClicks)
            DynamicTextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
                copyPropertiesExceptNull(source.weeklyClickPackage, dest.weeklyClickPackage)
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_ROI ->
                copyPropertiesExceptNull(source.averageRoi, dest.averageRoi)
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CRR ->
                copyPropertiesExceptNull(source.averageCrr, dest.averageCrr)
            DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR ->
                copyPropertiesExceptNull(source.payForConversionCrr, dest.payForConversionCrr)
            DynamicTextCampaignSearchStrategyTypeEnum.SERVING_OFF -> {
                // no properties
            }
        }
    }
}
