package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignStrategy
import org.slf4j.LoggerFactory
import ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyConverterConstants.DEFAULT_STRATEGY_NAME
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.retargeting.model.Goal.MOBILE_GOAL_IDS
import java.lang.invoke.MethodHandles

private val logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass())

fun createMobileAppCampaignGetStrategy(campaign: MobileContentCampaign): MobileAppCampaignStrategy {
    val biddingStrategy = toMobileAppCampaignExternalStrategy(
        campaign.strategy,
        campaign.contextLimit,
    )

    return biddingStrategy.apply {
        if (search.biddingStrategyType == null) {
            search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.UNKNOWN
            logger.warn("Campaign ${campaign.id} has unknown strategy ${campaign.strategy}")
        }

        if (network.biddingStrategyType == null) {
            network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.UNKNOWN
            logger.warn("Campaign ${campaign.id} has unknown strategy ${campaign.strategy}")
        }

        // при автобюджетной стратегии в сетях NetworkDefault не заполняется
        // https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns.pm?rev=r9289187#L344-353
        if (campaign.strategy.strategyName.name.startsWith("AUTOBUDGET")) {
            network.networkDefault = null
        }
    }
}

fun toMobileAppCampaignExternalStrategy(
    internalStrategy: DbStrategy,
    contextLimit: Int?
) =
    MobileAppCampaignStrategy().apply {
        search = getSearchStrategy(internalStrategy)
        network = getNetworkStrategy(internalStrategy, contextLimit)
    }

private fun getSearchStrategy(internalStrategy: DbStrategy): MobileAppCampaignSearchStrategy {
    val search = MobileAppCampaignSearchStrategy()
    if (!internalStrategy.platform.isSearch()) {
        search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.SERVING_OFF
    } else {
        val strategyData = internalStrategy.strategyData
        when (strategyData.name) {
            DEFAULT_STRATEGY_NAME -> {
                search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.HIGHEST_POSITION
            }
            StrategyName.AUTOBUDGET_AVG_CLICK.name.lowercase() -> {
                search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPC
                search.averageCpc = toStrategyAverageCpc(strategyData)
            }
            StrategyName.AUTOBUDGET_AVG_CPI.name.lowercase() -> {
                if (strategyData.payForConversion == true) {
                    search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.PAY_FOR_INSTALL
                    search.payForInstall = toStrategyPayForInstall(strategyData)
                } else {
                    search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPI
                    search.averageCpi = toStrategyAverageCpi(strategyData)
                }
            }
            StrategyName.AUTOBUDGET.name.lowercase() -> {
                when (strategyData.goalId) {
                    null -> {
                        search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS
                        search.wbMaximumClicks = toStrategyMaximumClicks(strategyData)
                    }
                    in MOBILE_GOAL_IDS -> {
                        search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS
                        search.wbMaximumAppInstalls = toStrategyMaximumAppInstalls(strategyData)
                    }
                }
            }
            StrategyName.AUTOBUDGET_WEEK_BUNDLE.name.lowercase() -> {
                search.biddingStrategyType = MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
                search.weeklyClickPackage = toStrategyWeeklyClickPackage(strategyData)
            }
        }
    }
    return search
}

private fun getNetworkStrategy(
    internalStrategy: DbStrategy,
    contextLimit: Int?
): MobileAppCampaignNetworkStrategy {
    val network = MobileAppCampaignNetworkStrategy()
    val strategyData = internalStrategy.strategyData

    when (internalStrategy.platform) {
        CampaignsPlatform.SEARCH -> {
            network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.SERVING_OFF
        }
        CampaignsPlatform.BOTH -> {
            if (internalStrategy.strategy == null) {
                network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT
                network.networkDefault = toStrategyNetworkDefault(contextLimit)
            } else {
                network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE
            }
        }
        else -> when (strategyData.name) {
            DEFAULT_STRATEGY_NAME -> {
                network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE
            }
            StrategyName.AUTOBUDGET_AVG_CLICK.name.lowercase() -> {
                network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPC
                network.averageCpc = toStrategyAverageCpc(strategyData)
            }
            StrategyName.AUTOBUDGET_AVG_CPI.name.lowercase() -> {
                if (strategyData.payForConversion == true) {
                    network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.PAY_FOR_INSTALL
                    network.payForInstall = toStrategyPayForInstall(strategyData)
                } else {
                    network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPI
                    network.averageCpi = toStrategyAverageCpi(strategyData)
                }
            }
            StrategyName.AUTOBUDGET.name.lowercase() -> {
                when (strategyData.goalId) {
                    null -> {
                        network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS
                        network.wbMaximumClicks = toStrategyMaximumClicks(strategyData)
                    }
                    in MOBILE_GOAL_IDS -> {
                        network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS
                        network.wbMaximumAppInstalls = toStrategyMaximumAppInstalls(strategyData)
                    }
                }
            }
            StrategyName.AUTOBUDGET_WEEK_BUNDLE.name.lowercase() -> {
                network.biddingStrategyType = MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
                network.weeklyClickPackage = toStrategyWeeklyClickPackage(strategyData)
            }
        }
    }

    return network
}
