package ru.yandex.direct.api.v5.entity.campaigns.converter;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategy;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategy;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignStrategy;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignStrategyAdd;

import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;

import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.defaultStrategyData;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpc;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpi;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromPayForInstall;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbMaximumAppInstalls;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbMaximumClicks;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWeeklyClickPackage;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.toCampaignsAutobudget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpc;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpi;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromPayForInstall;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbMaximumAppInstalls;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbMaximumClicks;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWeeklyClickPackage;

@ParametersAreNonnullByDefault
public class MobileAppCampaignStrategyConverter {

    /**
     * IMPRESSIONS_BELOW_SEARCH - deprecated
     */
    private static final Map<MobileAppCampaignSearchStrategyTypeEnum, StrategyName>
            STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE =
            ImmutableMap.<MobileAppCampaignSearchStrategyTypeEnum, StrategyName>builder()
                    .put(MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPC, StrategyName.AUTOBUDGET_AVG_CLICK)
                    .put(MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPI, StrategyName.AUTOBUDGET_AVG_CPI)
                    .put(MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS, StrategyName.AUTOBUDGET)
                    .put(MobileAppCampaignSearchStrategyTypeEnum.HIGHEST_POSITION, StrategyName.DEFAULT_)
                    .put(MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS, StrategyName.AUTOBUDGET)
                    .put(MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
                            StrategyName.AUTOBUDGET_WEEK_BUNDLE)
                    .put(MobileAppCampaignSearchStrategyTypeEnum.PAY_FOR_INSTALL, StrategyName.AUTOBUDGET_AVG_CPI)
                    .build();

    private static final Map<MobileAppCampaignNetworkStrategyTypeEnum, StrategyName>
            STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE =
            ImmutableMap.<MobileAppCampaignNetworkStrategyTypeEnum, StrategyName>builder()
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE, StrategyName.DEFAULT_)
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPC, StrategyName.AUTOBUDGET_AVG_CLICK)
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPI, StrategyName.AUTOBUDGET_AVG_CPI)
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS, StrategyName.AUTOBUDGET)
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS, StrategyName.AUTOBUDGET)
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
                            StrategyName.AUTOBUDGET_WEEK_BUNDLE)
                    .put(MobileAppCampaignNetworkStrategyTypeEnum.PAY_FOR_INSTALL, StrategyName.AUTOBUDGET_AVG_CPI)
                    .build();

    private MobileAppCampaignStrategyConverter() {
    }

    public static DbStrategy toCampaignStrategy(MobileAppCampaignStrategy apiStrategy) {
        var dbStrategy = new DbStrategy();

        MobileAppCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType = apiStrategy.getNetwork()
                .getBiddingStrategyType();

        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));

        } else if (STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.containsKey(networkStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.get(networkStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromNetworkStrategyData(networkStrategyType, apiStrategy.getNetwork()));

        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType, networkStrategyType));
        dbStrategy.withStrategy(toCampOptionsStrategy(networkStrategyType));

        return dbStrategy;
    }

    public static DbStrategy toCampaignStrategy(MobileAppCampaignStrategyAdd apiStrategy) {
        var dbStrategy = new DbStrategy();

        MobileAppCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType = apiStrategy.getNetwork()
                .getBiddingStrategyType();

        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));

        } else if (STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.containsKey(networkStrategyType)) {
            StrategyName strategyName = STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.get(networkStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromNetworkStrategyData(networkStrategyType, apiStrategy.getNetwork()));

        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType, networkStrategyType));
        dbStrategy.withStrategy(toCampOptionsStrategy(networkStrategyType));

        return dbStrategy;
    }

    private static StrategyData fromSearchStrategyData(MobileAppCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       MobileAppCampaignSearchStrategyAdd strategy) {
        switch (searchStrategyType) {
            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());
            case AVERAGE_CPI:
                return fromAverageCpi(strategy.getAverageCpi());
            case WB_MAXIMUM_APP_INSTALLS:
                return fromWbMaximumAppInstalls(strategy.getWbMaximumAppInstalls());
            case HIGHEST_POSITION:
                return defaultStrategyData();
            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());
            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());
            case PAY_FOR_INSTALL:
                return fromPayForInstall(strategy.getPayForInstall());
            default:
                return null;
        }
    }

    private static StrategyData fromNetworkStrategyData(MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType,
                                                        MobileAppCampaignNetworkStrategyAdd strategy) {
        switch (networkStrategyType) {
            case MAXIMUM_COVERAGE:
                return defaultStrategyData();
            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());
            case AVERAGE_CPI:
                return fromAverageCpi(strategy.getAverageCpi());
            case WB_MAXIMUM_APP_INSTALLS:
                return fromWbMaximumAppInstalls(strategy.getWbMaximumAppInstalls());
            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());
            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());
            case PAY_FOR_INSTALL:
                return fromPayForInstall(strategy.getPayForInstall());
            default:
                return null;
        }
    }

    private static StrategyData fromSearchStrategyData(MobileAppCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       MobileAppCampaignSearchStrategy strategy) {
        switch (searchStrategyType) {
            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());
            case AVERAGE_CPI:
                return fromAverageCpi(strategy.getAverageCpi());
            case WB_MAXIMUM_APP_INSTALLS:
                return fromWbMaximumAppInstalls(strategy.getWbMaximumAppInstalls());
            case HIGHEST_POSITION:
                return defaultStrategyData();
            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());
            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());
            case PAY_FOR_INSTALL:
                return fromPayForInstall(strategy.getPayForInstall());
            default:
                return null;
        }
    }

    private static StrategyData fromNetworkStrategyData(MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType,
                                                        MobileAppCampaignNetworkStrategy strategy) {
        switch (networkStrategyType) {
            case MAXIMUM_COVERAGE:
                return defaultStrategyData();
            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());
            case AVERAGE_CPI:
                return fromAverageCpi(strategy.getAverageCpi());
            case WB_MAXIMUM_APP_INSTALLS:
                return fromWbMaximumAppInstalls(strategy.getWbMaximumAppInstalls());
            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());
            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());
            case PAY_FOR_INSTALL:
                return fromPayForInstall(strategy.getPayForInstall());
            default:
                return null;
        }
    }

    @Nullable
    private static CampaignsPlatform toCampaignsPlatform(MobileAppCampaignSearchStrategyTypeEnum searchStrategyType,
                                                         MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType) {

        boolean isSearchStop = searchStrategyType == MobileAppCampaignSearchStrategyTypeEnum.SERVING_OFF;
        boolean isNetworkStop = networkStrategyType == MobileAppCampaignNetworkStrategyTypeEnum.SERVING_OFF;

        if (isSearchStop) {
            return isNetworkStop ? null : CampaignsPlatform.CONTEXT;
        } else {
            return isNetworkStop ? CampaignsPlatform.SEARCH : CampaignsPlatform.BOTH;
        }
    }

    private static CampOptionsStrategy toCampOptionsStrategy(
            MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType) {
        boolean isDifferentPlaces = (networkStrategyType != MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT)
                && (networkStrategyType != MobileAppCampaignNetworkStrategyTypeEnum.SERVING_OFF);
        return isDifferentPlaces ? CampOptionsStrategy.DIFFERENT_PLACES : null;
    }

}
