package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.CampaignUpdateItem
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSetting
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignUpdateItem
import com.yandex.direct.api.v5.general.YesNoEnum
import org.apache.commons.beanutils.BeanUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.buildModelChanges
import ru.yandex.direct.api.v5.common.processJaxbElement
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsConvertedRequest
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTimeTargeting
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.feature.FeatureName

@Component
class MobileAppCampaignUpdateItemConverter @Autowired constructor(
    private val featureService: FeatureService
) : CampaignUpdateItemConverter<MobileContentCampaign> {

    override fun convert(
        item: CampaignUpdateItem,
        uidAndClientId: UidAndClientId,
        timeZones: Map<String, GeoTimezone>,
        knownSsps: Set<String>,
        campaignFromDb: BaseCampaign?
    ): UpdateCampaignsConvertedRequest<MobileContentCampaign> {
        check(campaignFromDb == null || campaignFromDb is MobileContentCampaign)
        val mobileAppItem: MobileAppCampaignUpdateItem? = item.mobileAppCampaign
        val settings = mobileAppItem?.settings.orEmpty()
            .associateBy(MobileAppCampaignSetting::getOption) { it.value == YesNoEnum.YES }

        val modelChanges = buildModelChanges<MobileContentCampaign>(item.id) {
            val oldTimeTarget = campaignFromDb?.let { (it as CampaignWithTimeTargeting).timeTarget }
            val oldCommonCampaign = campaignFromDb?.let { (it as CommonCampaign) }
            processCommonFields(item, timeZones, oldTimeTarget, oldCommonCampaign)
            processDailyBudget(item)
            processJaxbElement(item.negativeKeywords, MobileContentCampaign.MINUS_KEYWORDS) { it!!.items }
            processExcludedSites(item, knownSsps)
            processCheckPositionInterval(item)
            processJaxbElement(mobileAppItem?.strategyId, CampaignWithPackageStrategy.STRATEGY_ID)

            val apiStrategy = mobileAppItem?.biddingStrategy

            val newStrategy = apiStrategy?.let {
                val newApiStrategy = mergeOldAndNewStrategies(it, campaignFromDb as MobileContentCampaign)
                MobileAppCampaignStrategyConverter.toCampaignStrategy(newApiStrategy)
            }
            processStrategy(newStrategy)
            processContextLimit(apiStrategy?.network?.toContextLimit())

            // ENABLE_RELATED_KEYWORDS, ENABLE_AUTOFOCUS, ENABLE_BEHAVIORAL_TARGETING - deprecated
            processCommonSettings(
                // enableCpcHold перезапишется в ядре, если стратегия - автобюджетная
                // https://a.yandex-team.ru/arc_vcs/direct/core/src/main/java/ru/yandex/direct/core/entity/campaign/service/type/update/CampaignWithNetworkSettingsUpdateOperationSupport.java?rev=r7408600#L42
                enableCpcHold = settings[MobileAppCampaignSettingsEnum.MAINTAIN_NETWORK_CPC],
                isOrderPhraseLengthPrecedenceEnabled = settings[MobileAppCampaignSettingsEnum.CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED],
            )
            processHasExtendedGeoTargeting(settings[MobileAppCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING])

            val advancedGeoTargeting = featureService.isEnabledForClientId(
                uidAndClientId.clientId,
                FeatureName.ADVANCED_GEOTARGETING
            )
            if (advancedGeoTargeting) {
                processUseCurrentRegion(settings[MobileAppCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING])
                processUseRegularRegion(settings[MobileAppCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING])
            }
            processFavoriteForIds(uidAndClientId.uid, settings[MobileAppCampaignSettingsEnum.ADD_TO_FAVORITES])
        }

        return UpdateCampaignsConvertedRequest(
            modelChanges,
            requireServicing = settings[MobileAppCampaignSettingsEnum.REQUIRE_SERVICING],
        )
    }

    companion object {
        /**
         * Мёржит apiStrategy данными с стратегии из oldCampaign, при условии, что biddingStrategyType одинаковый и
         * возвращает объединённую стратегию
         */
        fun mergeOldAndNewStrategies(
            apiStrategy: MobileAppCampaignStrategy?,
            campaignFromDb: MobileContentCampaign
        ): MobileAppCampaignStrategy {
            val result = MobileAppCampaignStrategy().apply {
                if (apiStrategy != null) {
                    BeanUtils.copyProperties(this, apiStrategy)
                }
            }
            val oldStrategy = toMobileAppCampaignExternalStrategy(campaignFromDb.strategy, campaignFromDb.contextLimit)
            if (result.search == null) {
                result.search = oldStrategy.search
            } else if (result.search.biddingStrategyType == oldStrategy.search.biddingStrategyType) {
                copyProperties(result.search, oldStrategy.search)
                result.search = oldStrategy.search
            }
            if (result.network == null) {
                result.network = oldStrategy.network
            } else if (result.network.biddingStrategyType == oldStrategy.network.biddingStrategyType) {
                copyProperties(result.network, oldStrategy.network)
                result.network = oldStrategy.network
            }
            return result
        }

        @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
        private fun copyProperties(
            source: MobileAppCampaignSearchStrategy,
            dest: MobileAppCampaignSearchStrategy
        ) {
            check(source.biddingStrategyType == dest.biddingStrategyType)
            when (source.biddingStrategyType) {
                MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPC ->
                    copyPropertiesExceptNull(source.averageCpc, dest.averageCpc)
                MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPI ->
                    copyPropertiesExceptNull(source.averageCpi, dest.averageCpi)
                MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS ->
                    copyPropertiesExceptNull(source.wbMaximumAppInstalls, dest.wbMaximumAppInstalls)
                MobileAppCampaignSearchStrategyTypeEnum.HIGHEST_POSITION -> {
                    // DEFAULT_STRATEGY_NAME, no properties
                }
                MobileAppCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH -> {
                    // not used
                }
                MobileAppCampaignSearchStrategyTypeEnum.SERVING_OFF -> {
                    // no properties
                }
                MobileAppCampaignSearchStrategyTypeEnum.UNKNOWN -> {
                    // not used
                }
                MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS ->
                    copyPropertiesExceptNull(source.wbMaximumClicks, dest.wbMaximumClicks)
                MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
                    copyPropertiesExceptNull(source.weeklyClickPackage, dest.weeklyClickPackage)
                MobileAppCampaignSearchStrategyTypeEnum.PAY_FOR_INSTALL ->
                    copyPropertiesExceptNull(source.payForInstall, dest.payForInstall)
            }
        }

        @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
        private fun copyProperties(
            source: MobileAppCampaignNetworkStrategy,
            dest: MobileAppCampaignNetworkStrategy
        ) {
            check(source.biddingStrategyType == dest.biddingStrategyType)
            when (source.biddingStrategyType) {
                MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPC ->
                    copyPropertiesExceptNull(source.averageCpc, dest.averageCpc)
                MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPI ->
                    copyPropertiesExceptNull(source.averageCpi, dest.averageCpi)
                MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS ->
                    copyPropertiesExceptNull(source.wbMaximumAppInstalls, dest.wbMaximumAppInstalls)
                MobileAppCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE -> {
                    // DEFAULT_STRATEGY_NAME, no properties
                }
                MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT -> {
                    copyPropertiesExceptNull(source.networkDefault, dest.networkDefault)
                }
                MobileAppCampaignNetworkStrategyTypeEnum.SERVING_OFF -> {
                    // no properties
                }
                MobileAppCampaignNetworkStrategyTypeEnum.UNKNOWN -> {
                    // not used
                }
                MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS ->
                    copyPropertiesExceptNull(source.wbMaximumClicks, dest.wbMaximumClicks)
                MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
                    copyPropertiesExceptNull(source.weeklyClickPackage, dest.weeklyClickPackage)
                MobileAppCampaignNetworkStrategyTypeEnum.PAY_FOR_INSTALL ->
                    copyPropertiesExceptNull(source.payForInstall, dest.payForInstall)
            }
        }
    }
}
