package ru.yandex.direct.api.v5.entity.campaigns.converter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.CampaignAddItem;
import com.yandex.direct.api.v5.campaigns.SmartCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSetting;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSettingsEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsApiValidationSignals;
import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractCheckPositionIntervalEvent;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDisabledDomains;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDisabledIps;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDisabledSsp;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractEmail;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractEnableCheckPositionEvent;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractMeaningfulGoals;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractMinusKeywords;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractSendAccountNews;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractSmsFlags;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractSmsTimeInterval;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractTimeTarget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractWarningBalance;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.toAttributionModel;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.toDate;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.SmartCampaignStrategyConverter.toCampaignStrategy;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ADD_OPENSTAT_TAG_TO_URL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ADD_TO_FAVORITES;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_DAY_BUDGET;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_DAY_BUDGET_SHOW_MODE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ENABLE_COMPANY_INFO;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_HAS_EXTENDED_GEO_TARGETING;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_HAS_SITE_MONITORING;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_HAS_TITLE_SUBSTITUTION;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class SmartCampaignAddItemConverter {

    private static final Map<SmartCampaignSettingsEnum, Boolean> DEFAULT_SETTINGS = Map.of(
            SmartCampaignSettingsEnum.ADD_TO_FAVORITES, DEFAULT_ADD_TO_FAVORITES,
            SmartCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING, DEFAULT_HAS_EXTENDED_GEO_TARGETING
    );

    public static CampaignsApiValidationSignals.CampaignWrapper convertSmartCampaign(
            CampaignAddItem item,
            Long chiefUid,
            String userEmail,
            Map<String, GeoTimezone> timeZones,
            CampaignSource campaignSource,
            Set<String> knownSsps,
            boolean advancedGeoTargeting) {
        SmartCampaignAddItem smartCampaignItem = item.getSmartCampaign();

        Map<SmartCampaignSettingsEnum, Boolean> settings =
                convertSmartCampaignSettings(smartCampaignItem.getSettings(), advancedGeoTargeting);
        Set<Long> favoriteForUids = settings.get(SmartCampaignSettingsEnum.ADD_TO_FAVORITES) ? Set.of(chiefUid) : null;

        var campaign = new SmartCampaign()
                .withSource(campaignSource)
                .withType(CampaignType.PERFORMANCE)
                .withName(item.getName())
                .withStrategy(toCampaignStrategy(smartCampaignItem.getBiddingStrategy()))

                /*
                  На самом деле это значение не всегда применится, в ядре есть такой код:
                  if (isAutoBudget || isDifferentPlaces) {
                    campaign.setContextLimit(AUTO_CONTEXT_LIMIT);
                  }
                  Здесь проставляем всегда, потому что знаем, что ядро когда надо - перезатрёт.
                  null ядро заменит на 0, покрывается авто-тестом StrategyDefaultBidsManualTest
                 */
                .withContextLimit(CampaignConversionUtilsKt.toContextLimit(
                        smartCampaignItem.getBiddingStrategy().getNetwork()))

                .withFavoriteForUids(favoriteForUids)
                .withStartDate(toDate(item.getStartDate()))
                .withEndDate(toDate(item.getEndDate()))
                .withEmail(userEmail)
                .withSmsTime(extractSmsTimeInterval(item))
                .withSmsFlags(extractSmsFlags(item))
                // В API5 RelevantKeywords не реализованы для SmartCampaign поэтому здесь проставляем "отключены"
                .withBroadMatch(new BroadMatch()
                        .withBroadMatchFlag(false)
                        .withBroadMatchLimit(0)
                        .withBroadMatchGoalId(0L))
                .withTimeZoneId(item.getTimeZone() == null ? 0 : timeZones.get(item.getTimeZone()).getTimezoneId())
                .withTimeTarget(extractTimeTarget(item))
                .withHasAddMetrikaTagToUrl(false)
                .withMetrikaCounters(List.of(smartCampaignItem.getCounterId()))
                .withFio(item.getClientInfo())
                .withHasExtendedGeoTargeting(settings.get(SmartCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING))
                .withUseCurrentRegion(settings.get(SmartCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING))
                .withUseRegularRegion(settings.get(SmartCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING))
                .withMinusKeywords(extractMinusKeywords(item))
                .withEmail(nvl(extractEmail(item.getNotification()), userEmail))
                .withWarningBalance(extractWarningBalance(item.getNotification()))
                .withCheckPositionIntervalEvent(extractCheckPositionIntervalEvent(item.getNotification()))
                .withEnableSendAccountNews(extractSendAccountNews(item))
                .withEnableCheckPositionEvent(extractEnableCheckPositionEvent(item.getNotification()))
                .withDisabledIps(extractDisabledIps(item))
                .withDisabledSsp(extractDisabledSsp(item, knownSsps))
                .withDisabledDomains(extractDisabledDomains(item, knownSsps))
                .withAttributionModel(toAttributionModel(smartCampaignItem.getAttributionModel()))
                .withMeaningfulGoals(extractMeaningfulGoals(smartCampaignItem))
                .withStrategyId(smartCampaignItem.getStrategyId())
                .withBannerHrefParams(smartCampaignItem.getTrackingParams())
                // не используется, заполняем дефолтным значением
                .withEnableCompanyInfo(DEFAULT_ENABLE_COMPANY_INFO)
                // не используется, заполняем дефолтным значением
                .withEnableCpcHold(false)
                // не используется, заполняем дефолтным значением
                .withHasSiteMonitoring(DEFAULT_HAS_SITE_MONITORING)
                // не используется, заполняем дефолтным значением
                .withHasTitleSubstitution(DEFAULT_HAS_TITLE_SUBSTITUTION)
                // не используется, заполняем дефолтным значением
                .withDayBudget(DEFAULT_DAY_BUDGET)
                // не используется, заполняем дефолтным значением
                .withDayBudgetShowMode(DEFAULT_DAY_BUDGET_SHOW_MODE)
                // не используется, заполняем дефолтным значением
                .withHasAddOpenstatTagToUrl(DEFAULT_ADD_OPENSTAT_TAG_TO_URL)
                // не используется, заполняем дефолтным значением
                .withPlacementTypes(emptySet());

        return new CampaignsApiValidationSignals.CampaignWrapper()
                .withRealCampaign(campaign)
                .withRequireServicing(settings.get(SmartCampaignSettingsEnum.REQUIRE_SERVICING));
    }

    private static Map<SmartCampaignSettingsEnum, Boolean> convertSmartCampaignSettings(
            List<SmartCampaignSetting> settings, boolean advancedGeoTargeting) {
        Map<SmartCampaignSettingsEnum, Boolean> valueByOption = new HashMap<>();
        for (SmartCampaignSetting s : settings) {
            valueByOption.put(s.getOption(), s.getValue() == YesNoEnum.YES);
        }

        if (advancedGeoTargeting) {
            if (!valueByOption.containsKey(SmartCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING)) {
                valueByOption.put(SmartCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING, true);
            }
            if (!valueByOption.containsKey(SmartCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING)) {
                valueByOption.put(SmartCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING, true);
            }
        }
        return StreamEx.of(SmartCampaignSettingsEnum.values())
                .mapToEntry(option -> valueByOption.getOrDefault(option, DEFAULT_SETTINGS.get(option)))
                .nonNullValues()
                .toMap();
    }

}
