package ru.yandex.direct.api.v5.entity.campaigns.converter;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategy;
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategy;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.SmartCampaignStrategy;
import com.yandex.direct.api.v5.campaigns.SmartCampaignStrategyAdd;

import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;

import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpaPerCampaign;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpaPerFilter;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpcPerCampaign;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCpcPerFilter;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromAverageRoi;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromPayForConversionCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromPayForConversionPerCampaign;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromPayForConversionPerFilter;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.fromWbMaximumConversionRate;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.toCampOptionsStrategy;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyAddConverter.toCampaignsAutobudget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpaPerCampaign;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpaPerFilter;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpcPerCampaign;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCpcPerFilter;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromAverageRoi;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromPayForConversionCrr;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromPayForConversionPerCampaign;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromPayForConversionPerFilter;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyUpdateConverter.fromWbMaximumConversionRate;

/**
 * Таблица совместимости стратегий https://yandex.ru/dev/direct/doc/dg/objects/campaign-strategies.html
 */
@ParametersAreNonnullByDefault
public class SmartCampaignStrategyConverter {

    private static final Map<SmartCampaignSearchStrategyTypeEnum, StrategyName> STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE =
            ImmutableMap.<SmartCampaignSearchStrategyTypeEnum, StrategyName>builder()
                    .put(SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPC_PER_CAMPAIGN,
                            StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP)
                    .put(SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPC_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER)
                    .put(SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPA_PER_CAMPAIGN,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP)
                    .put(SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPA_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER)
                    .put(SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_PER_CAMPAIGN,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP)
                    .put(SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER)
                    .put(SmartCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
                            StrategyName.AUTOBUDGET)
                    .put(SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR,
                            StrategyName.AUTOBUDGET_CRR)
                    .put(SmartCampaignSearchStrategyTypeEnum.AVERAGE_ROI, StrategyName.AUTOBUDGET_ROI)
                    .put(SmartCampaignSearchStrategyTypeEnum.AVERAGE_CRR, StrategyName.AUTOBUDGET_CRR)
                    .build();

    private static final Map<SmartCampaignNetworkStrategyTypeEnum, StrategyName> STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE =
            ImmutableMap.<SmartCampaignNetworkStrategyTypeEnum, StrategyName>builder()
                    .put(SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPC_PER_CAMPAIGN,
                            StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP)
                    .put(SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPC_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER)
                    .put(SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPA_PER_CAMPAIGN,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP)
                    .put(SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPA_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER)
                    .put(SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_PER_CAMPAIGN,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP)
                    .put(SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_PER_FILTER,
                            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER)
                    .put(SmartCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
                            StrategyName.AUTOBUDGET)
                    .put(SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_CRR,
                            StrategyName.AUTOBUDGET_CRR)
                    .put(SmartCampaignNetworkStrategyTypeEnum.AVERAGE_ROI, StrategyName.AUTOBUDGET_ROI)
                    .put(SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CRR, StrategyName.AUTOBUDGET_CRR)
                    .build();

    private SmartCampaignStrategyConverter() {
    }

    public static DbStrategy toCampaignStrategy(SmartCampaignStrategy apiStrategy) {
        var dbStrategy = new DbStrategy();

        SmartCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        SmartCampaignNetworkStrategyTypeEnum networkStrategyType = apiStrategy.getNetwork().getBiddingStrategyType();

        StrategyName strategyName;
        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));
        } else if (STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.containsKey(networkStrategyType)) {
            strategyName = STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.get(networkStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromNetworkStrategyData(networkStrategyType, apiStrategy.getNetwork()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType, networkStrategyType));
        dbStrategy.withStrategy(toCampOptionsStrategy(strategyName));
        return dbStrategy;
    }

    public static DbStrategy toCampaignStrategy(SmartCampaignStrategyAdd apiStrategy) {
        var dbStrategy = new DbStrategy();

        SmartCampaignSearchStrategyTypeEnum searchStrategyType = apiStrategy.getSearch().getBiddingStrategyType();
        SmartCampaignNetworkStrategyTypeEnum networkStrategyType = apiStrategy.getNetwork().getBiddingStrategyType();

        StrategyName strategyName;
        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));
        } else if (STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.containsKey(networkStrategyType)) {
            strategyName = STRATEGY_NAME_BY_NETWORK_STRATEGY_TYPE.get(networkStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromNetworkStrategyData(networkStrategyType, apiStrategy.getNetwork()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        dbStrategy.withPlatform(toCampaignsPlatform(searchStrategyType, networkStrategyType));
        dbStrategy.withStrategy(toCampOptionsStrategy(strategyName));
        return dbStrategy;
    }

    @Nullable
    private static CampaignsPlatform toCampaignsPlatform(SmartCampaignSearchStrategyTypeEnum searchStrategyType,
                                                         SmartCampaignNetworkStrategyTypeEnum networkStrategyType) {

        boolean isSearchStop = searchStrategyType == SmartCampaignSearchStrategyTypeEnum.SERVING_OFF;
        boolean isNetworkStop = networkStrategyType == SmartCampaignNetworkStrategyTypeEnum.SERVING_OFF;

        if (isSearchStop) {
            return isNetworkStop ? null : CampaignsPlatform.CONTEXT;
        } else {
            return isNetworkStop ? CampaignsPlatform.SEARCH : CampaignsPlatform.BOTH;
        }
    }

    @Nullable
    private static StrategyData fromSearchStrategyData(SmartCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       SmartCampaignSearchStrategyAdd strategy) {
        switch (searchStrategyType) {
            case AVERAGE_CPC_PER_CAMPAIGN:
                return fromAverageCpcPerCampaign(strategy.getAverageCpcPerCampaign());
            case AVERAGE_CPC_PER_FILTER:
                return fromAverageCpcPerFilter(strategy.getAverageCpcPerFilter());
            case AVERAGE_CPA_PER_CAMPAIGN:
                return fromAverageCpaPerCampaign(strategy.getAverageCpaPerCampaign());
            case AVERAGE_CPA_PER_FILTER:
                return fromAverageCpaPerFilter(strategy.getAverageCpaPerFilter());
            case PAY_FOR_CONVERSION_PER_CAMPAIGN:
                return fromPayForConversionPerCampaign(strategy.getPayForConversionPerCampaign());
            case PAY_FOR_CONVERSION_PER_FILTER:
                return fromPayForConversionPerFilter(strategy.getPayForConversionPerFilter());
            case WB_MAXIMUM_CONVERSION_RATE:
                return fromWbMaximumConversionRate(strategy.getWbMaximumConversionRate());
            case PAY_FOR_CONVERSION_CRR:
                return fromPayForConversionCrr(strategy.getPayForConversionCrr());
            case AVERAGE_ROI:
                return fromAverageRoi(strategy.getAverageRoi());
            case AVERAGE_CRR:
                return fromAverageCrr(strategy.getAverageCrr());
            default:
                return null;
        }
    }

    @Nullable
    private static StrategyData fromNetworkStrategyData(SmartCampaignNetworkStrategyTypeEnum networkStrategyType,
                                                        SmartCampaignNetworkStrategyAdd strategy) {
        switch (networkStrategyType) {
            case AVERAGE_CPC_PER_CAMPAIGN:
                return fromAverageCpcPerCampaign(strategy.getAverageCpcPerCampaign());
            case AVERAGE_CPC_PER_FILTER:
                return fromAverageCpcPerFilter(strategy.getAverageCpcPerFilter());
            case AVERAGE_CPA_PER_CAMPAIGN:
                return fromAverageCpaPerCampaign(strategy.getAverageCpaPerCampaign());
            case AVERAGE_CPA_PER_FILTER:
                return fromAverageCpaPerFilter(strategy.getAverageCpaPerFilter());
            case PAY_FOR_CONVERSION_PER_CAMPAIGN:
                return fromPayForConversionPerCampaign(strategy.getPayForConversionPerCampaign());
            case PAY_FOR_CONVERSION_PER_FILTER:
                return fromPayForConversionPerFilter(strategy.getPayForConversionPerFilter());
            case WB_MAXIMUM_CONVERSION_RATE:
                return fromWbMaximumConversionRate(strategy.getWbMaximumConversionRate());
            case PAY_FOR_CONVERSION_CRR:
                return fromPayForConversionCrr(strategy.getPayForConversionCrr());
            case AVERAGE_ROI:
                return fromAverageRoi(strategy.getAverageRoi());
            case AVERAGE_CRR:
                return fromAverageCrr(strategy.getAverageCrr());
            default:
                return null;
        }
    }

    @Nullable
    private static StrategyData fromSearchStrategyData(SmartCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       SmartCampaignSearchStrategy strategy) {
        switch (searchStrategyType) {
            case AVERAGE_CPC_PER_CAMPAIGN:
                return fromAverageCpcPerCampaign(strategy.getAverageCpcPerCampaign());
            case AVERAGE_CPC_PER_FILTER:
                return fromAverageCpcPerFilter(strategy.getAverageCpcPerFilter());
            case AVERAGE_CPA_PER_CAMPAIGN:
                return fromAverageCpaPerCampaign(strategy.getAverageCpaPerCampaign());
            case AVERAGE_CPA_PER_FILTER:
                return fromAverageCpaPerFilter(strategy.getAverageCpaPerFilter());
            case PAY_FOR_CONVERSION_PER_CAMPAIGN:
                return fromPayForConversionPerCampaign(strategy.getPayForConversionPerCampaign());
            case PAY_FOR_CONVERSION_PER_FILTER:
                return fromPayForConversionPerFilter(strategy.getPayForConversionPerFilter());
            case WB_MAXIMUM_CONVERSION_RATE:
                return fromWbMaximumConversionRate(strategy.getWbMaximumConversionRate());
            case PAY_FOR_CONVERSION_CRR:
                return fromPayForConversionCrr(strategy.getPayForConversionCrr());
            case AVERAGE_ROI:
                return fromAverageRoi(strategy.getAverageRoi());
            case AVERAGE_CRR:
                return fromAverageCrr(strategy.getAverageCrr());
            default:
                return null;
        }
    }

    @Nullable
    private static StrategyData fromNetworkStrategyData(SmartCampaignNetworkStrategyTypeEnum networkStrategyType,
                                                        SmartCampaignNetworkStrategy strategy) {
        switch (networkStrategyType) {
            case AVERAGE_CPC_PER_CAMPAIGN:
                return fromAverageCpcPerCampaign(strategy.getAverageCpcPerCampaign());
            case AVERAGE_CPC_PER_FILTER:
                return fromAverageCpcPerFilter(strategy.getAverageCpcPerFilter());
            case AVERAGE_CPA_PER_CAMPAIGN:
                return fromAverageCpaPerCampaign(strategy.getAverageCpaPerCampaign());
            case AVERAGE_CPA_PER_FILTER:
                return fromAverageCpaPerFilter(strategy.getAverageCpaPerFilter());
            case PAY_FOR_CONVERSION_PER_CAMPAIGN:
                return fromPayForConversionPerCampaign(strategy.getPayForConversionPerCampaign());
            case PAY_FOR_CONVERSION_PER_FILTER:
                return fromPayForConversionPerFilter(strategy.getPayForConversionPerFilter());
            case WB_MAXIMUM_CONVERSION_RATE:
                return fromWbMaximumConversionRate(strategy.getWbMaximumConversionRate());
            case PAY_FOR_CONVERSION_CRR:
                return fromPayForConversionCrr(strategy.getPayForConversionCrr());
            case AVERAGE_ROI:
                return fromAverageRoi(strategy.getAverageRoi());
            case AVERAGE_CRR:
                return fromAverageCrr(strategy.getAverageCrr());
            default:
                return null;
        }
    }

}
