package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.CampaignUpdateItem
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.SmartCampaignSetting
import com.yandex.direct.api.v5.campaigns.SmartCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.SmartCampaignStrategy
import com.yandex.direct.api.v5.campaigns.SmartCampaignUpdateItem
import com.yandex.direct.api.v5.general.YesNoEnum
import org.apache.commons.beanutils.BeanUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.buildModelChanges
import ru.yandex.direct.api.v5.common.processJaxbElement
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsConvertedRequest
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBannerHrefParams
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTimeTargeting
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.feature.FeatureName

@Component
class SmartCampaignUpdateItemConverter @Autowired constructor(
    private val featureService: FeatureService
) : CampaignUpdateItemConverter<SmartCampaign> {

    override fun convert(
        item: CampaignUpdateItem,
        uidAndClientId: UidAndClientId,
        timeZones: Map<String, GeoTimezone>,
        knownSsps: Set<String>,
        campaignFromDb: BaseCampaign?
    ): UpdateCampaignsConvertedRequest<SmartCampaign> {
        check(campaignFromDb == null || campaignFromDb is SmartCampaign)
        val smartItem: SmartCampaignUpdateItem? = item.smartCampaign
        val settings = smartItem?.settings.orEmpty()
            .associateBy(SmartCampaignSetting::getOption) { it.value == YesNoEnum.YES }

        val modelChanges = buildModelChanges<SmartCampaign>(item.id) {
            val oldTimeTarget = campaignFromDb?.let { (it as CampaignWithTimeTargeting).timeTarget }
            val oldCommonCampaign = campaignFromDb?.let { (it as CommonCampaign) }
            processCommonFields(item, timeZones, oldTimeTarget, oldCommonCampaign)
            processAttributionModel(smartItem?.attributionModel)
            processNotNull(
                smartItem?.counterId,
                CampaignWithMetrikaCounters.METRIKA_COUNTERS
            ) { listOf(it.toLong()) }
            processDailyBudget(item)
            processJaxbElement(item.negativeKeywords, SmartCampaign.MINUS_KEYWORDS) { it!!.items }
            processExcludedSites(item, knownSsps)
            processCheckPositionInterval(item)
            processPriorityGoals(smartItem?.priorityGoals)
            processJaxbElement(smartItem?.strategyId, CampaignWithPackageStrategy.STRATEGY_ID)
            processJaxbElement(smartItem?.trackingParams, CampaignWithBannerHrefParams.BANNER_HREF_PARAMS)

            val advancedGeoTargeting = featureService.isEnabledForClientId(
                uidAndClientId.clientId,
                FeatureName.ADVANCED_GEOTARGETING
            )
            if (advancedGeoTargeting) {
                processHasExtendedGeoTargeting(settings[SmartCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING])
                processUseCurrentRegion(settings[SmartCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING])
                processUseRegularRegion(settings[SmartCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING])
            }
            val apiStrategy = smartItem?.biddingStrategy
            val newStrategy = apiStrategy?.let {
                val fullApiStrategy = mergeOldAndNewStrategies(it, campaignFromDb as SmartCampaign)
                SmartCampaignStrategyConverter.toCampaignStrategy(fullApiStrategy)
            }

            processStrategy(newStrategy)
            processContextLimit(apiStrategy?.network?.toContextLimit())

            // enableCpcHold проставится в ядре, если стратегия - автобюджетная
            // https://a.yandex-team.ru/arc_vcs/direct/core/src/main/java/ru/yandex/direct/core/entity/campaign/service/type/update/CampaignWithNetworkSettingsUpdateOperationSupport.java?rev=r7408600#L42
            processFavoriteForIds(uidAndClientId.uid, settings[SmartCampaignSettingsEnum.ADD_TO_FAVORITES])
        }

        return UpdateCampaignsConvertedRequest(
            modelChanges,
            requireServicing = settings[SmartCampaignSettingsEnum.REQUIRE_SERVICING],
        )
    }

    companion object {
        /**
         * Мёржит apiStrategy с данными стратегии из oldCampaign, при условии, что biddingStrategyType одинаковыйи
         * возвращает объединённую стратегию
         */
        fun mergeOldAndNewStrategies(
            apiStrategy: SmartCampaignStrategy?,
            campaignFromDb: SmartCampaign
        ): SmartCampaignStrategy {
            val result = SmartCampaignStrategy().apply {
                if (apiStrategy != null) {
                    BeanUtils.copyProperties(this, apiStrategy)
                }
            }
            val oldStrategy = toSmartCampaignExternalStrategy(campaignFromDb.strategy, campaignFromDb.contextLimit)
            if (result.search == null) {
                result.search = oldStrategy.search
            } else if (result.search.biddingStrategyType == oldStrategy.search.biddingStrategyType) {
                copyProperties(result.search, oldStrategy.search)
                result.search = oldStrategy.search
            }
            if (result.network == null) {
                result.network = oldStrategy.network
            } else if (result.network.biddingStrategyType == oldStrategy.network.biddingStrategyType) {
                copyProperties(result.network, oldStrategy.network)
                result.network = oldStrategy.network
            }
            return result
        }

        @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
        private fun copyProperties(source: SmartCampaignSearchStrategy, dest: SmartCampaignSearchStrategy) {
            check(source.biddingStrategyType == dest.biddingStrategyType)
            when (source.biddingStrategyType) {
                SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPC_PER_CAMPAIGN ->
                    copyPropertiesExceptNull(source.averageCpcPerCampaign, dest.averageCpcPerCampaign)
                SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPC_PER_FILTER ->
                    copyPropertiesExceptNull(source.averageCpcPerFilter, dest.averageCpcPerFilter)
                SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPA_PER_CAMPAIGN ->
                    copyPropertiesExceptNull(source.averageCpaPerCampaign, dest.averageCpaPerCampaign)
                SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPA_PER_FILTER ->
                    copyPropertiesExceptNull(source.averageCpaPerFilter, dest.averageCpaPerFilter)
                SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_PER_CAMPAIGN ->
                    copyPropertiesExceptNull(source.payForConversionPerCampaign, dest.payForConversionPerCampaign)
                SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_PER_FILTER ->
                    copyPropertiesExceptNull(source.payForConversionPerFilter, dest.payForConversionPerFilter)
                SmartCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE ->
                    copyPropertiesExceptNull(source.wbMaximumConversionRate, dest.wbMaximumConversionRate)
                SmartCampaignSearchStrategyTypeEnum.AVERAGE_ROI ->
                    copyPropertiesExceptNull(source.averageRoi, dest.averageRoi)
                SmartCampaignSearchStrategyTypeEnum.AVERAGE_CRR ->
                    copyPropertiesExceptNull(source.averageCrr, dest.averageCrr)
                SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR ->
                    copyPropertiesExceptNull(source.payForConversionCrr, dest.payForConversionCrr)
                SmartCampaignSearchStrategyTypeEnum.SERVING_OFF -> {
                    // no properties
                }
                SmartCampaignSearchStrategyTypeEnum.UNKNOWN -> {
                    // not used
                }
            }
        }

        @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
        private fun copyProperties(source: SmartCampaignNetworkStrategy, dest: SmartCampaignNetworkStrategy) {
            check(source.biddingStrategyType == dest.biddingStrategyType)
            when (source.biddingStrategyType) {
                SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPC_PER_CAMPAIGN ->
                    copyPropertiesExceptNull(source.averageCpcPerCampaign, dest.averageCpcPerCampaign)
                SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPC_PER_FILTER ->
                    copyPropertiesExceptNull(source.averageCpcPerFilter, dest.averageCpcPerFilter)
                SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPA_PER_CAMPAIGN ->
                    copyPropertiesExceptNull(source.averageCpaPerCampaign, dest.averageCpaPerCampaign)
                SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPA_PER_FILTER ->
                    copyPropertiesExceptNull(source.averageCpaPerFilter, dest.averageCpaPerFilter)
                SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_PER_CAMPAIGN ->
                    copyPropertiesExceptNull(source.payForConversionPerCampaign, dest.payForConversionPerCampaign)
                SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_PER_FILTER ->
                    copyPropertiesExceptNull(source.payForConversionPerFilter, dest.payForConversionPerFilter)
                SmartCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE ->
                    copyPropertiesExceptNull(source.wbMaximumConversionRate, dest.wbMaximumConversionRate)
                SmartCampaignNetworkStrategyTypeEnum.AVERAGE_ROI ->
                    copyPropertiesExceptNull(source.averageRoi, dest.averageRoi)
                SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CRR ->
                    copyPropertiesExceptNull(source.averageCrr, dest.averageCrr)
                SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_CRR ->
                    copyPropertiesExceptNull(source.payForConversionCrr, dest.payForConversionCrr)
                SmartCampaignNetworkStrategyTypeEnum.SERVING_OFF -> {
                    // no properties
                }
                SmartCampaignNetworkStrategyTypeEnum.UNKNOWN -> {
                    // not used
                }
                SmartCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT -> {
                    copyPropertiesExceptNull(source.networkDefault, dest.networkDefault)
                }
            }
        }
    }
}
