package ru.yandex.direct.api.v5.entity.campaigns.converter;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.StrategyAverageCpaAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCpaPerCampaignAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCpaPerFilterAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCpcAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCpcPerCampaignAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCpcPerFilterAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCpiAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageCrrAdd;
import com.yandex.direct.api.v5.campaigns.StrategyAverageRoiAdd;
import com.yandex.direct.api.v5.campaigns.StrategyCpAverageCpvAdd;
import com.yandex.direct.api.v5.campaigns.StrategyCpDecreasedPriceForRepeatedImpressionsAdd;
import com.yandex.direct.api.v5.campaigns.StrategyCpMaximumImpressionsAdd;
import com.yandex.direct.api.v5.campaigns.StrategyMaximumAppInstallsAdd;
import com.yandex.direct.api.v5.campaigns.StrategyMaximumClicksAdd;
import com.yandex.direct.api.v5.campaigns.StrategyMaximumConversionRateAdd;
import com.yandex.direct.api.v5.campaigns.StrategyPayForConversionAdd;
import com.yandex.direct.api.v5.campaigns.StrategyPayForConversionCrrAdd;
import com.yandex.direct.api.v5.campaigns.StrategyPayForConversionPerCampaignAdd;
import com.yandex.direct.api.v5.campaigns.StrategyPayForConversionPerFilterAdd;
import com.yandex.direct.api.v5.campaigns.StrategyPayForInstallAdd;
import com.yandex.direct.api.v5.campaigns.StrategyWbAverageCpvAdd;
import com.yandex.direct.api.v5.campaigns.StrategyWbDecreasedPriceForRepeatedImpressionsAdd;
import com.yandex.direct.api.v5.campaigns.StrategyWbMaximumImpressionsAdd;
import com.yandex.direct.api.v5.campaigns.StrategyWeeklyClickPackageAdd;

import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertFromMicrosToDecimal;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertYesNoEnumToLong;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.toDate;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyConverterConstants.DEFAULT_STRATEGY_NAME;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyConverterConstants.FORMAT_VERSION;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.AUTOBUDGET_STRATEGIES;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CPI_GOAL_ID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
public class StrategyAddConverter {

    private StrategyAddConverter() {
        // no instantiation
    }

    public static CampaignsAutobudget toCampaignsAutobudget(StrategyName strategyName) {
        boolean isAutobudget = AUTOBUDGET_STRATEGIES.contains(strategyName);
        return isAutobudget ? CampaignsAutobudget.YES : CampaignsAutobudget.NO;
    }

    public static StrategyData defaultStrategyData() {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(DEFAULT_STRATEGY_NAME);
    }

    @Nullable
    public static CampOptionsStrategy toCampOptionsStrategy(StrategyName strategyName) {
        switch (strategyName) {
            case AUTOBUDGET_AVG_CPC_PER_CAMP:
                return CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_CAMP;
            case AUTOBUDGET_AVG_CPC_PER_FILTER:
                return CampOptionsStrategy.AUTOBUDGET_AVG_CPC_PER_FILTER;
            case AUTOBUDGET_AVG_CPA_PER_CAMP:
                return CampOptionsStrategy.AUTOBUDGET_AVG_CPA_PER_CAMP;
            case AUTOBUDGET_AVG_CPA_PER_FILTER:
                return CampOptionsStrategy.AUTOBUDGET_AVG_CPA_PER_FILTER;
            default:
                return null;
        }
    }

    public static StrategyData fromAverageCpa(StrategyAverageCpaAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPA.name().toLowerCase())
                .withPayForConversion(false)
                .withAvgCpa(convertToDbPrice(settingsStructure.getAverageCpa()))
                .withGoalId(settingsStructure.getGoalId())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()));
    }

    public static StrategyData fromAverageCpc(StrategyAverageCpcAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CLICK.name().toLowerCase())
                .withAvgBid(convertToDbPrice(settingsStructure.getAverageCpc()))
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()));
    }

    public static StrategyData fromAverageCrr(StrategyAverageCrrAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_CRR.name().toLowerCase())
                .withCrr(Long.valueOf(settingsStructure.getCrr()))
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withGoalId(settingsStructure.getGoalId());
        // payForConversion выставляется в ядре
    }

    public static StrategyData fromAverageCpi(StrategyAverageCpiAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPI.name().toLowerCase())
                .withAvgCpi(convertToDbPrice(settingsStructure.getAverageCpi()))
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()))
                .withPayForConversion(false);
    }

    public static StrategyData fromAverageRoi(StrategyAverageRoiAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_ROI.name().toLowerCase())
                .withReserveReturn(ifNotNull(settingsStructure.getReserveReturn(), Integer::longValue))
                .withRoiCoef(convertFromMicrosToDecimal(settingsStructure.getRoiCoef()))
                .withGoalId(settingsStructure.getGoalId())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()))
                .withProfitability(convertFromMicrosToDecimal(settingsStructure.getProfitability()));
    }

    public static StrategyData fromPayForConversion(StrategyPayForConversionAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPA.name().toLowerCase())
                .withPayForConversion(true)
                .withAvgCpa(convertToDbPrice(settingsStructure.getCpa()))
                .withGoalId(settingsStructure.getGoalId())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()));
    }

    public static StrategyData fromPayForConversionCrr(StrategyPayForConversionCrrAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_CRR.name().toLowerCase())
                .withPayForConversion(true)
                .withCrr(Long.valueOf(settingsStructure.getCrr()))
                .withGoalId(settingsStructure.getGoalId())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()));
    }

    public static StrategyData fromWbMaximumClicks(StrategyMaximumClicksAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET.name().toLowerCase())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()));
    }

    public static StrategyData fromWbMaximumConversionRate(StrategyMaximumConversionRateAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET.name().toLowerCase())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()))
                .withGoalId(settingsStructure.getGoalId());
    }

    public static StrategyData fromWbMaximumAppInstalls(StrategyMaximumAppInstallsAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET.name().toLowerCase())
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()))
                .withGoalId(DEFAULT_CPI_GOAL_ID);
    }

    public static StrategyData fromWeeklyClickPackage(StrategyWeeklyClickPackageAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_WEEK_BUNDLE.name().toLowerCase())
                .withLimitClicks(settingsStructure.getClicksPerWeek())
                .withAvgBid(convertToDbPrice(settingsStructure.getAverageCpc()))
                .withBid(convertToDbPrice(settingsStructure.getBidCeiling()));
    }

    public static StrategyData fromPayForInstall(StrategyPayForInstallAdd settingsStructure) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPI.name().toLowerCase())
                .withPayForConversion(true)
                .withAvgCpi(convertToDbPrice(settingsStructure.getAverageCpi()))
                .withSum(convertToDbPrice(settingsStructure.getWeeklySpendLimit()));
    }

    public static StrategyData fromManualCpm() {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.CPM_DEFAULT.name().toLowerCase());
    }

    // https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Direct/Strategy/AutobudgetMaxReach.pm
    public static StrategyData fromWbDecreasedPriceForRepeatedImpressions(
            StrategyWbDecreasedPriceForRepeatedImpressionsAdd wbDecreasedPriceForRepeatedImpressions) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_MAX_REACH.name().toLowerCase())
                .withAvgCpm(convertToDbPrice(wbDecreasedPriceForRepeatedImpressions.getAverageCpm()))
                .withSum(convertToDbPrice(wbDecreasedPriceForRepeatedImpressions.getSpendLimit()));
    }

    public static StrategyData fromCpDecreasedPriceForRepeatedImpressions(
            StrategyCpDecreasedPriceForRepeatedImpressionsAdd cpDecreasedPriceForRepeatedImpressions) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD.name().toLowerCase())
                .withStart(toDate(cpDecreasedPriceForRepeatedImpressions.getStartDate()))
                .withFinish(toDate(cpDecreasedPriceForRepeatedImpressions.getEndDate()))
                .withAutoProlongation(convertYesNoEnumToLong(
                        cpDecreasedPriceForRepeatedImpressions.getAutoContinue()))
                .withBudget(convertToDbPrice(cpDecreasedPriceForRepeatedImpressions.getSpendLimit()))
                .withAvgCpm(convertToDbPrice(cpDecreasedPriceForRepeatedImpressions.getAverageCpm()));
        // dailyChangeCount - проставлять не нужно, т.к. системное поле
    }

    public static StrategyData fromWbMaximumImpressions(StrategyWbMaximumImpressionsAdd wbMaximumImpressions) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_MAX_IMPRESSIONS.name().toLowerCase())
                .withAvgCpm(convertToDbPrice(wbMaximumImpressions.getAverageCpm()))
                .withSum(convertToDbPrice(wbMaximumImpressions.getSpendLimit()));
    }

    public static StrategyData fromCpMaximumImpressions(StrategyCpMaximumImpressionsAdd cpMaximumImpressions) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD.name().toLowerCase())
                .withStart(toDate(cpMaximumImpressions.getStartDate()))
                .withFinish(toDate(cpMaximumImpressions.getEndDate()))
                .withBudget(convertToDbPrice(cpMaximumImpressions.getSpendLimit()))
                .withAvgCpm(convertToDbPrice(cpMaximumImpressions.getAverageCpm()))
                .withAutoProlongation(convertYesNoEnumToLong(cpMaximumImpressions.getAutoContinue()));
    }

    public static StrategyData fromWbAverageCpv(StrategyWbAverageCpvAdd wbAverageCpv) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPV.name().toLowerCase())
                .withSum(convertToDbPrice(wbAverageCpv.getSpendLimit()))
                .withAvgCpv(convertToDbPrice(wbAverageCpv.getAverageCpv()))
                .withPayForConversion(true);
    }

    public static StrategyData fromCpAverageCpv(StrategyCpAverageCpvAdd cpAverageCpv) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD.name().toLowerCase())
                .withStart(toDate(cpAverageCpv.getStartDate()))
                .withFinish(toDate(cpAverageCpv.getEndDate()))
                .withBudget(convertToDbPrice(cpAverageCpv.getSpendLimit()))
                .withAvgCpv(convertToDbPrice(cpAverageCpv.getAverageCpv()))
                .withAutoProlongation(convertYesNoEnumToLong(cpAverageCpv.getAutoContinue()))
                .withPayForConversion(true);
    }

    public static StrategyData fromAverageCpcPerCampaign(StrategyAverageCpcPerCampaignAdd averageCpcPerCampaign) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP.name().toLowerCase())
                .withAvgBid(convertToDbPrice(averageCpcPerCampaign.getAverageCpc()))
                .withBid(convertToDbPrice(averageCpcPerCampaign.getBidCeiling()))
                .withSum(convertToDbPrice(averageCpcPerCampaign.getWeeklySpendLimit()));
    }

    public static StrategyData fromAverageCpcPerFilter(StrategyAverageCpcPerFilterAdd averageCpcPerFilter) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER.name().toLowerCase())
                .withFilterAvgBid(convertToDbPrice(averageCpcPerFilter.getFilterAverageCpc()))
                .withBid(convertToDbPrice(averageCpcPerFilter.getBidCeiling()))
                .withSum(convertToDbPrice(averageCpcPerFilter.getWeeklySpendLimit()));
    }

    public static StrategyData fromAverageCpaPerCampaign(StrategyAverageCpaPerCampaignAdd averageCpaPerCampaign) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP.name().toLowerCase())
                .withAvgCpa(convertToDbPrice(averageCpaPerCampaign.getAverageCpa()))
                .withGoalId(averageCpaPerCampaign.getGoalId())
                .withPayForConversion(false)
                .withBid(convertToDbPrice(averageCpaPerCampaign.getBidCeiling()))
                .withSum(convertToDbPrice(averageCpaPerCampaign.getWeeklySpendLimit()));
    }

    public static StrategyData fromAverageCpaPerFilter(StrategyAverageCpaPerFilterAdd averageCpaPerFilter) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER.name().toLowerCase())
                .withFilterAvgCpa(convertToDbPrice(averageCpaPerFilter.getFilterAverageCpa()))
                .withGoalId(averageCpaPerFilter.getGoalId())
                .withPayForConversion(false)
                .withBid(convertToDbPrice(averageCpaPerFilter.getBidCeiling()))
                .withSum(convertToDbPrice(averageCpaPerFilter.getWeeklySpendLimit()));
    }

    public static StrategyData fromPayForConversionPerCampaign(
            StrategyPayForConversionPerCampaignAdd payForConversionPerCampaign) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP.name().toLowerCase())
                .withAvgCpa(convertToDbPrice(payForConversionPerCampaign.getCpa()))
                .withGoalId(payForConversionPerCampaign.getGoalId())
                .withPayForConversion(true)
                .withSum(convertToDbPrice(payForConversionPerCampaign.getWeeklySpendLimit()));
    }

    public static StrategyData fromPayForConversionPerFilter(
            StrategyPayForConversionPerFilterAdd payForConversionPerFilter) {
        return new StrategyData()
                .withVersion(FORMAT_VERSION)
                .withName(StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER.name().toLowerCase())
                .withFilterAvgCpa(convertToDbPrice(payForConversionPerFilter.getCpa()))
                .withGoalId(payForConversionPerFilter.getGoalId())
                .withPayForConversion(true)
                .withSum(convertToDbPrice(payForConversionPerFilter.getWeeklySpendLimit()));
    }

}
