package ru.yandex.direct.api.v5.entity.campaigns.converter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.CampaignAddItem;
import com.yandex.direct.api.v5.campaigns.TextCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.TextCampaignSetting;
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingsEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsApiValidationSignals;
import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractBroadMatchFlag;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractBroadMatchGoalId;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractBroadMatchLimit;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractCheckPositionIntervalEvent;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDayBudget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDayBudgetShowMode;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDisabledDomains;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDisabledIps;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractDisabledSsp;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractEmail;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractEnableCheckPositionEvent;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractMeaningfulGoals;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractMinusKeywords;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractSendAccountNews;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractSmsFlags;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractSmsTimeInterval;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractTimeTarget;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.extractWarningBalance;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.getEnableCpcHold;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.toAttributionModel;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.toDate;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.TextCampaignStrategyConverter.toCampaignStrategy;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ADD_METRIKA_TAG_TO_URL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ADD_OPENSTAT_TAG_TO_URL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ADD_TO_FAVORITES;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ENABLE_COMPANY_INFO;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ENABLE_CPC_HOLD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_EXCLUDE_PAUSED_COMPETING_ADS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_HAS_EXTENDED_GEO_TARGETING;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_HAS_SITE_MONITORING;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_HAS_TITLE_SUBSTITUTION;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class TextCampaignAddItemConverter {

    // CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED (IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED в модели) - должно быть null,
    // если у пользователя нет фичи, смотри CampaignWithOrderPhraseLengthPrecedenceForbiddenValidator, поэтому  дефолт
    // не проставляем, чтобы не перезатиреть null
    // ENABLE_RELATED_KEYWORDS, ENABLE_AUTOFOCUS, ENABLE_BEHAVIORAL_TARGETING - deprecated
    private static final Map<TextCampaignSettingsEnum, Boolean> DEFAULT_SETTINGS = Map.of(
            TextCampaignSettingsEnum.ADD_METRICA_TAG, DEFAULT_ADD_METRIKA_TAG_TO_URL,
            TextCampaignSettingsEnum.ADD_OPENSTAT_TAG, DEFAULT_ADD_OPENSTAT_TAG_TO_URL,
            TextCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING, DEFAULT_HAS_EXTENDED_GEO_TARGETING,
            TextCampaignSettingsEnum.ENABLE_COMPANY_INFO, DEFAULT_ENABLE_COMPANY_INFO,
            TextCampaignSettingsEnum.ENABLE_EXTENDED_AD_TITLE, DEFAULT_HAS_TITLE_SUBSTITUTION,
            TextCampaignSettingsEnum.ENABLE_SITE_MONITORING, DEFAULT_HAS_SITE_MONITORING,
            TextCampaignSettingsEnum.EXCLUDE_PAUSED_COMPETING_ADS, DEFAULT_EXCLUDE_PAUSED_COMPETING_ADS,
            TextCampaignSettingsEnum.MAINTAIN_NETWORK_CPC, DEFAULT_ENABLE_CPC_HOLD,
            TextCampaignSettingsEnum.ADD_TO_FAVORITES, DEFAULT_ADD_TO_FAVORITES
    );

    public static CampaignsApiValidationSignals.CampaignWrapper convertTextCampaign(
            CampaignAddItem item,
            Long chiefUid,
            String userEmail,
            Map<String, GeoTimezone> timeZones,
            CampaignSource campaignSource,
            Set<String> knownSsps, boolean advancedGeoTargeting) {
        TextCampaignAddItem textCampaignItem = item.getTextCampaign();

        Map<TextCampaignSettingsEnum, Boolean> settings = convertTextCampaignSettings(
                textCampaignItem.getSettings(), advancedGeoTargeting
        );
        Set<Long> favoriteForUids = settings.get(TextCampaignSettingsEnum.ADD_TO_FAVORITES) ? Set.of(chiefUid) : null;

        DbStrategy strategy = toCampaignStrategy(textCampaignItem.getBiddingStrategy());
        var campaign = new TextCampaign()
                .withSource(campaignSource)
                .withType(CampaignType.TEXT)
                .withName(item.getName())
                .withStrategy(strategy)

                /*
                На самом деле это значение не всегда применится, в ядре есть такой код:
                if (isAutoBudget || isDifferentPlaces) {
                  campaign.setContextLimit(AUTO_CONTEXT_LIMIT);
                }
                Здесь проставляем всегда, потому что знаем, что ядро когда надо - перезатрёт.
                null ядро заменит на 0, покрывается авто-тестом StrategyDefaultBidsManualTest
                */
                .withContextLimit(CampaignConversionUtilsKt.toContextLimit(
                        textCampaignItem.getBiddingStrategy().getNetwork()))

                .withHasAddMetrikaTagToUrl(settings.get(TextCampaignSettingsEnum.ADD_METRICA_TAG))
                .withHasAddOpenstatTagToUrl(settings.get(TextCampaignSettingsEnum.ADD_OPENSTAT_TAG))
                .withHasExtendedGeoTargeting(settings.get(TextCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING))
                .withUseCurrentRegion(settings.get(TextCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING))
                .withUseRegularRegion(settings.get(TextCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING))
                .withEnableCompanyInfo(settings.get(TextCampaignSettingsEnum.ENABLE_COMPANY_INFO))
                .withHasTitleSubstitution(settings.get(TextCampaignSettingsEnum.ENABLE_EXTENDED_AD_TITLE))
                .withHasSiteMonitoring(settings.get(TextCampaignSettingsEnum.ENABLE_SITE_MONITORING))
                .withExcludePausedCompetingAds(settings.get(TextCampaignSettingsEnum.EXCLUDE_PAUSED_COMPETING_ADS))
                .withEnableCpcHold(getEnableCpcHold(strategy.isAutoBudget(),
                        settings.get(TextCampaignSettingsEnum.MAINTAIN_NETWORK_CPC)))
                .withIsOrderPhraseLengthPrecedenceEnabled(
                        settings.get(TextCampaignSettingsEnum.CAMPAIGN_EXACT_PHRASE_MATCHING_ENABLED))
                .withFavoriteForUids(favoriteForUids)
                .withStartDate(toDate(item.getStartDate()))
                .withEndDate(toDate(item.getEndDate()))
                .withEmail(userEmail)
                .withSmsTime(extractSmsTimeInterval(item))
                .withSmsFlags(extractSmsFlags(item))
                .withDayBudget(extractDayBudget(item))
                .withDayBudgetShowMode(extractDayBudgetShowMode(item))
                .withBroadMatch(new BroadMatch()
                        .withBroadMatchFlag(extractBroadMatchFlag(textCampaignItem))
                        .withBroadMatchLimit(extractBroadMatchLimit(textCampaignItem))
                        .withBroadMatchGoalId(extractBroadMatchGoalId(textCampaignItem)))
                .withTimeZoneId(item.getTimeZone() == null ? 0 : timeZones.get(item.getTimeZone()).getTimezoneId())
                .withTimeTarget(extractTimeTarget(item))
                .withMetrikaCounters(CampaignsAddRequestConverter.toMetrikaCounters(textCampaignItem.getCounterIds()))
                .withFio(item.getClientInfo())
                .withMinusKeywords(extractMinusKeywords(item))
                .withEmail(nvl(extractEmail(item.getNotification()), userEmail))
                .withWarningBalance(extractWarningBalance(item.getNotification()))
                .withCheckPositionIntervalEvent(extractCheckPositionIntervalEvent(item.getNotification()))
                .withEnableSendAccountNews(extractSendAccountNews(item))
                .withEnableCheckPositionEvent(extractEnableCheckPositionEvent(item.getNotification()))
                .withDisabledIps(extractDisabledIps(item))
                .withDisabledSsp(extractDisabledSsp(item, knownSsps))
                .withDisabledDomains(extractDisabledDomains(item, knownSsps))
                .withPlacementTypes(emptySet())
                .withAttributionModel(toAttributionModel(textCampaignItem.getAttributionModel()))
                .withMeaningfulGoals(extractMeaningfulGoals(textCampaignItem))
                .withStrategyId(textCampaignItem.getStrategyId())
                .withBannerHrefParams(textCampaignItem.getTrackingParams());

        return new CampaignsApiValidationSignals.CampaignWrapper()
                .withRealCampaign(campaign)
                .withRequireServicing(settings.get(TextCampaignSettingsEnum.REQUIRE_SERVICING));
    }

    private static Map<TextCampaignSettingsEnum, Boolean> convertTextCampaignSettings(
            List<TextCampaignSetting> settings, boolean advancedGeoTargeting) {
        Map<TextCampaignSettingsEnum, Boolean> valueByOption = new HashMap<>();
        for (TextCampaignSetting s : settings) {
            valueByOption.put(s.getOption(), s.getValue() == YesNoEnum.YES);
        }
        if (advancedGeoTargeting) {
            if (!valueByOption.containsKey(TextCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING)) {
                valueByOption.put(TextCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING, true);
            }
            if (!valueByOption.containsKey(TextCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING)) {
                valueByOption.put(TextCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING, true);
            }
        }
        return StreamEx.of(TextCampaignSettingsEnum.values())
                .mapToEntry(option -> valueByOption.getOrDefault(option, DEFAULT_SETTINGS.get(option)))
                .nonNullValues()
                .toMap();
    }

}
