package ru.yandex.direct.api.v5.entity.campaigns.converter

import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.TextCampaignStrategy
import org.slf4j.LoggerFactory
import ru.yandex.direct.api.v5.entity.campaigns.converter.StrategyConverterConstants.DEFAULT_STRATEGY_NAME
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import java.lang.invoke.MethodHandles

private val logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass())

fun createTextCampaignGetStrategy(campaign: TextCampaign): TextCampaignStrategy {
    val biddingStrategy = toTextCampaignExternalStrategy(campaign.strategy, campaign.contextLimit)

    return biddingStrategy.apply {
        if (search.biddingStrategyType == null) {
            search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.UNKNOWN
            logger.warn("Campaign ${campaign.id} has unknown strategy ${campaign.strategy}")
        }

        if (network.biddingStrategyType == null) {
            network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.UNKNOWN
            logger.warn("Campaign ${campaign.id} has unknown strategy ${campaign.strategy}")
        }

        // при автобюджетной стратегии в сетях NetworkDefault не заполняется
        // https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns.pm?rev=r9289187#L344-353
        if (campaign.strategy.strategyName.name.startsWith("AUTOBUDGET")) {
            biddingStrategy.network.networkDefault = null
        }
    }
}

fun toTextCampaignExternalStrategy(
    internalStrategy: DbStrategy,
    contextLimit: Int?
) =
    TextCampaignStrategy().apply {
        search = getSearchStrategy(internalStrategy)
        network = getNetworkStrategy(internalStrategy, contextLimit)
    }

private fun getSearchStrategy(internalStrategy: DbStrategy): TextCampaignSearchStrategy {
    val search = TextCampaignSearchStrategy()
    if (!internalStrategy.platform.isSearch()) {
        search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.SERVING_OFF
    } else {
        val strategyData = internalStrategy.strategyData
        when (strategyData.name) {
            DEFAULT_STRATEGY_NAME -> {
                search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.HIGHEST_POSITION
            }
            StrategyName.AUTOBUDGET_AVG_CPA.name.lowercase() -> {
                if (strategyData.payForConversion == true) {
                    search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION
                    search.payForConversion = toStrategyPayForConversion(strategyData)
                } else {
                    search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.AVERAGE_CPA
                    search.averageCpa = toStrategyAverageCpa(strategyData)
                }
            }
            StrategyName.AUTOBUDGET_AVG_CLICK.name.lowercase() -> {
                search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.AVERAGE_CPC
                search.averageCpc = toStrategyAverageCpc(strategyData)
            }
            StrategyName.AUTOBUDGET_CRR.name.lowercase() -> {
                if (strategyData.payForConversion == true) {
                    search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR
                    search.payForConversionCrr = toStrategyPayForConversionCrr(strategyData)
                } else {
                    search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.AVERAGE_CRR
                    search.averageCrr = toStrategyAverageCrr(strategyData)
                }
            }
            StrategyName.AUTOBUDGET_ROI.name.lowercase() -> {
                search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.AVERAGE_ROI
                search.averageRoi = toStrategyAverageRoi(strategyData)
            }
            StrategyName.AUTOBUDGET.name.lowercase() -> {
                when (strategyData.goalId) {
                    null -> {
                        search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS
                        search.wbMaximumClicks = toStrategyMaximumClicks(strategyData)
                    }
                    else -> {
                        search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE
                        search.wbMaximumConversionRate = toStrategyMaximumConversionRate(strategyData)
                    }
                }
            }
            StrategyName.AUTOBUDGET_WEEK_BUNDLE.name.lowercase() -> {
                search.biddingStrategyType = TextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
                search.weeklyClickPackage = toStrategyWeeklyClickPackage(strategyData)
            }
        }
    }
    return search
}

private fun getNetworkStrategy(
    internalStrategy: DbStrategy,
    contextLimit: Int?
): TextCampaignNetworkStrategy {
    val network = TextCampaignNetworkStrategy()
    val strategyData = internalStrategy.strategyData

    when (internalStrategy.platform) {
        CampaignsPlatform.SEARCH -> {
            network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.SERVING_OFF
        }
        CampaignsPlatform.BOTH -> {
            if (internalStrategy.strategy == null) {
                network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT
                network.networkDefault = toStrategyNetworkDefault(contextLimit)
            } else {
                network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE
            }
        }
        else -> when (strategyData.name) {
            DEFAULT_STRATEGY_NAME -> {
                network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.MAXIMUM_COVERAGE
            }
            StrategyName.AUTOBUDGET_AVG_CPA.name.lowercase() -> {
                if (strategyData.payForConversion == true) {
                    network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION
                    network.payForConversion = toStrategyPayForConversion(strategyData)
                } else {
                    network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPA
                    network.averageCpa = toStrategyAverageCpa(strategyData)
                }
            }
            StrategyName.AUTOBUDGET_AVG_CLICK.name.lowercase() -> {
                network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPC
                network.averageCpc = toStrategyAverageCpc(strategyData)
            }
            StrategyName.AUTOBUDGET_CRR.name.lowercase() -> {
                if (strategyData.payForConversion == true) {
                    network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_CRR
                    network.payForConversionCrr = toStrategyPayForConversionCrr(strategyData)
                } else {
                    network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.AVERAGE_CRR
                    network.averageCrr = toStrategyAverageCrr(strategyData)
                }
            }
            StrategyName.AUTOBUDGET_ROI.name.lowercase() -> {
                network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.AVERAGE_ROI
                network.averageRoi = toStrategyAverageRoi(strategyData)
            }
            StrategyName.AUTOBUDGET.name.lowercase() -> {
                when (strategyData.goalId) {
                    null -> {
                        network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS
                        network.wbMaximumClicks = toStrategyMaximumClicks(strategyData)
                    }
                    else -> {
                        network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE
                        network.wbMaximumConversionRate = toStrategyMaximumConversionRate(strategyData)
                    }
                }
            }
            StrategyName.AUTOBUDGET_WEEK_BUNDLE.name.lowercase() -> {
                network.biddingStrategyType = TextCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
                network.weeklyClickPackage = toStrategyWeeklyClickPackage(strategyData)
            }
        }
    }
    return network
}
