package ru.yandex.direct.api.v5.entity.campaigns.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.api.v5.entity.campaigns.container.CampaignAnyFieldEnum
import ru.yandex.direct.api.v5.entity.campaigns.container.GetCampaignsContainer
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesCampaignService
import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer
import ru.yandex.direct.core.entity.campaign.container.ApiCampaignsSelectionCriteria
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAttributionModel
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBannerHrefParams
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCheckPositionEvent
import ru.yandex.direct.core.entity.campaign.model.CampaignWithClicks
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCurrencyConverted
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisabledDomainsAndSsp
import ru.yandex.direct.core.entity.campaign.model.CampaignWithEnableCompanyInfo
import ru.yandex.direct.core.entity.campaign.model.CampaignWithEshowsSettings
import ru.yandex.direct.core.entity.campaign.model.CampaignWithExcludePausedCompetingAds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithFavorite
import ru.yandex.direct.core.entity.campaign.model.CampaignWithImpressionRate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMinusKeywords
import ru.yandex.direct.core.entity.campaign.model.CampaignWithNetworkSettings
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalAddMetrikaTagToUrl
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalAddOpenstatTagToUrl
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOrderPhraseLengthPrecedence
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPlacementTypes
import ru.yandex.direct.core.entity.campaign.model.CampaignWithShows
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSiteMonitoring
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSourceId
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStopTime
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campoperationqueue.CampOperationQueueRepository
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientNdsService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository
import ru.yandex.direct.core.entity.timetarget.service.GeoTimezoneMappingService
import ru.yandex.direct.core.entity.user.repository.UserRepository
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.rbac.RbacService
import java.math.BigDecimal
import kotlin.reflect.KClass

@Service
class CampaignDataFetcher(
    private val affectedCampaignIdsContainer: AffectedCampaignIdsContainer,
    private val aggregatedStatusesCampaignService: AggregatedStatusesCampaignService,
    private val auth: ApiAuthenticationSource,
    private val campOperationQueueRepository: CampOperationQueueRepository,
    private val campaignRepository: CampaignRepository,
    private val campaignSumAvailableForTransferCalculator: CampaignSumAvailableForTransferCalculator,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val clientNdsService: ClientNdsService,
    private val clientService: ClientService,
    private val featureService: FeatureService,
    private val geoTimezoneRepository: GeoTimezoneRepository,
    private val rbacService: RbacService,
    private val shardHelper: ShardHelper,
    private val userRepository: UserRepository,
    private val userService: UserService,
) {
    fun getData(
        selectionCriteria: ApiCampaignsSelectionCriteria,
        requestedFields: Set<CampaignAnyFieldEnum>,
        limitOffset: LimitOffset,
    ): List<GetCampaignsContainer> {
        val shard = shardHelper.getShardByClientId(auth.chiefSubclient.clientId)
        val campaigns = loadCampaignModels(
            shard = shard,
            selectionCriteria = selectionCriteria,
            requestedFields = requestedFields,
            limitOffset = limitOffset,
        )
        val campaignIds = campaigns.map { it.id }
        val walletIds: Set<Long> = campaigns
            .map { it.walletId }
            .filterTo(HashSet()) { (it ?: 0L) != 0L }

        // данные загружаются лениво (при первом использовании)
        // так избегаются лишние запросы, если пользователь запросил не все поля

        // клиенты, агентства, менеджеры, связанные с кампаниями
        val clientsData by lazy {
            loadClients(
                campaigns,
                loadAgencies = CampaignAnyFieldEnum.REPRESENTED_BY in requestedFields,
                loadChiefSubclient = CampaignAnyFieldEnum.FUNDS in requestedFields,
            )
        }

        val ndsRatio by lazy {
            loadClientNdsRatio(clientsData.getValue(auth.chiefSubclient.clientId))
        }

        val sumsForTransfer by lazy {
            val clientCurrency = clientsData
                .getValue(auth.chiefSubclient.clientId)
                .workCurrency
                .currency
            calculateSumsAvailableForTransfer(campaigns, ndsRatio, clientCurrency)
        }

        val managerFios by lazy {
            val managerUids = campaigns.mapNotNullTo(HashSet()) { it.managerUid }
            userService.getUserFiosByUids(managerUids)
        }

        val geoTimezones by lazy {
            loadGeoTimezones(campaigns)
        }

        val aggregatedStatusWallets by lazy {
            if (walletIds.isNotEmpty()) {
                aggregatedStatusesCampaignService
                    .getWallets(shard, walletIds)
                    .associateBy { it.id }
            } else {
                emptyMap()
            }
        }

        val wallets by lazy {
            if (walletIds.isNotEmpty()) {
                campaignRepository
                    .getWalletsByWalletCampaignIds(shard, walletIds)
                    .associateBy { it.id }
            } else {
                emptyMap()
            }
        }

        val campaignsWithBanners: Set<Long> by lazy {
            campaignRepository.getCampaignIdsWithBanners(shard, campaignIds)
        }

        val queueOperations by lazy {
            campOperationQueueRepository.getCampaignOperationNames(shard, campaignIds)
        }

        val campaignsWithActiveBanners by lazy {
            campaignRepository.getCampaignsWithActiveBanners(shard, campaignIds)
        }

        val advancedGeoTargeting by lazy {
            featureService.isEnabled(auth.chiefSubclient.uid, FeatureName.ADVANCED_GEOTARGETING)
        }

        return campaigns.map { model ->
            GetCampaignsContainer(
                campaign = model,
                requestedFields = requestedFields,
                clientUid = auth.chiefSubclient.uid,
                ndsRatioSupplier = { ndsRatio },
                sumForTransferSupplier = { sumsForTransfer[model.id] },
                managerFioSupplier = { managerFios[model.managerUid] },
                agencyNameSupplier = { clientsData[ClientId.fromNullableLong(model.agencyId)]?.name },
                timezoneSupplier = {
                    geoTimezones[model.timeZoneId] ?: GeoTimezoneMappingService.DEFAULT_GEO_TIMEZONE_RUS
                },
                aggregatedStatusWalletSupplier = { aggregatedStatusWallets[model.walletId] },
                walletSupplier = { wallets[model.walletId] },
                queueOperationsSupplier = { queueOperations[model.id].orEmpty() },
                hasBannersSupplier = { model.id in campaignsWithBanners },
                hasActiveBannersSupplier = { model.id in campaignsWithActiveBanners },
                advancedGeoTargetingSupplier = { advancedGeoTargeting },
            )
        }
    }

    private fun loadCampaignModels(
        shard: Int,
        selectionCriteria: ApiCampaignsSelectionCriteria,
        requestedFields: Set<CampaignAnyFieldEnum>,
        limitOffset: LimitOffset,
    ): List<CommonCampaign> {
        val campaignClassesToLoad = requestedFields
            .flatMap { extractModelClassesToLoad(it) }
            .mapTo(mutableSetOf()) { it.java }

        val visibleIds = getVisibleCampaignIds(shard, selectionCriteria.ids)
        val selectionCriteriaWithIds = selectionCriteria.copy(ids = visibleIds)

        return campaignTypedRepository
            .getSafelyOrderedByCampaignId(
                shard,
                selectionCriteriaWithIds.toConditionFilter(),
                limitOffset,
                campaignClassesToLoad,
            )
            .let { @Suppress("UNCHECKED_CAST") (it as List<CommonCampaign>) }
            .also { affectedCampaignIdsContainer.ids += it.map(BaseCampaign::getId) }
    }

    /**
     * Возвращает ID кампаний из запроса, доступные текущему оператору
     * Если в запросе не был указан фильтр по ID - вернётся список всех кампаний.
     *
     * Аналог [функции на перле](https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns.pm?rev=r9289187#L454-463)
     */
    private fun getVisibleCampaignIds(shard: Int, requestedIds: Collection<Long>): Set<Long> {
        val ids = requestedIds.ifEmpty {
            userRepository.getCampaignIdsForUsers(shard, listOf(auth.chiefSubclient.uid))
        }

        // CampaignSubObjectAccessChecker не используется,
        // потому что не даёт прочитать кампании из Дзена
        return rbacService.getVisibleCampaigns(auth.operator.uid, ids)
    }

    /**
     * Возвращает интерфейсы кампании,
     * поля которых нужны для того, чтобы вернуть поле [field] из API
     *
     * [CommonCampaign] загружается всегда (для простоты)
     */
    fun extractModelClassesToLoad(
        field: CampaignAnyFieldEnum,
    ): Set<KClass<out BaseCampaign>> =
        setOf(CommonCampaign::class) + when (field) {
            // Явно прописаны кейсы, в которых не нужно загружать данные никаких тайп-саппортов
            // Так при добавлении нового поля не получится забыть обновить эту функцию
            CampaignAnyFieldEnum.BLOCKED_IPS -> emptySet()
            CampaignAnyFieldEnum.CLIENT_INFO -> emptySet()
            CampaignAnyFieldEnum.CURRENCY -> emptySet()
            CampaignAnyFieldEnum.DAILY_BUDGET -> setOf(CampaignWithDayBudget::class)
            CampaignAnyFieldEnum.END_DATE -> emptySet()
            CampaignAnyFieldEnum.EXCLUDED_SITES -> setOf(CampaignWithDisabledDomainsAndSsp::class)
            CampaignAnyFieldEnum.FUNDS -> setOf(CampaignWithStrategy::class, CampaignWithStopTime::class)
            CampaignAnyFieldEnum.ID -> emptySet()
            CampaignAnyFieldEnum.NAME -> emptySet()
            CampaignAnyFieldEnum.NEGATIVE_KEYWORDS -> setOf(CampaignWithMinusKeywords::class)
            CampaignAnyFieldEnum.NOTIFICATION -> setOf(CampaignWithCheckPositionEvent::class)
            CampaignAnyFieldEnum.REPRESENTED_BY -> emptySet()
            CampaignAnyFieldEnum.SOURCE_ID -> setOf(CampaignWithSourceId::class)
            CampaignAnyFieldEnum.START_DATE -> emptySet()
            CampaignAnyFieldEnum.STATE -> setOf(CampaignWithCurrencyConverted::class)
            CampaignAnyFieldEnum.STATISTICS -> setOf(CampaignWithShows::class, CampaignWithClicks::class)
            CampaignAnyFieldEnum.STATUS -> emptySet()
            CampaignAnyFieldEnum.STATUS_CLARIFICATION -> campaignClassesForStatusClarification
            CampaignAnyFieldEnum.STATUS_PAYMENT -> emptySet()
            CampaignAnyFieldEnum.TIME_TARGETING -> emptySet()
            CampaignAnyFieldEnum.TIME_ZONE -> emptySet()
            CampaignAnyFieldEnum.TYPE -> emptySet()

            CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_COUNTER_IDS -> setOf(CampaignWithMetrikaCounters::class)
            CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_SETTINGS -> campaignClassesForSettings
            CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_BIDDING_STRATEGY -> setOf(CampaignWithStrategy::class)
            CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_ATTRIBUTION_MODEL -> setOf(CampaignWithAttributionModel::class)

            CampaignAnyFieldEnum.CPM_BANNER_CAMPAIGN_COUNTER_IDS -> setOf(CampaignWithMetrikaCounters::class)
            CampaignAnyFieldEnum.CPM_BANNER_CAMPAIGN_FREQUENCY_CAP -> setOf(CampaignWithImpressionRate::class)
            CampaignAnyFieldEnum.CPM_BANNER_CAMPAIGN_VIDEO_TARGET -> setOf(CampaignWithEshowsSettings::class)
            CampaignAnyFieldEnum.CPM_BANNER_CAMPAIGN_SETTINGS -> campaignClassesForSettings
            CampaignAnyFieldEnum.CPM_BANNER_CAMPAIGN_BIDDING_STRATEGY -> setOf(CampaignWithStrategy::class)

            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_PLACEMENT_TYPES -> setOf(CampaignWithPlacementTypes::class)
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_COUNTER_IDS -> setOf(CampaignWithMetrikaCounters::class)
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_SETTINGS -> campaignClassesForSettings
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_BIDDING_STRATEGY -> setOf(CampaignWithStrategy::class)
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_PRIORITY_GOALS -> setOf(CampaignWithMeaningfulGoals::class)
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_ATTRIBUTION_MODEL -> setOf(CampaignWithAttributionModel::class)
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_STRATEGY_ID -> setOf(CampaignWithPackageStrategy::class)
            CampaignAnyFieldEnum.DYNAMIC_TEXT_CAMPAIGN_TRACKING_PARAMS -> setOf(CampaignWithBannerHrefParams::class)

            CampaignAnyFieldEnum.MOBILE_APP_CAMPAIGN_SETTINGS -> campaignClassesForSettings
            CampaignAnyFieldEnum.MOBILE_APP_CAMPAIGN_BIDDING_STRATEGY -> setOf(CampaignWithNetworkSettings::class)
            CampaignAnyFieldEnum.MOBILE_APP_CAMPAIGN_STRATEGY_ID -> setOf(CampaignWithPackageStrategy::class)

            CampaignAnyFieldEnum.SMART_CAMPAIGN_COUNTER_ID -> setOf(CampaignWithMetrikaCounters::class)
            CampaignAnyFieldEnum.SMART_CAMPAIGN_SETTINGS -> campaignClassesForSettings
            CampaignAnyFieldEnum.SMART_CAMPAIGN_BIDDING_STRATEGY -> setOf(CampaignWithNetworkSettings::class)
            CampaignAnyFieldEnum.SMART_CAMPAIGN_PRIORITY_GOALS -> setOf(CampaignWithMeaningfulGoals::class)
            CampaignAnyFieldEnum.SMART_CAMPAIGN_ATTRIBUTION_MODEL -> setOf(CampaignWithAttributionModel::class)
            CampaignAnyFieldEnum.SMART_CAMPAIGN_STRATEGY_ID -> setOf(CampaignWithPackageStrategy::class)
            CampaignAnyFieldEnum.SMART_CAMPAIGN_TRACKING_PARAMS -> setOf(CampaignWithBannerHrefParams::class)

            CampaignAnyFieldEnum.TEXT_CAMPAIGN_COUNTER_IDS -> setOf(CampaignWithMetrikaCounters::class)
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_RELEVANT_KEYWORDS -> setOf(CampaignWithBroadMatch::class)
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_SETTINGS -> campaignClassesForSettings
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_BIDDING_STRATEGY -> setOf(CampaignWithNetworkSettings::class)
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_PRIORITY_GOALS -> setOf(CampaignWithMeaningfulGoals::class)
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_ATTRIBUTION_MODEL -> setOf(CampaignWithAttributionModel::class)
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_STRATEGY_ID -> setOf(CampaignWithPackageStrategy::class)
            CampaignAnyFieldEnum.TEXT_CAMPAIGN_TRACKING_PARAMS -> setOf(CampaignWithBannerHrefParams::class)
        }

    private fun loadClients(
        campaigns: List<CommonCampaign>,
        loadAgencies: Boolean,
        loadChiefSubclient: Boolean,
    ): Map<ClientId, Client> {
        val allClientIds = buildSet {
            if (loadAgencies) {
                campaigns.mapNotNullTo(this) {
                    ClientId.fromNullableLong(it.agencyId)
                }
            }

            if (loadChiefSubclient) {
                add(auth.chiefSubclient.clientId)
            }
        }
        return clientService.massGetClientsByClientIds(allClientIds)
    }

    private fun loadClientNdsRatio(client: Client): BigDecimal {
        return clientNdsService
            .massGetEffectiveClientNds(listOf(client))
            .firstOrNull()
            ?.nds
            ?.asRatio()
            ?: run {
                logger.warn("No NDS for client ${client.clientId}")
                BigDecimal.ZERO
            }
    }

    private fun calculateSumsAvailableForTransfer(
        campaigns: List<CommonCampaign>,
        ndsRatio: BigDecimal,
        clientCurrency: Currency,
    ): Map<Long, BigDecimal> {
        // поле SumAvailableForTransfer заполняется только для кампаний без кошелька
        val campaignsWithSumAvailableForTransfer = campaigns
            .filter { (it.walletId ?: 0L) == 0L }
            .map { it as CampaignWithStrategy }
            .ifEmpty { return emptyMap() }

        return campaignSumAvailableForTransferCalculator
            .calculate(
                ndsRatio = ndsRatio,
                clientCurrency = clientCurrency,
                campaigns = campaignsWithSumAvailableForTransfer
            )
    }

    /**
     * В перле таймзоны кэшировались, в джаве же кэширование таймзон нигде не используется
     * Если нужны будут оптимизации - можно будет сделать кэширование
     *
     * Московская таймзона (с ID = 0) не загружается
     */
    private fun loadGeoTimezones(campaigns: List<CommonCampaign>): Map<Long, GeoTimezone> {
        val timezoneIds = campaigns
            .map { it.timeZoneId }
            .filterTo(HashSet()) { (it ?: 0L) != 0L }
            .ifEmpty { return emptyMap() }

        return geoTimezoneRepository
            .getGeoTimezonesByTimezoneIds(timezoneIds)
            .associateBy(GeoTimezone::getTimezoneId)
    }

    companion object {
        private val logger = LoggerFactory.getLogger(CampaignDataFetcher::class.java)

        private val campaignClassesForSettings = setOf(
            CampaignWithEnableCompanyInfo::class,
            CampaignWithExcludePausedCompetingAds::class,
            CampaignWithFavorite::class,
            CampaignWithOptionalAddMetrikaTagToUrl::class,
            CampaignWithOptionalAddOpenstatTagToUrl::class,
            CampaignWithOrderPhraseLengthPrecedence::class,
            CampaignWithSiteMonitoring::class,
        )

        private val campaignClassesForStatusClarification = setOf(
            CampaignWithCurrencyConverted::class,
            CampaignWithDayBudget::class,
            CampaignWithStopTime::class,
            CampaignWithStrategy::class,
        )
    }
}
