package ru.yandex.direct.api.v5.entity.campaigns.service

import com.yandex.direct.api.v5.campaigns.CampaignStateGetEnum
import com.yandex.direct.api.v5.campaigns.CampaignStatusPaymentEnum
import com.yandex.direct.api.v5.general.StatusEnum
import org.springframework.stereotype.Service
import ru.yandex.direct.api.v5.entity.campaigns.StatusClarificationTranslations
import ru.yandex.direct.api.v5.entity.campaigns.StatusClarificationTranslations.Companion.INSTANCE
import ru.yandex.direct.api.v5.entity.campaigns.container.GetCampaignsContainer
import ru.yandex.direct.api.v5.entity.campaigns.service.CampaignSumAvailableForTransferCalculator.Companion.TRANSFER_DELAY_AFTER_STOP
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCurrencyConverted
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStopTime
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.core.entity.campaign.model.TimeTargetStatus
import ru.yandex.direct.core.entity.campaign.service.TimeTargetStatusService
import ru.yandex.direct.core.entity.campaign.service.calcSumTotalIncludingOverdraft
import ru.yandex.direct.core.entity.campoperationqueue.model.CampQueueOperationName
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.i18n.Translatable
import ru.yandex.direct.i18n.types.ConcatTranslatable
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.utils.NumberUtils
import java.math.BigDecimal
import java.time.Clock
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.OffsetDateTime
import java.time.ZoneId
import java.time.ZonedDateTime
import java.time.temporal.ChronoUnit

@Service
class CampaignStatusCalculator(
    private val timeTargetStatusService: TimeTargetStatusService,
    private val clock: Clock,
) {
    fun calculateStatus(
        campaign: CommonCampaign,
    ): StatusEnum =
        when (campaign.statusModerate) {
            CampaignStatusModerate.YES -> StatusEnum.ACCEPTED
            CampaignStatusModerate.NO -> StatusEnum.REJECTED
            CampaignStatusModerate.SENT -> StatusEnum.MODERATION
            CampaignStatusModerate.READY -> StatusEnum.MODERATION
            CampaignStatusModerate.NEW -> StatusEnum.DRAFT
            else -> StatusEnum.UNKNOWN
        }

    fun calculateStatusPayment(
        campaign: CommonCampaign,
    ): CampaignStatusPaymentEnum =
        if (campaign.walletId != 0L || campaign.statusModerate == CampaignStatusModerate.YES) {
            CampaignStatusPaymentEnum.ALLOWED
        } else {
            CampaignStatusPaymentEnum.DISALLOWED
        }

    fun calculateState(
        campaign: CommonCampaign,
    ): CampaignStateGetEnum {
        campaign as CampaignWithCurrencyConverted

        val isConverted = campaign.currencyConverted && campaign.currency == CurrencyCode.YND_FIXED
        val isEnded = campaign.endDate != null && campaign.endDate < LocalDate.now()

        return when {
            campaign.statusArchived && isConverted ->
                CampaignStateGetEnum.CONVERTED

            campaign.statusArchived && !isConverted ->
                CampaignStateGetEnum.ARCHIVED

            !isConverted && isEnded ->
                CampaignStateGetEnum.ENDED

            !isConverted && !campaign.statusShow ->
                CampaignStateGetEnum.SUSPENDED

            !isConverted && !campaign.statusActive ->
                CampaignStateGetEnum.OFF

            !isEnded && campaign.statusShow && campaign.statusActive ->
                CampaignStateGetEnum.ON

            else ->
                CampaignStateGetEnum.UNKNOWN
        }
    }

    /**
     * Аналог функций
     * [`Campaign::CalcCampStatus_mass`](https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9368424#L5670-5687),
     * [`Campaign::get_camp_status_info_mass`](https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9368424#L5415-5612),
     * [`Campaign::_resolve_campaign_status`](https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9368424#L5691-5795)
     *
     * На вики есть оригинальная [спецификация](https://wiki.yandex-team.ru/users/andreyka/FeaturesDesign/API5CampaignsDesign/#variantystatusclarification)
     *
     * Также в [DIRECT-164613](https://st.yandex-team.ru/DIRECT-164613#624ea9ce21999173b60581f1) есть описание статусов,
     * полученное разбором перлового кода
     */
    fun calculateStatusClarification(container: GetCampaignsContainer): Translatable {
        val campaign = container.campaign
        val aggrWallet = container.aggregatedStatusWallet
        val state = calculateState(container.campaign)
        val strategyData: StrategyData? = (campaign as CampaignWithStrategy).strategy.strategyData

        val sumTotal = calcSumTotalIncludingOverdraft(
            campaignSum = campaign.sum,
            campaignSumSpent = campaign.sumSpent,
            walletSum = aggrWallet?.sum,
            autoOverdraftAddition = aggrWallet?.autoOverdraftAddition,
        )

        val moneyFinished = sumTotal < Currencies.EPSILON
            && campaign.orderId != 0L
            && campaign.sum + (container.wallet?.sum ?: BigDecimal.ZERO) > Currencies.EPSILON
        val stopped = !moneyFinished
            && sumTotal > Currencies.EPSILON
            && !campaign.statusShow
        val canBeShown = !moneyFinished
            && !stopped
            && sumTotal > Currencies.EPSILON
            && campaign.statusShow
            && (campaign.statusModerate == CampaignStatusModerate.YES || campaign.statusActive)
            && campaign.statusPostModerate != CampaignStatusPostmoderate.YES

        val now = clock.instant()
        val today = DateTimeUtils.instantToMoscowDateTime(now)
        val todayDate = today.toLocalDate()

        // Вместо campaign.statusModerate используется это значение
        // https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9368424#L5565-5569
        val statusModerate: CampaignStatusModerate? =
            if (campaign.statusModerate == CampaignStatusModerate.YES && campaign.statusPostModerate == CampaignStatusPostmoderate.YES) {
                CampaignStatusModerate.SENT
            } else {
                campaign.statusModerate
            }

        // вычисляется лениво, у статуса про временные корректировки низкий приоритет
        val timeTargetDescription by lazy {
            describeTimeTarget(campaign, container.geoTimezone, now)
        }

        with(StatusClarificationTranslations) {
            val result = mutableListOf<Translatable>()

            when {
                // Сконвертирована
                state == CampaignStateGetEnum.CONVERTED -> result += INSTANCE.converted()

                // Кампания перенесена в архив
                state == CampaignStateGetEnum.ARCHIVED -> {
                    result += INSTANCE.archived()

                    // Ожидает разархивирования
                    if (CampQueueOperationName.UNARC in container.queueOperations) {
                        result += INSTANCE.unarchiving()
                    }
                }

                // Нет объявлений
                !container.hasBanners && statusModerate != CampaignStatusModerate.NEW ->
                    result += INSTANCE.noAds()

                // Средства на счете закончились
                moneyFinished && statusModerate == CampaignStatusModerate.YES -> result += INSTANCE.fundsRunOut()

                // Кампания остановлена
                stopped -> result += INSTANCE.stopped()

                // Нет активных объявлений
                canBeShown && !container.hasActiveBanners -> result += INSTANCE.noActiveAds()

                // Ожидает старта стратегии {0}
                canBeShown && strategyData?.start?.isAfter(todayDate) == true -> {
                    result += awaitingStrategyLaunch(strategyData.start)
                }

                // Начало {0}
                canBeShown && campaign.startDate?.isAfter(todayDate) == true -> {
                    result += startDate(campaign.startDate)
                }

                // Кампания закончилась {0}
                canBeShown && campaign.endDate?.isBefore(todayDate) == true -> {
                    result += ended(campaign.endDate)
                }

                // Закончился период действия стратегии
                canBeShown && strategyData?.finish?.isBefore(todayDate) == true -> {
                    result += INSTANCE.strategyPeriodExpired()

                    // Режим автопродления
                    if (strategyData.autoProlongation != 0L) {
                        result += INSTANCE.strategyAutoProlongation()
                    }
                }

                // Показы приостановлены по дневному ограничению общего счёта в {0} (MSK)
                canBeShown
                    && (campaign.walletId ?: 0L) != 0L
                    && aggrWallet?.status?.budgetLimitationStopTime?.toLocalDate()?.equals(todayDate) == true ->
                    result += pausedByWalletDayBudget(aggrWallet.status.budgetLimitationStopTime)

                // Показы приостановлены по дневному ограничению в {0} (MSK)
                canBeShown
                    && NumberUtils.greaterThanZero((campaign as CampaignWithDayBudget).dayBudget)
                    && campaign.dayBudgetStopTime?.toLocalDate()?.equals(todayDate) == true ->
                    result += pausedByDayBudget(campaign.dayBudgetStopTime)

                // Идут показы; Показы начнутся в %s; Показы начнутся завтра в %s; Показы начнутся %s в %s
                canBeShown && timeTargetDescription != null ->
                    result += timeTargetDescription!!

                // Ждёт оплаты
                statusModerate == CampaignStatusModerate.YES && campaign.sumToPay > Currencies.EPSILON ->
                    result += INSTANCE.awaitingPayment()

                // Черновик; Допущено модератором; Отклонено модератором; Ожидает модерации
                statusModerate != null -> {
                    result += describeModerationStatus(statusModerate)

                    // Ждёт оплаты
                    if (statusModerate != CampaignStatusModerate.NO && campaign.sumToPay > Currencies.EPSILON) {
                        result += INSTANCE.awaitingPayment()
                    }
                }
            }

            // Ожидает архивирования
            if (state != CampaignStateGetEnum.ARCHIVED && CampQueueOperationName.ARC in container.queueOperations) {
                result += INSTANCE.archiving()
            }

            // Идет активизация
            val isActivating = isActivating(campaign, container, sumTotal, today)
            if (isActivating && statusModerate == CampaignStatusModerate.YES) {
                result += INSTANCE.activating()
            }

            val waitStart = campaign.startDate?.isAfter(todayDate) == true
                || strategyData?.start?.isAfter(todayDate) == true
            val finished = campaign.endDate?.isBefore(todayDate) == true
            if (!(canBeShown && waitStart) && !(canBeShown && finished) && campaign.endDate != null) {
                result += if (campaign.endDate >= todayDate) {
                    // Дата окончания кампании {0}
                    endDate(campaign.endDate)
                } else {
                    // Кампания закончилась {0}
                    ended(campaign.endDate)
                }
            }

            return result.singleOrNull() ?: ConcatTranslatable(". ", result)
        }
    }

    /**
     * Аналог словаря [`BannersCommon::MODERATE_STATUS`](https://a.yandex-team.ru/arc_vcs/direct/perl/protected/BannersCommon.pm?rev=r8259610#L120-127)
     */
    private fun describeModerationStatus(statusModerate: CampaignStatusModerate): Translatable =
        with(INSTANCE) {
            when (statusModerate) {
                CampaignStatusModerate.NEW -> draft()
                CampaignStatusModerate.YES -> acceptedOnModeration()
                CampaignStatusModerate.NO -> rejectedOnModeration()
                else -> awaitingModeration()
            }
        }

    /**
     * Аналог функции [`TimeTarget::timetarget_status`](https://a.yandex-team.ru/arc_vcs/direct/perl/protected/TimeTarget.pm?rev=r9287185#L512-529)
     */
    private fun describeTimeTarget(
        campaign: CommonCampaign,
        geoTimezone: GeoTimezone,
        now: Instant,
    ): Translatable? {
        val info = timeTargetStatusService.getTimeTargetStatus(campaign.timeTarget, geoTimezone, now)

        return with(StatusClarificationTranslations) {
            when (info.status) {
                null -> return null

                // Идут показы
                TimeTargetStatus.ACTIVE -> INSTANCE.campaignIsInProgress()

                // Показы начнутся в 12:00
                TimeTargetStatus.TODAY -> impressionsWillBeginToday(info.activationTime, geoTimezone)

                // Показы начнутся завтра в 12:00
                TimeTargetStatus.TOMORROW -> impressionsWillBeginTomorrow(info.activationTime, geoTimezone)

                // Показы начнутся в воскресенье в 12:00
                TimeTargetStatus.THIS_WEEK -> impressionsWillBeginThisWeek(info.activationTime, geoTimezone)

                // Показы начнутся XX.XX в 12:00
                TimeTargetStatus.LATER_THAN_WEEK -> {
                    // алгоритм поиска ближайшей даты показа не рассматривает даты позже, чем через 30 день
                    // поэтому если никакой даты не нашлось - считаем, что показы начнутся через 30 дней в 12:00 по Москве
                    val activationTime = info.activationTime
                        ?: fallbackActivationTime(now, clientTimeZone = geoTimezone.timezone)
                    impressionsWillBeginLaterThanThisWeek(activationTime, geoTimezone)
                }
            }
        }
    }

    /**
     * Время первого показа кампании по расписанию.
     * Используется в случаях, когда показов не будет в течение месяца
     * Из перлового кода не ясно, какое значение должно быть в этом случае (явно это не указано),
     * поэтому сейчас возвращается 12:00 по Москве
     */
    private fun fallbackActivationTime(now: Instant, clientTimeZone: ZoneId): OffsetDateTime =
        ZonedDateTime
            .ofInstant(now, clock.zone)
            .withZoneSameInstant(DateTimeUtils.MSK)
            .withHour(12)
            .withMinute(0)
            .plusDays(30)
            .withZoneSameInstant(clientTimeZone)
            .toOffsetDateTime()

    /**
     * Копия проверки на активизацию
     * [Campaign::get_camp_status_info_mass](https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9368424#L5575-5584)
     */
    private fun isActivating(
        campaign: CampaignWithStrategy,
        container: GetCampaignsContainer,
        sumTotal: BigDecimal,
        today: LocalDateTime?,
    ): Boolean {
        val expectedStatusActive = sumTotal > Currencies.EPSILON && campaign.statusShow
        val stopTime: LocalDateTime? = (campaign as CampaignWithStopTime).stopTime

        return sumTotal > Currencies.EPSILON
            && (
            campaign.statusBsSynced != CampaignStatusBsSynced.YES
                || campaign.statusActive != null && campaign.statusActive != expectedStatusActive
                || stopTime != null && stopTime.until(today, ChronoUnit.SECONDS) < TRANSFER_DELAY_AFTER_STOP.seconds
            )
            && campaign.statusPostModerate == CampaignStatusPostmoderate.ACCEPTED
            && !(!container.hasActiveBanners && campaign.statusActive == false && campaign.statusBsSynced == CampaignStatusBsSynced.YES)
    }
}
