package ru.yandex.direct.api.v5.entity.campaigns.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStopTime
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.statistics.service.OrderStatService
import ru.yandex.direct.currency.Currency
import java.math.BigDecimal
import java.math.RoundingMode
import java.time.Duration
import java.time.LocalDateTime

@Service
class CampaignSumAvailableForTransferCalculator(
    private val orderStatService: OrderStatService,
    private val campaignService: CampaignService,
) {

    /**
     * Аналог функции `Campaign::get_camp_sum_available` в перле
     * Не работает с валютными кампаниями
     *
     * https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9240412#L3669
     */
    fun calculate(
        ndsRatio: BigDecimal,
        clientCurrency: Currency,
        campaigns: List<CampaignWithStrategy>,
    ): Map<Long, BigDecimal> {
        val campaignMoneyIsBlocked = getMoneyOnCampaignIsBlocked(campaigns)

        val campaignsWithAllMoneyAvailableResults = campaigns
            .filter { isAllMoneyAvailableForTransfer(it) }
            .associateBy(BaseCampaign::getId) {
                (it.sum - it.sumSpent)
                    .coerceAtLeast(BigDecimal.ZERO)
                    .divide(BigDecimal.ONE + ndsRatio, RoundingMode.FLOOR)
            }

        val campaignsWithTransferCost = campaigns
            .filter { it.id !in campaignsWithAllMoneyAvailableResults.keys }
        val minRestSumsForCampaignId = getCampMinRest(
            clientCurrency,
            campaignsWithTransferCost,
        )
        val campaignsWithTransferCostResults = campaignsWithTransferCost
            .associateBy(BaseCampaign::getId) { campaign ->
                val minTransferSum = campaign.currency.currency.minTransferMoney
                val sumReserve = maxOf(
                    minTransferSum,
                    minRestSumsForCampaignId[campaign.id] ?: BigDecimal.ZERO
                )

                val total = (campaign.sum - campaign.sumSpent) / (BigDecimal.ONE + ndsRatio)
                val sumAvailable = total - sumReserve
                if (sumAvailable > minTransferSum) {
                    sumAvailable
                } else {
                    BigDecimal.ZERO
                }
            }

        return campaigns
            .associateBy(BaseCampaign::getId) { campaign ->
                if (campaignMoneyIsBlocked[campaign.id] == true) {
                    BigDecimal.ZERO
                } else {
                    campaignsWithAllMoneyAvailableResults[campaign.id]
                        ?: campaignsWithTransferCostResults.getValue(campaign.id)
                }
            }
    }

    private fun getMoneyOnCampaignIsBlocked(
        campaigns: List<CampaignWithStrategy>,
    ): Map<Long, Boolean> {
        val campaignsForBlockedMoneyChecks = campaigns.map {
            Campaign()
                .withId(it.id)
                .withType(it.type)
                .withWalletId(it.walletId)
                .withUserId(it.uid)
                .withManagerUserId(it.managerUid)
                .withAgencyUserId(it.agencyUid)
                .withStatusPostModerate(it.statusPostModerate)
                .withStatusModerate(it.statusModerate)
        }

        return campaignService.moneyOnCampaignsIsBlocked(
            campaignsForBlockedMoneyChecks,
            false,
            true, //  в перле функция mass_check_block_money_camps вызывалась без аналогичного параметра
        )
    }

    /**
     * Аналог функции Campaign::is_all_money_transfer_available.
     * Возвращает true, если кампания, вероятно, ещё показывается в БК
     *
     * https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9240412#L3543
     */
    private fun isAllMoneyAvailableForTransfer(campaign: CampaignWithStrategy): Boolean {
        val now = LocalDateTime.now()

        val campaignStopTime = (campaign as? CampaignWithStopTime?)?.stopTime
        val isStopping = !campaign.statusShow
            && campaignStopTime != null
            && now.minusMinutes(30) < campaignStopTime

        val isFinished = campaign.endDate != null && campaign.endDate < now.toLocalDate()

        return !campaign.statusShow && !isStopping
            || campaign.statusModerate == CampaignStatusModerate.NEW
            || isFinished
            || campaign.statusModerate == CampaignStatusModerate.NO && !campaign.statusActive
    }

    /**
     * Аналог функции `Campaign::get_camp_min_rest` из перла.
     * Возвращает минимально возможный остаток на кампании,
     * которого хватит на [TRANSFER_DELAY_AFTER_STOP] секунд
     *
     * https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9245901#L3601
     */
    private fun getCampMinRest(
        clientCurrency: Currency,
        campaigns: List<CampaignWithStrategy>,
    ): Map<Long, BigDecimal?> {
        if (campaigns.isEmpty()) {
            return emptyMap()
        }

        val campaignsById = campaigns
            .associateBy { it.id }

        val forecastForCampaignsWithCampWeeklyAutobudget = campaigns
            .filter(this::hasCampWeeklyAutobudget)
            .associateBy(BaseCampaign::getId) { campaign ->
                val transferSeconds = TRANSFER_DELAY_AFTER_STOP.seconds.toBigDecimal()
                val weekSeconds = Duration.ofDays(7).seconds.toBigDecimal()
                campaign.strategy.strategyData.sum * transferSeconds / weekSeconds
            }

        val campaignIdsWithoutAutobudget =
            campaigns.mapTo(HashSet()) { it.id } - forecastForCampaignsWithCampWeeklyAutobudget.keys

        val orderStatForecasts = orderStatService
            .getCampBsstatForecast(campaignIdsWithoutAutobudget, clientCurrency.code)
            .mapValues { (cid, forecast) ->
                val campaign = campaignsById.getValue(cid)
                if (campaign is CampaignWithDayBudget && campaign.dayBudget != null) {
                    minOf(
                        forecast.bigDecimalValue(),
                        campaign.dayBudget,
                    )
                } else {
                    forecast.bigDecimalValue()
                }
            }
            .withDefault { BigDecimal.ZERO }

        return campaigns
            .associateBy(BaseCampaign::getId) { campaign ->
                forecastForCampaignsWithCampWeeklyAutobudget[campaign.id]
                    ?: orderStatForecasts.getValue(campaign.id)
            }
    }

    /**
     * Аналог `Campaign::has_camp_weekly_autobudget`
     *
     * https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign.pm?rev=r9240412#L3578
     */
    private fun hasCampWeeklyAutobudget(
        campaign: CampaignWithStrategy,
    ): Boolean {
        return campaign.strategy.isAutoBudget
            && (campaign.strategy.strategyData.sum ?: 0L) != 0L
            && campaign.statusBsSynced == CampaignStatusBsSynced.YES
    }

    companion object {
        val TRANSFER_DELAY_AFTER_STOP: Duration = Duration.ofMinutes(30)
    }
}
