package ru.yandex.direct.api.v5.entity.campaigns.validation;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignStub;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Фабричные методы для инстанцирования специальных объектов –
 * способ передать информацию об ошибках с первого (externalRequest) на второй (internalRequest) этап валидации.
 */
@ParametersAreNonnullByDefault
public class CampaignsApiValidationSignals {

    private CampaignsApiValidationSignals() {
        // no instantiation
    }

    public static ValidationResult<BaseCampaign, DefectType> getExternalRequestDefects(
            BaseCampaign campaignWrapper) {
        CampaignWrapper campaignWrapper2 = (CampaignWrapper) campaignWrapper;
        // реальная кампания отправляется в ядро и потом результат будет мёржиться с ядром, важно чтобы модели совпадали
        // если realCampaign null - то это значит, что кампания не валидная, она не будет отправляться в ядро и
        // результат ни с чем мёржить не нужно
        return new ValidationResult<>(campaignWrapper2.getCampaign(),
                campaignWrapper2.getDefects(),
                campaignWrapper2.getWarnings());
    }

    @SuppressWarnings("unchecked")
    public static List<BaseCampaign> getCampaigns(List<BaseCampaign> wrappedCampaigns) {
        return ((List<CampaignsApiValidationSignals.CampaignWrapper>) (List) wrappedCampaigns)
                .stream()
                .map(CampaignsApiValidationSignals.CampaignWrapper::getCampaign)
                .collect(Collectors.toList());
    }

    /**
     * Костыль - instance кампании внутри которого ValidationResult
     */
    public static class CampaignWrapper extends CampaignStub {
        private List<DefectType> defects;
        private List<DefectType> warnings;
        private BaseCampaign campaign;
        private Boolean requireServicing;

        private List<DefectType> getDefects() {
            return defects;
        }

        private List<DefectType> getWarnings() {
            return warnings;
        }

        @Nullable
        private BaseCampaign getCampaign() {
            return campaign;
        }

        public Boolean getRequireServicing() {
            return requireServicing;
        }

        public CampaignWrapper withDefects(List<DefectType> defects) {
            this.defects = defects;
            return this;
        }

        public CampaignWrapper withWarnings(List<DefectType> warnings) {
            this.warnings = warnings;
            return this;
        }

        public CampaignWrapper withRealCampaign(@Nullable BaseCampaign realCampaign) {
            this.campaign = realCampaign;
            return this;
        }

        public CampaignWrapper withRequireServicing(Boolean requireServicing) {
            this.requireServicing = requireServicing;
            return this;
        }

    }
}
