package ru.yandex.direct.api.v5.entity.campaigns.validation

import com.yandex.direct.api.v5.campaigns.DailyBudget
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.PlacementType
import com.yandex.direct.api.v5.campaigns.TextCampaignSettingsEnum
import com.yandex.direct.api.v5.campaigns.TimeTargetingOnPublicHolidays
import com.yandex.direct.api.v5.general.ArrayOfString
import com.yandex.direct.api.v5.general.YesNoEnum
import ru.yandex.direct.api.v5.common.ConverterUtils
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.campaignNotFound
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.campaignTypeNotSupported
import ru.yandex.direct.api.v5.entity.campaigns.converter.toCampaignWarnPlaceInterval
import ru.yandex.direct.api.v5.entity.campaigns.converter.toDate
import ru.yandex.direct.api.v5.entity.campaigns.converter.toLocalTime
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.api.v5.validation.DefectTypes
import ru.yandex.direct.api.v5.validation.validateList
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.API5_EDIT_CAMPAIGNS
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.API5_EDIT_CAMPAIGNS_EXT
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.API5_VISIBLE
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource
import ru.yandex.direct.core.entity.campaign.model.CampaignWarnPlaceInterval
import ru.yandex.direct.utils.NumberUtils
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult
import java.time.format.DateTimeParseException

/**
 * Общая валидация, используемая в Add и Update.
 */
object CampaignsCommonRequestValidator {

    private val DEPRECATED_SETTINGS: Set<Set<Enum<*>>> =
        setOf(
            setOf(
                DynamicTextCampaignSettingsEnum.ENABLE_BEHAVIORAL_TARGETING
            ),
            setOf(
                TextCampaignSettingsEnum.ENABLE_AUTOFOCUS,
                TextCampaignSettingsEnum.ENABLE_RELATED_KEYWORDS,
                TextCampaignSettingsEnum.ENABLE_BEHAVIORAL_TARGETING
            ),
            setOf(
                MobileAppCampaignSettingsEnum.ENABLE_AUTOFOCUS,
                MobileAppCampaignSettingsEnum.ENABLE_BEHAVIORAL_TARGETING
            )
        )

    @JvmStatic
    fun <S : Enum<S>> settingIsSupported(setting: S): DefectType? =
        if (DEPRECATED_SETTINGS.any { it.contains(setting) })
            CampaignDefectTypes.settingIsDeprecated(setting.toString())
        else null

    fun startDateFormatValid(startDate: String?): DefectType? =
        if (dateFormatValid(startDate)) null else DefectTypes.badParamsInvalidFormat("StartDate")

    fun endDateFormatValid(endDate: String?): DefectType? =
        if (dateFormatValid(endDate)) null else DefectTypes.badParamsInvalidFormat("EndDate")

    fun dateFormatValid(date: String?): Boolean {
        return date.isNullOrEmpty() || runCatching { toDate(date) }.isSuccess
    }

    fun checkPositionIntervalValid(checkPositionInterval: Int?): DefectType? =
        checkPositionInterval?.let {
            try {
                toCampaignWarnPlaceInterval(it)
            } catch (e: IllegalArgumentException) {
                return DefectTypes.fieldMustBeInList("CheckPositionInterval",
                    CampaignWarnPlaceInterval.values().map { x -> CampaignWarnPlaceInterval.toSource(x)!!.literal })
            }
            return null
        }

    fun timeTargetingDayOfTheWeekOutOfRange(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            try {
                getDayOfWeek(oneDay) in 1..7
            } catch (e: NumberFormatException) {
                false
            }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingDayOfTheWeekOutOfRange() else null
    }

    fun timeTargetingDayOfTheWeekHasDuplicates(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.let { days ->
            days.map { oneDay -> getDayOfWeek(oneDay) }
                .groupingBy { it }
                .eachCount()
                .all { it.value == 1 }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingDayOfTheWeekHasDuplicates() else null
    }

    fun timeTargetingBidsIncomplete(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            oneDay.split(',').size == 25
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingBidsIncomplete() else null
    }

    fun timeTargetingBidsMustNotBeNegative(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            getBids(oneDay).all { it.toInt() >= 0 }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingBidsMustNotBeNegative() else null
    }

    fun timeTargetingBidsInCpmBannerCampaignMustBeOnOrOff(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            getBids(oneDay).all { it.toInt() in listOf(0, 100) }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingBidsInCpmBannerCampaignMustBeOnOrOff() else null
    }

    fun timeTargetingBidsMustDevideTen(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            getBids(oneDay).all { it.toInt() % 10 == 0 }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingBidsMustDevideTen() else null
    }

    fun timeTargetingBidsOutOfRange(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            getBids(oneDay).all { it.toInt() in 0..200 }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingBidsOutOfRange() else null
    }

    fun timeTargetingBidsAreNotIntegers(schedule: ArrayOfString?): DefectType? {
        val ok = schedule?.items?.all { oneDay ->
            getBids(oneDay).all { it.toIntOrNull() != null }
        }
        return if (ok == false) CampaignDefectTypes.timeTargetingBidsAreNotIntegers() else null
    }

    fun holidaysScheduleCantContainStartHourWhenSuspendOnHolidaysYes(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        val ok = holidaysSchedule?.let {
            it.suspendOnHolidays != YesNoEnum.YES
                || it.startHour == null
        }
        return if (ok == false) CampaignDefectTypes.holidaysScheduleCantContainStartHourWhenSuspendOnHolidaysYes() else null
    }

    fun holidaysScheduleCantContainEndHourWhenSuspendOnHolidaysYes(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        val ok = holidaysSchedule?.let {
            it.suspendOnHolidays != YesNoEnum.YES
                || it.endHour == null
        }
        return if (ok == false) CampaignDefectTypes.holidaysScheduleCantContainEndHourWhenSuspendOnHolidaysYes() else null
    }

    fun holidaysScheduleCantContainBidPercentWhenSuspendOnHolidaysYes(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        val ok = holidaysSchedule?.let {
            it.suspendOnHolidays != YesNoEnum.YES
                || it.bidPercent == null
        }
        return if (ok == false) CampaignDefectTypes.holidaysScheduleCantContainBidPercentWhenSuspendOnHolidaysYes() else null
    }

    fun holidayBidPercentMustBeOneHundredInCpmBannerCampaign(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        val ok = holidaysSchedule?.bidPercent?.let { it == 100 }
        return if (ok == false) CampaignDefectTypes.holidayBidPercentMustBeOneHundredInCpmBannerCampaign() else null
    }

    fun holidayBidPercentOutOfRange(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        val ok = holidaysSchedule?.bidPercent?.let { it in 10..200 }
        return if (ok == false) CampaignDefectTypes.holidayBidPercentOutOfRange() else null
    }

    fun holidayBidPercentMustDevideTen(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        val ok = holidaysSchedule?.bidPercent?.let { it % 10 == 0 }
        return if (ok == false) CampaignDefectTypes.holidayBidPercentMustDevideTen() else null
    }

    fun startHourAndEndHourMustBeSetWhenSuspendOnHolidaysNotYes(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        if (holidaysSchedule == null)
            return null
        val ok = holidaysSchedule.suspendOnHolidays == YesNoEnum.YES
            || (holidaysSchedule.startHour != null && holidaysSchedule.endHour != null)
        return if (!ok) CampaignDefectTypes.startHourAndEndHourMustBeSetWhenSuspendOnHolidaysNotYes() else null
    }

    fun startHourMustBeLessThanEndHour(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        if (holidaysSchedule == null)
            return null
        val ok = holidaysSchedule.startHour == null
            || holidaysSchedule.endHour == null
            || holidaysSchedule.startHour < holidaysSchedule.endHour
        return if (!ok) CampaignDefectTypes.startHourMustBeLessThanEndHour() else null
    }

    fun startHourOutOfRange(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        if (holidaysSchedule == null)
            return null
        val ok = holidaysSchedule.startHour?.let { it in 0..23 }
        return if (ok == false) CampaignDefectTypes.startHourOutOfRange() else null
    }

    fun endHourOutOfRange(holidaysSchedule: TimeTargetingOnPublicHolidays?): DefectType? {
        if (holidaysSchedule == null)
            return null
        val ok = holidaysSchedule.endHour?.let { it in 1..24 }
        return if (ok == false) CampaignDefectTypes.endHourOutOfRange() else null
    }

    @JvmStatic
    fun duplicatesInPlacementTypes(placementTypes: List<PlacementType>?): DefectType? {
        if (placementTypes == null)
            return null
        val arrayOfString = ArrayOfString().withItems(placementTypes.map { it.type.name })
        return duplicatesIn(arrayOfString, "PlacementTypes")
    }

    fun duplicatesIn(array: ArrayOfString?, arrayName: String): DefectType? {
        if (array == null)
            return null
        val duplicates = array.items.groupBy { it }.filterValues { it.size > 1 }.keys.toList()
        return if (duplicates.isEmpty()) null else DefectTypes.duplicatedElements(duplicates, arrayName)
    }

    fun dayBudgetAmountMustBePositiveAndNotLessThan(dailyBudget: DailyBudget?): DefectType? {
        val isPositive = dailyBudget?.amount?.let { it > 0 }
        val isZeroAfterConvert =
            dailyBudget?.amount?.let { NumberUtils.isZero(ConverterUtils.convertToDbPrice(it)) }
        return if (isPositive == false)
            DefectTypes.fieldShouldBePositive("DailyBudget Amount")
        else if (isZeroAfterConvert == true)
            DefectTypes.badParamsNotLessThan(300, "DailyBudget Amount")
        else
            null
    }

    fun startDateIsNotEmpty(startDate: String?): DefectType? =
        if (dateIsEmpty(startDate)) DefectTypes.emptyValue("StartDate") else null

    fun endDateIsNotEmpty(endDate: String?): DefectType? =
        if (dateIsEmpty(endDate)) DefectTypes.emptyValue("EndDate") else null

    fun dateIsEmpty(date: String?): Boolean = date?.isEmpty() ?: false

    private fun getDayOfWeek(oneDayTimeTargeting: String) = oneDayTimeTargeting.substringBefore(',').toInt()

    private fun getBids(oneDayTimeTargeting: String) = oneDayTimeTargeting.split(',').drop(1)

    fun validateCampaignsAreSupported(
        cids: List<Long>,
        campaignTypeSourceByCids: MutableMap<Long, CampaignTypeSource>,
        isExt: Boolean
    ): ValidationResult<List<Long>, DefectType> =
        validateList(cids) {
            checkEach(allowedCampaignType(campaignTypeSourceByCids, API5_VISIBLE, campaignNotFound()))
            val allowedCampaignTypes = if (isExt) API5_EDIT_CAMPAIGNS_EXT else API5_EDIT_CAMPAIGNS
            checkEach(
                allowedCampaignType(campaignTypeSourceByCids, allowedCampaignTypes, campaignTypeNotSupported()),
                When.isValid()
            )
            checkEach(allowedCampaignSource(campaignTypeSourceByCids), When.isValid())
        }

    private fun allowedCampaignType(
        campaignTypeSourceByCids: Map<Long, CampaignTypeSource>,
        allowedCampaignTypes: Set<CampaignType>,
        defectType: DefectType
    ) = Constraint.fromPredicate(
        { cid: Long ->
            campaignTypeSourceByCids[cid] == null
                || allowedCampaignTypes.contains(campaignTypeSourceByCids[cid]!!.campaignType)
        }, defectType
    )

    private fun allowedCampaignSource(campaignTypeSourceByCids: Map<Long, CampaignTypeSource>) =
        Constraint.fromPredicate(
            { cid: Long ->
                campaignTypeSourceByCids[cid] == null
                    || AvailableCampaignSources.isSupportedInAPI5(campaignTypeSourceByCids[cid]!!.campaignsSource)
            }, campaignTypeNotSupported()
        )

    fun timeFormatValid(time: String?, field: String): DefectType? =
    time?.let {
        try {
            it.toLocalTime()
        } catch (e: DateTimeParseException) {
            return DefectTypes.invalidFormat(field)
        }
        return null
    }

}
