package ru.yandex.direct.api.v5.entity.campaigns.validation

import com.yandex.direct.api.v5.campaigns.CampaignUpdateItem
import com.yandex.direct.api.v5.campaigns.PriorityGoalsUpdateSetting
import com.yandex.direct.api.v5.campaigns.RelevantKeywordsSetting
import com.yandex.direct.api.v5.campaigns.UpdateRequest
import com.yandex.direct.api.v5.general.OperationEnum
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.campaignTypeNotSupported
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsContainer
import ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignsUpdateRequestConverter.Companion.convertCampaignTypeToExternal
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsCommonRequestValidator.dateFormatValid
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsCommonRequestValidator.dateIsEmpty
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsCommonRequestValidator.timeFormatValid
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.api.v5.validation.DefectTypes
import ru.yandex.direct.api.v5.validation.validateObject
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources.isSupportedInAPI5
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSource
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult

@Component
class CampaignsUpdateRequestValidator {
    fun validateExternalRequest(request: UpdateRequest): ValidationResult<UpdateRequest, DefectType>? {
        val vb: ItemValidationBuilder<UpdateRequest, DefectType> = ItemValidationBuilder.of(request)

        vb.item(request.campaigns, "campaigns")
            .check(campaignsCountNotMoreThanMax())

        return vb.result.takeIf { it.hasAnyErrors() }
    }

    /**
     * Валидация внутреннего запроса происходит в
     * [ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignsUpdateRequestConverter]
     *
     * Этот метод не валидирует запрос, а просто возвращает уже найденные ошибки и предупреждения
     */
    fun validateInternalRequest(
        request: List<UpdateCampaignsContainer<*>>,
    ): ValidationResult<List<UpdateCampaignsContainer<*>>, DefectType> =
        ListValidationBuilder.of<UpdateCampaignsContainer<*>, DefectType>(request)
            .checkEachBy { _, item ->
                ValidationResult(
                    item,
                    item.errors,
                    item.warnings,
                )
            }
            .result

    companion object SimpleValidations {
        private val API5_EDIT_CAMPAIGNS_STRING = CampaignTypeKinds.API5_EDIT_CAMPAIGNS
            .joinToString()

        private const val UPDATE_IDS_LIMIT = 10

    /**
     * Валидирует данные за рамками структур "TextCampaign", "CpmBannerCampaign" etc. (в основном это дефекты
     * общие для всех типов кампаний).
     */
    fun validateCampaignUpdateItemCommonDefects(
        item: CampaignUpdateItem,
        timeZones: Map<String, GeoTimezone>,
        type: CampaignType?,
        requestCampaignIds: List<Long>
    ): ValidationResult<CampaignUpdateItem, DefectType> =
        validateObject(item) {
            check({ typeNotFound(type) }, When.isValid())
            check({ duplicatedCampaign(it, requestCampaignIds) }, When.isValid())
            check { timeZoneDoesntExist(it, timeZones) }
            check(::timeTargetingDayOfTheWeekOutOfRange)
            check(::timeTargetingDayOfTheWeekHasDuplicates, When.isValid())
            check(::timeTargetingBidsIncomplete, When.isValid())
            check(::timeTargetingBidsAreNotIntegers, When.isValid())
            check(::timeTargetingBidsMustNotBeNegative, When.isValid())
            check(
                { timeTargetingBidsInCpmBannerCampaignMustBeOnOrOff(it, type) },
                When.isValid()
            )
            check(::timeTargetingBidsMustDevideTen, When.isValid())
            check(::timeTargetingBidsOutOfRange, When.isValid())
            check(::holidaysScheduleCantContainStartHourWhenSuspendOnHolidaysYes)
            check(::holidaysScheduleCantContainEndHourWhenSuspendOnHolidaysYes)
            check(::holidaysScheduleCantContainBidPercentWhenSuspendOnHolidaysYes)
            check { validateHolidayBidPercent(it, type) }
            check(::startHourAndEndHourMustBeSetWhenSuspendOnHolidaysNotYes)
            check(::startHourMustBeLessThanEndHour)
            check(::startHourOutOfRange)
            check(::endHourOutOfRange)
            check(::startDateIsNotEmpty)
            check(::endDateIsNotEmpty)
            check(::startDateFormatValid)
            check(::endDateFormatValid)
            check(::timeFromFormatValid)
            check(::timeToFormatValid)
            check(::checkPositionIntervalValid)
            check(::dayBudgetAmountMustBePositiveAndNotLessThan)
            item(item.blockedIps?.value, "BlockedIps")
                .check { CampaignsCommonRequestValidator.duplicatesIn(it, "BlockedIps") }
            check { negativeKeywordsAreNotAllowedInCpmBannerCampaign(it, type) }
            item(item.negativeKeywords?.value, "NegativeKeywords")
                .check { CampaignsCommonRequestValidator.duplicatesIn(it, "NegativeKeywords") }
        }

        private fun campaignsCountNotMoreThanMax() = Constraint<List<CampaignUpdateItem>, DefectType> { campaigns ->
            if (campaigns.size > UPDATE_IDS_LIMIT) {
                CampaignDefectTypes.maxCampaignsPerUpdateRequest(UPDATE_IDS_LIMIT)
            } else {
                null
            }
        }

        fun allowedCampaignType(
            campaignWithType: CampaignWithCampaignType?,
            allowedCampaignTypes: Set<CampaignType>,
            defectType: DefectType
        ): Constraint<CampaignUpdateItem, DefectType> = Constraint.fromPredicateOfNullable(
            { campaignWithType != null && allowedCampaignTypes.contains(campaignWithType.type) },
            defectType
        )

        fun allowedCampaignSource(campaignWithSource: CampaignWithSource?): Constraint<CampaignUpdateItem, DefectType> =
            Constraint.fromPredicate(
                { campaignWithSource == null || isSupportedInAPI5(campaignWithSource.source) },
                campaignTypeNotSupported()
            )

        /**
         * Аналог перловой проверки https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns.pm?blame=true&rev=r8950186#L868-890
         */
        fun validCampaignType() = Constraint<CampaignUpdateItem, DefectType> { item ->
            val structures = listOfNotNull(
                item.textCampaign,
                item.cpmBannerCampaign,
                item.dynamicTextCampaign,
                item.mobileAppCampaign,
                item.smartCampaign,
            )

            if (structures.count() > 1) {
                DefectTypes.possibleOnlyOneField(API5_EDIT_CAMPAIGNS_STRING)
            } else {
                null
            }
        }

        /**
         * В перле такой валидации нет. Полноценная валидация, что кампании не существует - в ядре
         * В api важно, чтобы был определен тип кампании, чтобы дожить до ядра
         * Если тип определить не удалось - значит кампании точно нет, иначе бы определился по ней
         */
        fun typeNotFound(type: CampaignType?): DefectType? =
            if (type == null) CampaignDefectTypes.campaignNotFound() else null

        fun campaignTypeNotMatchesWithTypeInRequest(
            typeFromDb: CampaignType?,
            typeFromRequest: CampaignType?
        ): DefectType? =
            if (typeFromDb != null && typeFromRequest != null && typeFromDb != typeFromRequest)
                CampaignDefectTypes.campaignTypeNotMatchesWithTypeInRequest(
                    convertCampaignTypeToExternal(typeFromDb)!!.name,
                    convertCampaignTypeToExternal(typeFromRequest)!!.name
                )
            else null

        fun campaignIdShouldBePositive(item: CampaignUpdateItem): DefectType? =
            if (item.id <= 0) DefectTypes.fieldShouldBePositive("Id") else null

        // Медленный алгоритм ок, потому что размер campaignIds не больше 10
        fun duplicatedCampaign(item: CampaignUpdateItem, campaignIds: List<Long>): DefectType? =
            if (campaignIds.count { it == item.id } > 1)
                CampaignDefectTypes.duplicatedCampaign() else null

        fun startDateIsNotEmpty(item: CampaignUpdateItem): DefectType? =
            if (dateIsEmpty(item.startDate)) DefectTypes.emptyValue("StartDate") else null

        fun endDateIsNotEmpty(item: CampaignUpdateItem): DefectType? =
            if (dateIsEmpty(item.endDate?.value)) DefectTypes.emptyValue("EndDate") else null

        fun startDateFormatValid(item: CampaignUpdateItem): DefectType? =
            if (dateFormatValid(item.startDate)) null else DefectTypes.invalidFormat("StartDate")

        fun endDateFormatValid(item: CampaignUpdateItem): DefectType? =
            if (dateFormatValid(item.endDate?.value)) null else DefectTypes.invalidFormat("EndDate")

        fun timeFromFormatValid(item: CampaignUpdateItem): DefectType? =
            timeFormatValid(item.notification?.smsSettings?.timeFrom, "TimeFrom")

        fun timeToFormatValid(item: CampaignUpdateItem): DefectType? =
            timeFormatValid(item.notification?.smsSettings?.timeTo, "TimeTo")

        fun timeZoneDoesntExist(item: CampaignUpdateItem, timeZones: Map<String, GeoTimezone>): DefectType? =
            if (item.timeZone != null && !timeZones.containsKey(item.timeZone))
                CampaignDefectTypes.timeZoneDoesntExist() else null

        fun timeTargetingDayOfTheWeekOutOfRange(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingDayOfTheWeekOutOfRange(item.timeTargeting?.schedule)

        fun timeTargetingDayOfTheWeekHasDuplicates(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingDayOfTheWeekHasDuplicates(item.timeTargeting?.schedule)

        fun timeTargetingBidsIncomplete(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingBidsIncomplete(item.timeTargeting?.schedule)

        fun timeTargetingBidsMustNotBeNegative(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingBidsMustNotBeNegative(item.timeTargeting?.schedule)

        fun timeTargetingBidsInCpmBannerCampaignMustBeOnOrOff(
            item: CampaignUpdateItem,
            type: CampaignType?
        ): DefectType? =
            if (type != CampaignType.CPM_BANNER) null
            else CampaignsCommonRequestValidator.timeTargetingBidsInCpmBannerCampaignMustBeOnOrOff(
                item.timeTargeting?.schedule)

        fun timeTargetingBidsMustDevideTen(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingBidsMustDevideTen(item.timeTargeting?.schedule)

        fun timeTargetingBidsOutOfRange(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingBidsOutOfRange(item.timeTargeting?.schedule)

        fun timeTargetingBidsAreNotIntegers(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.timeTargetingBidsAreNotIntegers(item.timeTargeting?.schedule)

        fun holidaysScheduleCantContainStartHourWhenSuspendOnHolidaysYes(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.holidaysScheduleCantContainStartHourWhenSuspendOnHolidaysYes(
                item.timeTargeting?.holidaysSchedule?.value)

        fun holidaysScheduleCantContainEndHourWhenSuspendOnHolidaysYes(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.holidaysScheduleCantContainEndHourWhenSuspendOnHolidaysYes(
                item.timeTargeting?.holidaysSchedule?.value)

        fun holidaysScheduleCantContainBidPercentWhenSuspendOnHolidaysYes(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.holidaysScheduleCantContainBidPercentWhenSuspendOnHolidaysYes(
                item.timeTargeting?.holidaysSchedule?.value)

        fun validateHolidayBidPercent(
            item: CampaignUpdateItem,
            type: CampaignType?
        ): DefectType? {
            val cpmDefect = if (type != CampaignType.CPM_BANNER) null
                    else CampaignsCommonRequestValidator.holidayBidPercentMustBeOneHundredInCpmBannerCampaign(
                item.timeTargeting?.holidaysSchedule?.value)
            return cpmDefect
                ?: CampaignsCommonRequestValidator.holidayBidPercentOutOfRange(
                    item.timeTargeting?.holidaysSchedule?.value)
                ?: CampaignsCommonRequestValidator.holidayBidPercentMustDevideTen(
                    item.timeTargeting?.holidaysSchedule?.value)
        }

        fun startHourAndEndHourMustBeSetWhenSuspendOnHolidaysNotYes(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.startHourAndEndHourMustBeSetWhenSuspendOnHolidaysNotYes(
                item.timeTargeting?.holidaysSchedule?.value)

        fun startHourMustBeLessThanEndHour(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.startHourMustBeLessThanEndHour(item.timeTargeting?.holidaysSchedule?.value)

        fun startHourOutOfRange(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.startHourOutOfRange(item.timeTargeting?.holidaysSchedule?.value)

        fun endHourOutOfRange(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.endHourOutOfRange(item.timeTargeting?.holidaysSchedule?.value)

        fun priorityGoalsNotSupportedOperation(priorityGoals: PriorityGoalsUpdateSetting?): DefectType? {
            val ok = priorityGoals?.items?.all { it.operation == OperationEnum.SET }
            return if (ok == false) CampaignDefectTypes.priorityGoalsNotSupportedOperation() else null
        }

        fun dayBudgetAmountMustBePositiveAndNotLessThan(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.dayBudgetAmountMustBePositiveAndNotLessThan(item.dailyBudget?.value)

        fun negativeKeywordsAreNotAllowedInCpmBannerCampaign(
            item: CampaignUpdateItem,
            type: CampaignType?
        ): DefectType? =
            // negativeKeywords можно обновлять не указывая структуру item.cpmBannerCampaign, поэтому проверяем тип
            // кампании по значению из БД
            if (type != CampaignType.CPM_BANNER || item.negativeKeywords?.value?.items == null) null
            else CampaignDefectTypes.negativeKeywordsAreNotAllowedInCpmBannerCampaign()

        // Перловая проверка https://a.yandex-team.ru/arc_vcs/direct/perl/api/services/v5/API/Service/Campaigns.pm?rev=r9103541#L1768
        fun absentValueBudgetPercent(
            relevantKeywords: RelevantKeywordsSetting?,
            oldTextCampaign: TextCampaign?
        ): DefectType? =
            if (relevantKeywords != null && relevantKeywords.budgetPercent == null
                && !oldTextCampaign!!.broadMatch.broadMatchFlag)
                DefectTypes.absentValueInField("BudgetPercent") else null

        fun checkPositionIntervalValid(item: CampaignUpdateItem): DefectType? =
            CampaignsCommonRequestValidator.checkPositionIntervalValid(item.notification?.emailSettings?.checkPositionInterval)
    }
}
