package ru.yandex.direct.api.v5.entity.campaigns.validation;

import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignStrategyAdd;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.strategiesAreNotCompatible;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.CpmBannerCampaignValidatorKt.getNetworkStrategyTypesCompatibleWith;

@ParametersAreNonnullByDefault
public class CpmBannerCampaignAddRequestValidator {

    private static final Map<CpmBannerCampaignSearchStrategyTypeEnum,
            Function<CpmBannerCampaignSearchStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE = Map.of();

    private static final Set<CpmBannerCampaignSearchStrategyTypeEnum> STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
            Set.of();

    /**
     * MANUAL_CPM - не имеет настроек
     */
    private static final Map<CpmBannerCampaignNetworkStrategyTypeEnum,
            Function<CpmBannerCampaignNetworkStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE = Map.of(
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS,
            CpmBannerCampaignNetworkStrategyAdd::getWbDecreasedPriceForRepeatedImpressions,

            CpmBannerCampaignNetworkStrategyTypeEnum.CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS,
            CpmBannerCampaignNetworkStrategyAdd::getCpDecreasedPriceForRepeatedImpressions,

            CpmBannerCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_IMPRESSIONS,
            CpmBannerCampaignNetworkStrategyAdd::getWbMaximumImpressions,

            CpmBannerCampaignNetworkStrategyTypeEnum.CP_MAXIMUM_IMPRESSIONS,
            CpmBannerCampaignNetworkStrategyAdd::getCpMaximumImpressions,

            CpmBannerCampaignNetworkStrategyTypeEnum.WB_AVERAGE_CPV,
            CpmBannerCampaignNetworkStrategyAdd::getWbAverageCpv,

            CpmBannerCampaignNetworkStrategyTypeEnum.CP_AVERAGE_CPV,
            CpmBannerCampaignNetworkStrategyAdd::getCpAverageCpv

    );

   private static final Set<CpmBannerCampaignNetworkStrategyTypeEnum> STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
        Set.of();

    public static ValidationResult<CpmBannerCampaignAddItem, DefectType> validateCpmBannerCampaign(
            CpmBannerCampaignAddItem cpmBannerCampaignAdd) {
        var vb = ItemValidationBuilder.of(cpmBannerCampaignAdd, DefectType.class);

        vb.item(cpmBannerCampaignAdd.getBiddingStrategy(), "BiddingStrategy")
                .checkBy(CpmBannerCampaignAddRequestValidator::validateCampaignStrategy);

        return vb.getResult();
    }

    private static ValidationResult<CpmBannerCampaignStrategyAdd, DefectType> validateCampaignStrategy(
            CpmBannerCampaignStrategyAdd strategy) {
        var vb = ItemValidationBuilder.of(strategy, DefectType.class);

        vb.check(CpmBannerCampaignAddRequestValidator::validateSearchStrategyConsistent)
                .check(CpmBannerCampaignAddRequestValidator::validateNetworkStrategyConsistent)
                .check(CpmBannerCampaignAddRequestValidator::validateStrategyTypesAreCompatible, When.isValid());

        vb.item(strategy.getNetwork(), "Network")
                .checkBy(CpmBannerCampaignAddRequestValidator::validateNetwork, When.notNull());

        return vb.getResult();
    }

    private static ValidationResult<CpmBannerCampaignNetworkStrategyAdd, DefectType> validateNetwork(
            CpmBannerCampaignNetworkStrategyAdd network) {
        var vb = ItemValidationBuilder.of(network, DefectType.class);

        vb.item(network.getCpDecreasedPriceForRepeatedImpressions(), "CpDecreasedPriceForRepeatedImpressions")
                .check(CampaignsAddRequestValidatorKt::startDateIsNotEmpty, When.notNull())
                .check(CampaignsAddRequestValidatorKt::startDateFormatValid, When.notNull())
                .check(CampaignsAddRequestValidatorKt::endDateIsNotEmpty, When.notNull())
                .check(CampaignsAddRequestValidatorKt::endDateFormatValid, When.notNull());

        vb.item(network.getCpAverageCpv(), "CpAverageCpv")
                .check(CampaignsAddRequestValidatorKt::startDateIsNotEmpty, When.notNull())
                .check(CampaignsAddRequestValidatorKt::startDateFormatValid, When.notNull())
                .check(CampaignsAddRequestValidatorKt::endDateIsNotEmpty, When.notNull())
                .check(CampaignsAddRequestValidatorKt::endDateFormatValid, When.notNull());

        vb.item(network.getCpMaximumImpressions(), "CpMaximumImpressions")
                .check(CampaignsAddRequestValidatorKt::startDateIsNotEmpty, When.notNull())
                .check(CampaignsAddRequestValidatorKt::startDateFormatValid, When.notNull())
                .check(CampaignsAddRequestValidatorKt::endDateIsNotEmpty, When.notNull())
                .check(CampaignsAddRequestValidatorKt::endDateFormatValid, When.notNull());

        return vb.getResult();
    }

    @Nullable
    private static DefectType validateSearchStrategyConsistent(CpmBannerCampaignStrategyAdd strategy) {
        CpmBannerCampaignSearchStrategyAdd searchStrategy = strategy.getSearch();
        CpmBannerCampaignSearchStrategyTypeEnum searchStrategyType = searchStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(searchStrategy, searchStrategyType,
                STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE, STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE, true);
    }

    @Nullable
    private static DefectType validateNetworkStrategyConsistent(CpmBannerCampaignStrategyAdd strategy) {
        CpmBannerCampaignNetworkStrategyAdd networkStrategy = strategy.getNetwork();
        CpmBannerCampaignNetworkStrategyTypeEnum networkStrategyType = networkStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(networkStrategy, networkStrategyType,
                STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE, STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE, false);
    }

    @Nullable
    private static DefectType validateStrategyTypesAreCompatible(CpmBannerCampaignStrategyAdd strategy) {
        CpmBannerCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();
        CpmBannerCampaignNetworkStrategyTypeEnum networkStrategyType = strategy.getNetwork().getBiddingStrategyType();

        return getNetworkStrategyTypesCompatibleWith(searchStrategyType).contains(networkStrategyType) ?
                null:
                strategiesAreNotCompatible();
    }

}
