package ru.yandex.direct.api.v5.entity.campaigns.validation

import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignStrategy
import com.yandex.direct.api.v5.campaigns.CpmBannerCampaignUpdateItem
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.entity.campaigns.converter.toCpmBannerCampaignExternalStrategy
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult
import java.util.function.Function

object CpmBannerCampaignUpdateRequestValidator {

    private val STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE =
        mapOf<CpmBannerCampaignSearchStrategyTypeEnum, Function<CpmBannerCampaignSearchStrategy, Any>>()

    private val STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
        setOf<CpmBannerCampaignSearchStrategyTypeEnum>()

    /**
     * MANUAL_CPM - не имеет настроек
     */
    private val STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE =
        mapOf<CpmBannerCampaignNetworkStrategyTypeEnum?, Function<CpmBannerCampaignNetworkStrategy, Any>?>(
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS to
                Function { it.wbDecreasedPriceForRepeatedImpressions },
            CpmBannerCampaignNetworkStrategyTypeEnum.CP_DECREASED_PRICE_FOR_REPEATED_IMPRESSIONS to
                Function { it.cpDecreasedPriceForRepeatedImpressions },
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_IMPRESSIONS to
                Function { it.wbMaximumImpressions },
            CpmBannerCampaignNetworkStrategyTypeEnum.CP_MAXIMUM_IMPRESSIONS to
                Function { it.cpMaximumImpressions },
            CpmBannerCampaignNetworkStrategyTypeEnum.WB_AVERAGE_CPV to
                Function { it.wbAverageCpv },
            CpmBannerCampaignNetworkStrategyTypeEnum.CP_AVERAGE_CPV to
                Function { it.cpAverageCpv })

    private val STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
        setOf<CpmBannerCampaignNetworkStrategyTypeEnum>()

    fun validateCpmBannerCampaign(
        item: CpmBannerCampaignUpdateItem,
        campaignFromDb: CpmBannerCampaign?
    ): ValidationResult<CpmBannerCampaignUpdateItem, DefectType> {
        val vb = ItemValidationBuilder.of<CpmBannerCampaignUpdateItem, DefectType>(item)
        vb.apply {
            item(item.biddingStrategy, "BiddingStrategy")
                .checkBy({ validateCampaignStrategy(it, campaignFromDb!!) }, When.notNull())
        }
        return vb.result
    }

    private fun validateCampaignStrategy(
        strategy: CpmBannerCampaignStrategy,
        campaignFromDb: CpmBannerCampaign
    ): ValidationResult<CpmBannerCampaignStrategy, DefectType> {
        val oldStrategy = toCpmBannerCampaignExternalStrategy(campaignFromDb.strategy)
        val vb = ItemValidationBuilder.of<CpmBannerCampaignStrategy, DefectType>(strategy)
        vb.apply {
            item(strategy.search, "Search")
                .check(::validateSearchStrategyConsistent, When.notNull())
            item(strategy.network, "Network")
                .check(::validateNetworkStrategyConsistent, When.notNull())
                .checkBy(::validateNetwork, When.notNull())
            check({ validateStrategyTypesAreCompatible(it, oldStrategy) }, When.isValid())
        }
        return vb.result
    }

    private fun validateSearchStrategyConsistent(
        searchStrategy: CpmBannerCampaignSearchStrategy
    ) =
        CampaignsRequestValidator.validateStrategyConsistent(
            searchStrategy,
            searchStrategy.biddingStrategyType,
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE,
            STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE,
            true
        )

    private fun validateNetworkStrategyConsistent(
        networkStrategy: CpmBannerCampaignNetworkStrategy
    ) =
        CampaignsRequestValidator.validateStrategyConsistent(
            networkStrategy,
            networkStrategy.biddingStrategyType,
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE,
            STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE,
            false
        )

    private fun validateNetwork(
        network: CpmBannerCampaignNetworkStrategy
    ): ValidationResult<CpmBannerCampaignNetworkStrategy, DefectType> {
        val vb = ItemValidationBuilder.of(network, DefectType::class.java)
        vb.apply {
            item(network.cpAverageCpv, "CpAverageCpv").apply {
                item(network.cpAverageCpv?.startDate, "StartDate")
                    .check(CampaignsCommonRequestValidator::startDateIsNotEmpty, When.notNull())
                    .check(CampaignsCommonRequestValidator::startDateFormatValid, When.notNull())
                item(network.cpAverageCpv?.endDate, "EndDate")
                    .check(CampaignsCommonRequestValidator::endDateIsNotEmpty, When.notNull())
                    .check(CampaignsCommonRequestValidator::endDateFormatValid, When.notNull())
            }
            item(network.cpDecreasedPriceForRepeatedImpressions, "CpDecreasedPriceForRepeatedImpressions").apply {
                item(network.cpDecreasedPriceForRepeatedImpressions?.startDate, "StartDate")
                    .check(CampaignsCommonRequestValidator::startDateIsNotEmpty, When.notNull())
                    .check(CampaignsCommonRequestValidator::startDateFormatValid, When.notNull())
                item(network.cpDecreasedPriceForRepeatedImpressions?.endDate, "EndDate")
                    .check(CampaignsCommonRequestValidator::endDateIsNotEmpty, When.notNull())
                    .check(CampaignsCommonRequestValidator::endDateFormatValid, When.notNull())
            }
            item(network.cpMaximumImpressions, "CpMaximumImpressions").apply {
                item(network.cpMaximumImpressions?.startDate, "StartDate")
                    .check(CampaignsCommonRequestValidator::startDateIsNotEmpty, When.notNull())
                    .check(CampaignsCommonRequestValidator::startDateFormatValid, When.notNull())
                item(network.cpMaximumImpressions?.endDate, "EndDate")
                    .check(CampaignsCommonRequestValidator::endDateIsNotEmpty, When.notNull())
                    .check(CampaignsCommonRequestValidator::endDateFormatValid, When.notNull())
            }
        }
        return vb.result
    }

    private fun validateStrategyTypesAreCompatible(
        strategy: CpmBannerCampaignStrategy,
        oldStrategy: CpmBannerCampaignStrategy
    ): DefectType? {
        val searchStrategyType = strategy.search?.biddingStrategyType ?: oldStrategy.search.biddingStrategyType!!
        val networkStrategyType = strategy.network?.biddingStrategyType ?: oldStrategy.network.biddingStrategyType!!
        val compatibleNetworkTypes = getNetworkStrategyTypesCompatibleWith(searchStrategyType)
        return if (!compatibleNetworkTypes.contains(networkStrategyType)) {
            CampaignDefectTypes.strategiesAreNotCompatible()
        } else null
    }
}
