package ru.yandex.direct.api.v5.entity.campaigns.validation;

import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSetting;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignStrategyAdd;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignStrategyAddBase;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.impressionsBelowSearchStrategyIsNotSupported;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.strategiesAreNotCompatible;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.weeklyClickPackageNotSupported;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.DynamicTextCampaignValidatorKt.getNetworkStrategyTypesCompatibleWith;

@ParametersAreNonnullByDefault
public class DynamicTextCampaignAddRequestValidator {

    private static final Map<DynamicTextCampaignSearchStrategyTypeEnum,
            Function<DynamicTextCampaignSearchStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE = Map.of(
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPA, DynamicTextCampaignStrategyAddBase::getAverageCpa,
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CPC, DynamicTextCampaignStrategyAddBase::getAverageCpc,
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_CRR, DynamicTextCampaignStrategyAddBase::getAverageCrr,
            DynamicTextCampaignSearchStrategyTypeEnum.AVERAGE_ROI, DynamicTextCampaignStrategyAddBase::getAverageRoi,
            DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION,
            DynamicTextCampaignStrategyAddBase::getPayForConversion,
            DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR,
            DynamicTextCampaignStrategyAddBase::getPayForConversionCrr,
            DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS,
            DynamicTextCampaignStrategyAddBase::getWbMaximumClicks,
            DynamicTextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
            DynamicTextCampaignStrategyAddBase::getWbMaximumConversionRate,
            DynamicTextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
            DynamicTextCampaignStrategyAddBase::getWeeklyClickPackage
    );

    private static final Set<DynamicTextCampaignSearchStrategyTypeEnum> STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
            Set.of();

    // ДО не может быть в сетях, тому кроме SERVING_OFF ничего быть не может
    private static final Map<DynamicTextCampaignNetworkStrategyTypeEnum,
            Function<DynamicTextCampaignNetworkStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE = Map.of(
    );

    private static final Set<DynamicTextCampaignNetworkStrategyTypeEnum> STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
            Set.of();

    public static ValidationResult<DynamicTextCampaignAddItem, DefectType> validateDynamicTextCampaign(
            DynamicTextCampaignAddItem dynamicTextCampaignAdd) {
        ItemValidationBuilder<DynamicTextCampaignAddItem, DefectType> vb = ItemValidationBuilder.of(
                dynamicTextCampaignAdd);

        vb.item(dynamicTextCampaignAdd.getBiddingStrategy(), "BiddingStrategy")
                .checkBy(DynamicTextCampaignAddRequestValidator::validateCampaignStrategy);
        vb.item(dynamicTextCampaignAdd.getPlacementTypes(), "PlacementTypes")
                .check(CampaignsCommonRequestValidator::duplicatesInPlacementTypes);
        vb.list(dynamicTextCampaignAdd.getSettings(), "Settings")
                .weakCheckEach((DynamicTextCampaignSetting setting) ->
                        CampaignsCommonRequestValidator.settingIsSupported(setting.getOption()));

        return vb.getResult();
    }

    public static ValidationResult<DynamicTextCampaignStrategyAdd, DefectType> validateCampaignStrategy(
            DynamicTextCampaignStrategyAdd strategy) {
        ItemValidationBuilder<DynamicTextCampaignStrategyAdd, DefectType> vb = ItemValidationBuilder.of(strategy);

        vb.check(DynamicTextCampaignAddRequestValidator::validateSearchStrategyConsistent)
                .check(DynamicTextCampaignAddRequestValidator::validateNetworkStrategyConsistent)
                .check(DynamicTextCampaignAddRequestValidator::validateSearchStrategyTypeIsSupported, When.isValid())
                .check(DynamicTextCampaignAddRequestValidator::validateNetworkStrategyTypeIsSupported, When.isValid())
                .check(DynamicTextCampaignAddRequestValidator::validateStrategyTypesAreCompatible, When.isValid());

        return vb.getResult();
    }

    @Nullable
    private static DefectType validateSearchStrategyConsistent(DynamicTextCampaignStrategyAdd strategy) {
        DynamicTextCampaignSearchStrategyAdd searchStrategy = strategy.getSearch();
        DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType = searchStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(searchStrategy, searchStrategyType,
                STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE, STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE, true);
    }

    @Nullable
    private static DefectType validateNetworkStrategyConsistent(DynamicTextCampaignStrategyAdd strategy) {
        DynamicTextCampaignNetworkStrategyAdd networkStrategy = strategy.getNetwork();
        DynamicTextCampaignNetworkStrategyTypeEnum networkStrategyType = networkStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(networkStrategy, networkStrategyType,
                STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE, STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE, false);
    }

    @Nullable
    private static DefectType validateSearchStrategyTypeIsSupported(DynamicTextCampaignStrategyAdd strategy) {
        DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();

        if (searchStrategyType == DynamicTextCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH) {
            return impressionsBelowSearchStrategyIsNotSupported();
        }

        if (searchStrategyType == DynamicTextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE) {
            return weeklyClickPackageNotSupported();
        }
        return null;
    }

    @Nullable
    private static DefectType validateNetworkStrategyTypeIsSupported(DynamicTextCampaignStrategyAdd strategy) {
        var strategyType = strategy.getNetwork().getBiddingStrategyType();

        if (strategyType == DynamicTextCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE) {
            return weeklyClickPackageNotSupported();
        }
        return null;
    }

    @Nullable
    private static DefectType validateStrategyTypesAreCompatible(DynamicTextCampaignStrategyAdd strategy) {
        DynamicTextCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();
        DynamicTextCampaignNetworkStrategyTypeEnum networkStrategyType = strategy.getNetwork().getBiddingStrategyType();

        return getNetworkStrategyTypesCompatibleWith(searchStrategyType).contains(networkStrategyType) ?
                null :
                strategiesAreNotCompatible();
    }

}
