package ru.yandex.direct.api.v5.entity.campaigns.validation;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.MobileAppCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSetting;
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignStrategyAdd;
import com.yandex.direct.api.v5.campaigns.StrategyNetworkDefaultAdd;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.impressionsBelowSearchStrategyIsNotSupported;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.strategiesAreNotCompatible;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.weeklyClickPackageNotSupported;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.MobileAppCampaignValidatorKt.getNetworkStrategyTypesCompatibleWith;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.MobileAppCampaignValidatorKt.settingIsConsistentWithStrategy;

@ParametersAreNonnullByDefault
public class MobileAppCampaignAddRequestValidator {

    private static final Map<MobileAppCampaignSearchStrategyTypeEnum,
            Function<MobileAppCampaignSearchStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE = Map.of(

            MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS,
            MobileAppCampaignSearchStrategyAdd::getWbMaximumClicks,

            MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS,
            MobileAppCampaignSearchStrategyAdd::getWbMaximumAppInstalls,

            MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPC,
            MobileAppCampaignSearchStrategyAdd::getAverageCpc,

            MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPI,
            MobileAppCampaignSearchStrategyAdd::getAverageCpi,

            MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
            MobileAppCampaignSearchStrategyAdd::getWeeklyClickPackage,

            MobileAppCampaignSearchStrategyTypeEnum.PAY_FOR_INSTALL,
            MobileAppCampaignSearchStrategyAdd::getPayForInstall

    );

    private static final Set<MobileAppCampaignSearchStrategyTypeEnum> STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
            Set.of();

    private static final Map<MobileAppCampaignNetworkStrategyTypeEnum,
            Function<MobileAppCampaignNetworkStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE = Map.of(
            MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT,
            MobileAppCampaignNetworkStrategyAdd::getNetworkDefault,

            MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS,
            MobileAppCampaignNetworkStrategyAdd::getWbMaximumClicks,

            MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS,
            MobileAppCampaignNetworkStrategyAdd::getWbMaximumAppInstalls,

            MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPC,
            MobileAppCampaignNetworkStrategyAdd::getAverageCpc,

            MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPI,
            MobileAppCampaignNetworkStrategyAdd::getAverageCpi,

            MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
            MobileAppCampaignNetworkStrategyAdd::getWeeklyClickPackage,

            MobileAppCampaignNetworkStrategyTypeEnum.PAY_FOR_INSTALL,
            MobileAppCampaignNetworkStrategyAdd::getPayForInstall
    );

    private static final Set<MobileAppCampaignNetworkStrategyTypeEnum> STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
            Set.of(MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT);

    public static ValidationResult<MobileAppCampaignAddItem, DefectType> validateMobileAppCampaign(
            MobileAppCampaignAddItem mobileAppCampaignAdd) {
        ItemValidationBuilder<MobileAppCampaignAddItem, DefectType> vb = ItemValidationBuilder.of(mobileAppCampaignAdd);

        vb.item(mobileAppCampaignAdd.getBiddingStrategy(), "BiddingStrategy")
                .checkBy(MobileAppCampaignAddRequestValidator::validateCampaignStrategy);
        ListValidationBuilder<MobileAppCampaignSetting, DefectType> settings =
                vb.list(mobileAppCampaignAdd.getSettings(), "Settings");
        settings
                .weakCheckEach((MobileAppCampaignSetting setting) ->
                        CampaignsCommonRequestValidator.settingIsSupported(setting.getOption()))
                .weakCheckEach((MobileAppCampaignSetting setting) ->
                        settingIsConsistentWithStrategy(setting, mobileAppCampaignAdd.getBiddingStrategy()));

        return vb.getResult();
    }

    private static ValidationResult<MobileAppCampaignStrategyAdd, DefectType> validateCampaignStrategy(
            MobileAppCampaignStrategyAdd strategy) {
        ItemValidationBuilder<MobileAppCampaignStrategyAdd, DefectType> vb = ItemValidationBuilder.of(strategy);

        vb.check(MobileAppCampaignAddRequestValidator::validateSearchStrategyConsistent)
                .check(MobileAppCampaignAddRequestValidator::validateNetworkStrategyConsistent)
                .check(MobileAppCampaignAddRequestValidator::validateSearchStrategyTypeIsSupported, When.isValid())
                .check(MobileAppCampaignAddRequestValidator::validateNetworkStrategyTypeIsSupported, When.isValid())
                .check(MobileAppCampaignAddRequestValidator::validateStrategyTypesAreCompatible, When.isValid())
                .check(MobileAppCampaignAddRequestValidator::validateContextStrategy, When.isValid())
                .weakCheck(MobileAppCampaignValidatorKt::limitPercentIsConsistentWithStrategy,
                        When.isValid());

        return vb.getResult();
    }

    @Nullable
    private static DefectType validateSearchStrategyConsistent(MobileAppCampaignStrategyAdd strategy) {
        return CampaignsRequestValidator.validateStrategyConsistent(
                strategy.getSearch(),
                strategy.getSearch().getBiddingStrategyType(),
                STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE,
                STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE,
                true);
    }

    @Nullable
    private static DefectType validateNetworkStrategyConsistent(MobileAppCampaignStrategyAdd strategy) {
        return CampaignsRequestValidator.validateStrategyConsistent(
                strategy.getNetwork(),
                strategy.getNetwork().getBiddingStrategyType(),
                STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE,
                STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE,
                false);
    }

    @Nullable
    private static DefectType validateSearchStrategyTypeIsSupported(MobileAppCampaignStrategyAdd strategy) {
        MobileAppCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();

        if (searchStrategyType == MobileAppCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH) {
            return impressionsBelowSearchStrategyIsNotSupported();
        }

        if (searchStrategyType == MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE) {
            return weeklyClickPackageNotSupported();
        }

        return null;
    }

    @Nullable
    private static DefectType validateNetworkStrategyTypeIsSupported(MobileAppCampaignStrategyAdd strategy) {
        var strategyType = strategy.getNetwork().getBiddingStrategyType();

        if (strategyType == MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE) {
            return weeklyClickPackageNotSupported();
        }
        return null;
    }

    @Nullable
    private static DefectType validateStrategyTypesAreCompatible(MobileAppCampaignStrategyAdd strategy) {
        MobileAppCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();
        MobileAppCampaignNetworkStrategyTypeEnum networkStrategyType = strategy.getNetwork().getBiddingStrategyType();

        if (!getNetworkStrategyTypesCompatibleWith(searchStrategyType).contains(networkStrategyType)) {
            return strategiesAreNotCompatible();
        }
        return null;
    }

    /**
     * в perl _validate_context_strategy
     */
    @Nullable
    private static DefectType validateContextStrategy(MobileAppCampaignStrategyAdd strategy) {
        Integer limitPercent = null;
        var strategyType = strategy.getNetwork().getBiddingStrategyType();
        if (strategyType == MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT) {
            limitPercent = Optional.of(strategy)
                    .map(MobileAppCampaignStrategyAdd::getNetwork)
                    .map(MobileAppCampaignNetworkStrategyAdd::getNetworkDefault)
                    .map(StrategyNetworkDefaultAdd::getLimitPercent)
                    .orElse(null);
        }
        return CampaignsRequestValidator.validateNetworkDefaultLimitPercent(limitPercent);
    }

}
